-------------------------------------------------------
--! @file 
--! @brief Highest-priority pending interrupt selector for the intMAN hardware
--! @author Josef Strnadel, Brno University of Technology, Faculty of Information Technology
--! @email strnadel@fit.vutbr.cz
--! @date 2013-04-17
-------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use work.intMAN_package.all;

--
--
--
--! Entity of the highest-priority pending interrupt select unit for the intMAN hardware (interface)          
--
entity ipri_highest is
	port (
			IPRI_ARR : 	in t_intpri_arr;							            --! interrupt priorities
			IPRI_HIGH: 	out STD_LOGIC_VECTOR(t_pri_width_range);	--! the highest interrupt priority
			IPRI_ADEC: 	out STD_LOGIC_VECTOR(t_int_range)         --! address dmx output with the highest-priority interrupt selected 
	);
end entity ipri_highest;

--
--
--
--! Architecture of the highest-priority pending interrupt select unit for the intMAN hardware (inner structure)     
--
architecture arch of ipri_highest is
--! highest priority (HPRI) value signal
signal SIG_IPRIHIGH: STD_LOGIC_VECTOR(t_pri_width_range);   
--! HPRI interrupt index signal
signal SIG_IPRIPOS: STD_LOGIC_VECTOR(t_int_width_range);    
--! HPRI interrupt index validity signal
signal SIG_IPRIPOSVALID: STD_LOGIC;                         
begin
  --! Process of searching the highest priority of pending interrupts   
	process(IPRI_ARR) is
	variable highest_pri 	  : STD_LOGIC_VECTOR(t_pri_width_range);     --! highest priority (HPRI) value
	variable highest_pos 	  : STD_LOGIC_VECTOR(t_int_width_range);     --! HPRI interrupt index
	begin
		highest_pri := (OTHERS => '1');								                   --! init to the lowest ('1...1') priority value
		highest_pos := (OTHERS => '0');								                   --! reset index

		for i in t_int_range loop
			if(IPRI_ARR(i) < highest_pri) then						                 --! new HPRI found
				highest_pri := IPRI_ARR(i);							                     --! store the HPRI value
				highest_pos := STD_LOGIC_VECTOR(TO_UNSIGNED(i, INT_WIDTH));  --! store the HPRI interrupt index  
			end if;
		end loop;
		
		SIG_IPRIHIGH <= highest_pri;
		SIG_IPRIPOS <= highest_pos;
	end process;

   --! Process of SIG_IPRIPOSVALID adjustment
   process(IPRI_ARR) is
   variable i: integer;
   variable j: STD_LOGIC;
   begin
     j := '0';
     for i in t_pri_width_range loop
         j := (j or (not SIG_IPRIHIGH(i)));                          -- index is valid if there is at least one '0' in the HPRI field 
     end loop;  
   SIG_IPRIPOSVALID <= j;
   end process;

	IPRI_HIGH <= SIG_IPRIHIGH;
	
  --! Process of constructing the HPRI interrupt select signal   
	process(IPRI_ARR) is
	variable i: integer;
	begin
		i := TO_INTEGER(UNSIGNED(SIG_IPRIPOS));
		
		FOR idx IN t_int_range LOOP
			if(idx = i) then
				IPRI_ADEC(idx) <= ('1' and SIG_IPRIPOSVALID);                --! make select-line active only if the HPRI index is valid
			else
				IPRI_ADEC(idx) <= '0';
			end if;
		end LOOP;
	end process;
	
end architecture arch;
