-------------------------------------------------------
--! @file 
--! @brief Package of supplements for the intMAN hardware
--! @author Josef Strnadel, Brno University of Technology, Faculty of Information Technology
--! @email strnadel@fit.vutbr.cz
--! @date 2013-04-17
-------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.all;

--
--
--
--! custom definitions utilized during the intMAN implementation
--
package intMAN_package is

  --
  --
  --
  -- clock setup block
  --	

  --! base clock rate [Hz]
	constant CLK_BASE_HZ         : integer := 10 ** 6;
  --! monitor clock (MONCLK) rate [Hz]					
	constant MONCLK_HZ           : integer := 1 * CLK_BASE_HZ; 			
  --! operating system time (OSTIME) rate [Hz]
	constant OSTIME_HZ           : integer := 100; 						
  --! number of MONCLK cycles (ticks) per one OSTIME cycle (tick)
	constant NMONTICKS_IN_OSTICK : integer := MONCLK_HZ / OSTIME_HZ;

  --
  --
  --
  -- priority setup block
  --	

  --! number of bits per joint interrupt/task priority
	constant PRI_WIDTH : integer := 8;  								
  --! bit-index range for the priority width
	subtype t_pri_width_range is integer range PRI_WIDTH-1 downto 0;
  --! priority value range
	subtype t_pri_range is integer range (2 ** PRI_WIDTH) - 1 downto 0;
  --! value at which the hard-level priority ends (0 means the highest priority)
	constant PRI_HARD_LAST : integer := (2 ** PRI_WIDTH)/2; 							
  --! the lowest priority (reserved for the intMAN purposes and set to the all-1 value) 
	constant PRI_LAST      : integer := (2 ** PRI_WIDTH) - 1; 							

	----------------------------
	-- interrupt setup block
	----------------------------
  --! the worst-case interrupt service time [s]
	constant INT_C_MAX           : real    := 150.0 / REAL(CLK_BASE_HZ); 
  --! the worst-case interrupt service time [MONCLK cycles]
	constant INT_C_MAX_NMONTICKS : integer := INTEGER(INT_C_MAX) * MONCLK_HZ;
  --! the worst-case interrupt service time [OSTIME cycles]
	constant INT_C_MAX_NOSTICKS  : integer := INT_C_MAX_NMONTICKS / NMONTICKS_IN_OSTICK;

  --! number of interrupt sources
	constant INT_NMAX : integer := 16;  								
  --! number of bits needed to address the interrupt sources (=log2(INT_NMAX))
	constant INT_WIDTH : integer := 4;  								
  --! interrupt address bit-index range 
	subtype t_int_width_range is integer range INT_WIDTH-1 downto 0;
  --! interrupt range
	subtype t_int_range is integer range INT_NMAX-1 downto 0;

  --! maximum number of pending interrupts per one interrupt source
	subtype t_intpend_cnt is integer range 0 to 15;
  --! array of pending interrupt couters
	type t_intpendcnt_arr is array (t_int_range) of t_intpend_cnt;
  --! vector of interrupt ready flags 
	subtype t_intrdy_vec is STD_LOGIC_VECTOR(t_int_range);

  --! level sensitivity configuration datatype
	subtype t_intlevel_cfg is STD_LOGIC_VECTOR(1 downto 0);
  --! edge sensitivity configuration datatype
	subtype t_intedge_cfg is STD_LOGIC_VECTOR(1 downto 0);

  --! interrupt sensitivity configuration data structure
	type t_intcfg is               										
	record
		level : t_intlevel_cfg; 
		edge  : t_intedge_cfg;
	end record;

  --! array of configurations for interrupt sources
	type t_intcfg_arr is array (t_int_range) of t_intcfg;

  -- interrupt sensitivity legend:
	-- ----------------------------
	-- level: log.0 (00), log. 1 (01), no (10, 11)
	-- edge: rising (01), falling (10), both (11), no (00)

  --! interrupt configuration 
	constant C_INTCFG : t_intcfg_arr := 
	(
		0 => (level => "00", edge => "11"),
		1 => (level => "00", edge => "11"),
		2 => (level => "00", edge => "11"),
		3 => (level => "00", edge => "11"),
		4 => (level => "00", edge => "00"),
		5 => (level => "00", edge => "00"),
		6 => (level => "00", edge => "11"),
		7 => (level => "00", edge => "10"),
		8 => (level => "00", edge => "01"),
		9 => (level => "00", edge => "01"),
		10 => (level => "00", edge => "01"),
		11 => (level => "00", edge => "01"),
		12 => (level => "00", edge => "01"),
		13 => (level => "00", edge => "01"),
		14 => (level => "00", edge => "01"),
		15 => (level => "00", edge => "01")
		);

  --! array of interrupt priorities 
	type t_intpri_arr is array (t_int_range) of STD_LOGIC_VECTOR(t_pri_width_range);

end package;

--! empty
package body intMAN_package is
end intMAN_package;
