-------------------------------------------------------
--! @file 
--! @brief Top design of the intMAN hardware
--! @author Josef Strnadel, Brno University of Technology, Faculty of Information Technology
--! @email strnadel@fit.vutbr.cz
--! @date 2013-04-17
-------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use work.intMAN_package.all;

--
--
--
--! intMAN unit entity (interface)     
--
entity intMAN_design is
	Port(
     MON_INT   		: in  STD_LOGIC;                            --! line to monitor whether the RT system is operating at the ISR (interrupt service routine) level
		 MON_TICK  		: in  STD_LOGIC;                            --! line to monitor whether the RT system is servicing the OS (operating system) time tick 
		 MON_CTX   		: in  STD_LOGIC;                            --! line to monitor whether the RT system is performing a task-level context-switch
		 MON_PRI   		: in  STD_LOGIC_VECTOR(t_pri_width_range);  --! line to monitor the priority of a task running in the RT system
		 MON_SLACK 		: in  STD_LOGIC;                            --! line to monitor whether there is slack time in the RT schedule
		 INT_ACK   		: in  STD_LOGIC;                            --! line to acknowledge an INT service
		 INT_OUT   		: out STD_LOGIC_VECTOR(t_int_range);        --! line to forward pending interrupt requests to the RT system
		 INT_IN    		: in  STD_LOGIC_VECTOR(t_int_range);        --! line to detect interrupt requests willing to stimulate the RT system
		 IPRI_LD		  : in  STD_LOGIC;                            --! line to enable update of an interrupt configuration 
		 IPRI_LD_ADDR	: in  STD_LOGIC_VECTOR(t_int_width_range);  --! line to address the interrupt configuration of which is going to be updated
		 IPRI_LD_DATA	: in  STD_LOGIC_VECTOR(t_pri_width_range);  --! line to adjust data going to be written into the configuration 
		 CLK_1M    		: in  STD_LOGIC;                            --! clock signal line
		 RST			    : in  STD_LOGIC);                           --! reset signal line
end intMAN_design;

--
--
--
--! intMAN unit architecture (inner structure)     
--
architecture arch of intMAN_design is
  
  --! interrupt detect/buffer/config/forward for the intMAN hardware     
	component int_buf_in
		port
		(
			reset        : in  STD_LOGIC;                            --! async reset   								
			level        : in  t_intlevel_cfg; 								       --! level-sensitivity select: log.0 (00), log. 1 (01), no (10, 11)
			edge         : in  t_intedge_cfg; 								       --! edge-sensitivity select: rising (01), falling (10), both (11), no (00)
			clk          : in  STD_LOGIC;                            --! clock
			int_in       : in  STD_LOGIC;   								         --! incomming interrupt line
			int_rdy      : out STD_LOGIC;   								         --! line to signalize there is a ready/pending interrupt 
			int_cnt      : out t_intpend_cnt;     						       --! number of ready/pending interrupts
			int_pri_load : in  STD_LOGIC;   							           --! priority update enable/disable
			int_pri_new  : in  STD_LOGIC_VECTOR(t_pri_width_range);  --! new interrupt priority	
			int_pri      : out STD_LOGIC_VECTOR(t_pri_width_range);	 --! interrupt priority storage		
			int_sel      : in STD_LOGIC;										         --! interrupt forward logic enable 
			int_out      : out STD_LOGIC        							       --! line to forward the interrupt 
		);
	end component;
	
  --! interrupt address unit for the intMAN hardware     
	component ipri_update_dmx
		port
		(
			sel           : in  std_logic;                           --! unit enable 
			int_addr      : in  STD_LOGIC_VECTOR(t_int_width_range); --! selected interrupt address
			ipri_load_sel : out STD_LOGIC_VECTOR(t_int_range)        --! address dmx output with no more than one interrupt selected  
		);
	end component;

  --! highest-priority pending interrupt selector for the intMAN hardware     
	component ipri_highest
		port 
		(
			IPRI_ARR : 	in t_intpri_arr;							            --! interrupt priorities
			IPRI_HIGH: 	out STD_LOGIC_VECTOR(t_pri_width_range);	--! the highest pending-interrupt priority
			IPRI_ADEC: 	out STD_LOGIC_VECTOR(t_int_range)         --! address dmx output with the highest-priority interrupt selected 
		);
	end component;

  --! priority-condition check for the intMAN hardware      
	component c_prio
	port (
		monpri : in  STD_LOGIC_VECTOR(t_pri_width_range);    --! priority of the running task (being monitored) 
		hstipri : in  STD_LOGIC_VECTOR(t_pri_width_range);   --! the highest pending-interrupt priority 
		valid : out STD_LOGIC                                --! priority-condition result
	);
	end component;

  --! slack-condition check for the intMAN hardware      
	component c_slack
	port (
		monpri   : in  STD_LOGIC_VECTOR(t_pri_width_range);   --! priority of the running task (being monitored) 
		monslack : in  STD_LOGIC;                             --! slack signal (being monitored)
		valid    : out STD_LOGIC                              --! slack-condition result
	);
	end component;

--
--
--
-- inner signals     
--

--! priority config update enable (signal)
signal SIG_IPRILOAD_SEL     : STD_LOGIC_VECTOR(t_int_range);
        
--! interrupt service start (signal)      
signal SIG_ISERV            : STD_LOGIC_VECTOR(t_int_range);
               
--! pending interrupt count (signal)
signal SIG_ICNT             : t_intpendcnt_arr;                            

--! interrupt ready (signal)
signal SIG_IRDY             : t_intrdy_vec;                                

--! interrupt config storage (signal)
signal SIG_ICFG_MEM         : t_intcfg_arr;                                 

--! interrupt config bus (signal)
signal SIG_ICFG             : t_intcfg_arr;                                

--! interrupt priorities (signal)
signal SIG_IPRIARR          : t_intpri_arr;                         

--! the highest pending-interrupt priority (signal)
signal SIG_IPRIHVAL         : STD_LOGIC_VECTOR(t_pri_width_range);	       

--! address dmx output with the highest-priority interrupt selected (signal)
signal SIG_INTSEL           : STD_LOGIC_VECTOR(t_int_range);               

--! priority-condition result (signal)
signal SIG_IPRICOND_VALID   : STD_LOGIC;                                   

--! slack-condition result (signal)
signal SIG_SLACKCOND_VALID  : STD_LOGIC;                                   




begin
	SIG_ISERV <= (OTHERS => '0');

  --! interrupt config storage unit init
	p_icfg: process(RST) is
	begin
		if(RST='1') then
			SIG_ICFG_MEM <= C_INTCFG;    -- after-reset init
		end if;
	end process p_icfg;
	
	SIG_ICFG <= SIG_ICFG_MEM;
	
  --
  --
  --
  -- port map section     
  --
  
  --! interrupt config selector unit (port map)
	IPUPDMX : ipri_update_dmx							
		port map 
		(
			sel   => IPRI_LD,
			int_addr => IPRI_LD_ADDR,
			ipri_load_sel => SIG_IPRILOAD_SEL
		);

  --! interrupt detect/buffer/config/forward unit (port map)
	GEN_IBUF : for I in t_int_range generate	
			INT_BUF: int_buf_in
				port map 
				(
					reset => RST,        
					level => SIG_ICFG(I).level,      
					edge => SIG_ICFG(I).edge,      
					clk => CLK_1M,          
					int_in => INT_IN(I),       
					int_rdy => SIG_IRDY(I),      
					int_cnt => SIG_ICNT(I),      
					int_pri_load => SIG_IPRILOAD_SEL(I),
					int_pri_new => IPRI_LD_DATA,  
					int_pri => SIG_IPRIARR(I),
					int_sel => SIG_IRDY(I) and SIG_INTSEL(I) and (SIG_IPRICOND_VALID or SIG_SLACKCOND_VALID),
					int_out => INT_OUT(I)					
				);
	end generate;

  --! highest-priority pending interrupt unit (port map)
	IP_HIGH : ipri_highest
		port map 
		(
			IPRI_ARR => SIG_IPRIARR, 
			IPRI_HIGH => SIG_IPRIHVAL,	
			IPRI_ADEC => SIG_INTSEL
		);

  --! priority condition check unit (port map)
	CPRI : c_prio
		port map
		(
			monpri => MON_PRI,
			hstipri => SIG_IPRIHVAL,
			valid => SIG_IPRICOND_VALID
		);
		
  --! priority condition check unit (port map)
	CSLACK: c_slack 
		port map
		(
			monpri  => MON_PRI,
			monslack => MON_SLACK,
			valid => SIG_SLACKCOND_VALID
		);
end arch;
