-------------------------------------------------------
--! @file 
--! @brief Interrupt detect/buffer/config/forward for the intMAN hardware
--! @author Josef Strnadel, Brno University of Technology, Faculty of Information Technology
--! @email strnadel@fit.vutbr.cz
--! @date 2013-04-22
-------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use work.intMAN_package.all;

--
--
--
--! Entity of the unit for detection/latch of an interrupt request (interface)     
--
entity int_req is
	Port(
		reset   : in  STD_LOGIC;        					                 --! async. reset
		level   : in  t_intlevel_cfg; 						                 --! level-sensitivity select: log.0 (00), log. 1 (01), no (10, 11)
		edge    : in  t_intedge_cfg; 							                 --! edge-sensitivity select: rising (01), falling (10), both (11), no (00)
		clk     : in  STD_LOGIC;									                 --! clock
		int_in  : in  STD_LOGIC;        					                 --! incomming interrupt line
		int_new : out STD_LOGIC;        					                 --! new interrupt-arrival flag
		int_rdy : out STD_LOGIC        						                 --! pending/ready interrupt flag
	);
end int_req;

--
--
--
--! Architecture of the unit for detection/latch of an interrupt request (inner structure)     
--
architecture arch of int_req is
  --! D-storage
	signal signal_d : STD_LOGIC;
  --! new int line
  signal SIG_new_int : STD_LOGIC;
  --! new int flag/latch
  signal SIG_new_int_latch : STD_LOGIC;

begin
		-- interrupt sensitivity legend:
		-- ----------------------------
		-- level: log.0 (00), log. 1 (01), no (10, 11)
		-- edge: rising (01), falling (10), both (11), no (00)
    
    --! process for detection/capturing of an arriving interrupt 
		process(clk)
		begin
			if (clk'event and clk = '1') then	                       -- rising edge
				signal_d <= int_in;
			end if;
		end process;
		
		SIG_new_int <= (not reset) and 
			(((not signal_d) and int_in and edge(0))					      -- rising edge detect + interrupt signal
			or ((not int_in) and signal_d and edge(1)) 					    -- falling edge detect + interrupt signal
			or ((not int_in) and (not level(1)) and (not level(0))) -- log.0-level detect + interrupt signal
			or (int_in and (not level(1)) and (level(0))) 				  -- log.1-level detect + interrupt signal
		);

    --! process for latching the pending interrupt request 
		newint_latch: process (reset, SIG_new_int)
		begin
		  if (SIG_new_int = '1') then
				SIG_new_int_latch <= '1';
		  elsif (reset = '1') then
				SIG_new_int_latch <= '0';
		  end if;
		end process;

		int_new <= SIG_new_int;
		int_rdy <= SIG_new_int_latch;
end arch;



----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use work.intMAN_package.all;

--
--
--
--! Entity of the interrupt detect/buffer/config/forward unit for the intMAN hardware (interface)     
--
entity int_buf_in is
	Port(
			reset        : in  STD_LOGIC;                            --! async reset   								
			level        : in  t_intlevel_cfg; 								       --! level-sensitivity select: log.0 (00), log. 1 (01), no (10, 11)
			edge         : in  t_intedge_cfg; 								       --! edge-sensitivity select: rising (01), falling (10), both (11), no (00)
			clk          : in  STD_LOGIC;                            --! clock
			int_in       : in  STD_LOGIC;   								         --! incomming interrupt line
			int_rdy      : out STD_LOGIC;   								         --! line to signalize there is a ready/pending interrupt 
			int_cnt      : out t_intpend_cnt;     						       --! number of ready/pending interrupts
			int_pri_load : in  STD_LOGIC;   							           --! priority update enable/disable
			int_pri_new  : in  STD_LOGIC_VECTOR(t_pri_width_range);  --! new interrupt priority	
			int_pri      : out STD_LOGIC_VECTOR(t_pri_width_range);	 --! interrupt priority storage		
			int_sel      : in STD_LOGIC;										         --! interrupt forward logic enable 
			int_out      : out STD_LOGIC        							       --! line to forward the interrupt 
	);
end int_buf_in;

--
--
--
--! Architecture of the interrupt detect/buffer/config/forward unit for the intMAN hardware (inner structure)     
--
architecture arch of int_buf_in is

  --! component for detection/latch of an interrupt request	
  component int_req
      port(
		  reset   : in  STD_LOGIC;        					               --! async. reset
		  level   : in  t_intlevel_cfg; 						               --! level-sensitivity select: log.0 (00), log. 1 (01), no (10, 11)
		  edge    : in  t_intedge_cfg; 							               --! edge-sensitivity select: rising (01), falling (10), both (11), no (00)
		  clk     : in  STD_LOGIC;									               --! clock
		  int_in  : in  STD_LOGIC;        					               --! incomming interrupt line
		  int_new : out STD_LOGIC;        					               --! new interrupt-arrival flag
		  int_rdy : out STD_LOGIC        						               --! pending/ready interrupt flag
		);
	end component;

  --! D-latch 
	signal signal_d    : STD_LOGIC;
  --! interrupt ready
	signal SIG_rdy     : STD_LOGIC;
  --! interrupt request reset
	signal SIG_rst     : STD_LOGIC;
  --! pending interrupts counter
	signal SIG_cnt     : t_intpend_cnt;
  --! priority
	signal SIG_pri     : STD_LOGIC_VECTOR(t_pri_width_range);

  --! level match flag
	signal SIG_int_l   : STD_LOGIC;
  --! edge match flag
	signal SIG_int_e   : STD_LOGIC;
  --! incomming interrupt line
	signal SIG_int     : STD_LOGIC;
  --! new interrupt flag
	signal SIG_int_new : STD_LOGIC;
  --! to service forward flag
	signal SIG_service : STD_LOGIC;

begin
    --
    --
    --
    -- port map section     
    --
  	
    --! mapping ports to the interrupt request detect/latch unit (port map)
    INTREQ : int_req							
		port map 
		(
			reset => SIG_rst,
			level => level,
			edge => edge,
			clk => clk,
			int_in => int_in,
			int_new => SIG_int_new,
			int_rdy => SIG_rdy
		);
	
	SIG_rst <= reset or SIG_service;

  --! control process of the interrupt detect/buffer/config/forward unit 
	process(reset, clk, int_pri_load, int_sel)	
	begin
		if(reset = '1') then                                    -- reset (async.)
			SIG_cnt <= 0;
		elsif(SIG_int_new'event and SIG_int_new = '1') then     -- new interrupt arrival (async.)
			SIG_cnt <= SIG_cnt+1;
		elsif(clk'event and clk='1') then                       -- clk-synchronous storage operations 
			if (int_pri_load = '1') then                          -- ... interrupt priority update enable 
				SIG_pri <= int_pri_new;
			elsif(int_sel = '1') then	                            -- ... if an interrupt is to be forwarded, signalize it
				SIG_service <= '1';
				
				if (SIG_cnt > 0) then                               
					SIG_cnt <= SIG_cnt - 1;                           -- ... and decrease the pending-interrupt counter
				end if;
			else 
				SIG_service <= '0';                                 
			end if;

  		-- interrupt sensitivity legend:
  		-- ----------------------------
  		-- level: log.0 (00), log. 1 (01), no (10, 11)
  		-- edge: rising (01), falling (10), both (11), no (00)
		
			case level is
				when "00" => SIG_int_l <= ('0' or not int_sel);
				when "01" => SIG_int_l <= ('1' and int_sel);
				when others => SIG_int_l <= SIG_int_l or '0';
			end case;

			case edge is
				when "01" => SIG_int_e <= ('1' and int_sel);
				when "10" => SIG_int_e <= ('0' or not int_sel);
				when "11" => SIG_int_e <= ('1' and int_sel);
				when others => SIG_int_e <= SIG_int_e or '0';
			end case;

			int_out <= SIG_int_l or SIG_int_e;                    -- forward the interrupt according to its config 
		end if;
	end process;

	gen1: for i in 0 to int_pri'LENGTH-1 generate		          -- interrupt priority is valid only if the request is pending/ready
		int_pri(i) <= (SIG_pri(i) or (not SIG_rdy));
	end generate gen1;

	int_cnt <= SIG_cnt;
	int_rdy <= SIG_rdy;

end arch;
