/*******************************************************\
* Copyright (C) 2006, ApS s.r.o Brno, AllRightsReserved *
\*******************************************************/

#define DEBUG_TYPE "delay-slot-filler"

#include "Codasip.h"
#include "CodasipTargetMachine.h"
#include "llvm/CodeGen/MachineFunctionPass.h"
#include "llvm/CodeGen/MachineInstrBuilder.h"
#include "llvm/Target/TargetInstrInfo.h"
#include "llvm/ADT/Statistic.h"

#include <set>

using namespace llvm;

namespace {
  struct Filler : public MachineFunctionPass {

    TargetMachine &TM;
    const TargetInstrInfo *TII;

    static char ID;
    Filler(TargetMachine &tm) 
      : MachineFunctionPass(ID), TM(tm), TII(tm.getInstrInfo()) { }

    virtual const char *getPassName() const {
      return "Codasip Delay Slot Filler";
    }

    bool runOnMachineBasicBlock(MachineFunction::iterator FI);
    bool runOnMachineFunction(MachineFunction &F) {
      bool Changed = false;
      for (MachineFunction::iterator FI = F.begin(), FE = F.end();
           FI != FE; ++FI)
        Changed |= runOnMachineBasicBlock(FI);
      return Changed;
    }

  };
  char Filler::ID = 0;
} // end of anonymous namespace

/// runOnMachineBasicBlock - Fill in delay slots for the given basic block.
/// Currently, we fill delay slots with NOPs. We assume there is only one
/// delay slot per delayed instruction.
bool Filler::
runOnMachineBasicBlock(MachineFunction::iterator FI) 
{
  MachineBasicBlock &MBB = *FI;
  bool Changed = false;
  
  for (MachineBasicBlock::iterator I = MBB.begin(); I != MBB.end(); ++I)
  {
    // how many nops?
    unsigned nops = 0;
    // get latency
    unsigned ici = I->getDesc().getSchedClass();
    // delay slot? (has precedency)
    if (I->getDesc().hasDelaySlot())
    {
      nops = ici;
      // branch at the end?
      const MCInstrDesc &dsc = I->getDesc();
      if (I == --MBB.end() && dsc.isBranch())
      {
        MachineOperand *op = NULL;
        for (unsigned i=0; i<I->getNumExplicitOperands(); ++i)
        {
          MachineOperand &op_ = I->getOperand(i);
          if (op_.isMBB()) {
            assert(op==NULL && "multiple targets, WTF?");
            op = &op_;
          }
        }
        // anything?
        if (op)
        {
          // the next?
          MachineBasicBlock *tar = op->getMBB();
          assert(tar);
          MachineFunction *mf = MBB.getParent();
          MachineFunction::iterator next = FI;
          ++next;
          if (next!=mf->end() && tar==next) {
            // we are at the end, we can afford this
            I->eraseFromParent();
            break;
          }
        }
      }
    }
    // register hazard?
    else
    {
      // get output register, if any
      std::set<unsigned> regs;
      for (unsigned i=0; i<I->getNumOperands(); ++i)
      {
        const MachineOperand &op = I->getOperand(i);
        if (op.isReg() && op.isDef()) {
          // calls won't get here, should be ok
          assert(regs.count(op.getReg())==0);
          regs.insert(op.getReg());
        }
      }
      // look forward
      if (!regs.empty())
      {
        MachineBasicBlock::iterator J = I;
        for (unsigned i=0; i<ici && ++J!=MBB.end(); ++i)
        {
          // used in input?
          for (unsigned j=0; j<J->getNumOperands(); ++j)
          {
            const MachineOperand &op = J->getOperand(j);
            if (op.isReg() && op.isUse()) {
              if (regs.count(op.getReg())) nops=i+1;
            }
          }
        }
      }
    }
    // insert nops
    if (nops)
    {
      MachineBasicBlock::iterator J = I; ++J;
      for (unsigned i=0; i<nops; ++i) TII->insertNoop(MBB,J);
      Changed = true;
    }
  }
  
  return Changed;
}

/// createCodasipDelaySlotFillerPass - Returns a pass that fills in delay
/// slots in Codasip MachineFunctions
FunctionPass *llvm::createCodasipDelaySlotFillerPass(CodasipTargetMachine &tm) {
  return new Filler(tm);
}

