/*****************************************************************************
 *
 * Copyright (C) 2007 Malaga university
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Alfonso Ariza Quintana
 *
 *****************************************************************************/

#include <string.h>
#include <assert.h>
#include "dymo_um_omnet.h"

#include "UDPPacket.h"
#include "IPControlInfo.h"
#include "IPv6ControlInfo.h"
#include "ICMPMessage_m.h"
#include "ICMPAccess.h"
#include "NotifierConsts.h"
#include "Ieee802Ctrl_m.h"
#include "Ieee80211Frame_m.h"


#include "ProtocolMap.h"
#include "IPAddress.h"
#include "IPvXAddress.h"
#include "ControlManetRouting.h"


const int UDP_HEADER_BYTES = 8;
typedef std::vector<IPAddress> IPAddressVector;

Define_Module(DYMOUM);

/* Constructor for the DYMOUM routing agent */

bool DYMOUM::log_file_fd_init=false;
int DYMOUM::log_file_fd = -1; 
bool DYMOUM::iswrite = false;
int DYMOUM::totalSend=0;
int DYMOUM::totalRreqSend=0;
int DYMOUM::totalRreqRec=0;
int DYMOUM::totalRrepSend=0;
int DYMOUM::totalRrepRec=0;
int DYMOUM::totalRrepAckSend=0;
int DYMOUM::totalRrepAckRec=0;
int DYMOUM::totalRerrSend=0;
int DYMOUM::totalRerrRec=0;

void NS_CLASS initialize(int stage)
{

	/*
	   Enable usage of some of the configuration variables from Tcl.

	   Note: Do NOT change the values of these variables in the constructor
	   after binding them! The desired default values should be set in
	   ~ns/tcl/lib/ns-default.tcl instead.
	 */
	if (stage==4)
	{

		sendMessageEvent = new cMessage();
		PromiscOperation = true;

	/* From main.c */
		progname = strdup("Dymo-UM");

	/* From debug.c */
	/* Note: log_nmsgs was never used anywhere */

		debug = 0;

		gateWayAddress = new IPAddress("0.0.0.0");
/* Set host parameters */
		memset(&this_host, 0, sizeof(struct host_info));
		memset(dev_indices, 0, sizeof(unsigned int) * DYMO_MAX_NR_INTERFACES);
		this_host.seqnum	= 1;
		this_host.nif		= 1;
		this_host.prefix	= 0;
		this_host.is_gw		= 0;

/* Search the 80211 interface */

		registerRoutingModule();
		NS_DEV_NR = getWlanInterfaceIndexByAddress();
		NS_IFINDEX = getWlanInterfaceIndexByAddress();

		for (int i = 0; i < DYMO_MAX_NR_INTERFACES; i++) 
		{
			DEV_NR(i).enabled=0;
		}

		for (int i = 0; i <getNumInterfaces(); i++)
		{
			DEV_NR(i).ifindex = i;
			dev_indices[getWlanInterfaceIndex(i)] = i;
			strcpy(DEV_NR(i).ifname, getInterfaceEntry(i)->getName());
			DEV_NR(i).ipaddr.s_addr = getInterfaceEntry(i)->ipv4Data()->getIPAddress().getInt();
		}
/* Set network interface parameters */
		for (int i=0;i < getNumWlanInterfaces();i++)
		{
			DEV_NR(getWlanInterfaceIndex(i)).enabled = 1;
			DEV_NR(getWlanInterfaceIndex(i)).sock = -1;
			DEV_NR(getWlanInterfaceIndex(i)).bcast.s_addr = DYMO_BROADCAST;
		}

		no_path_acc = 0;
		reissue_rreq = 0;
		s_bit = 0;
		
		if ((bool)par("no_path_acc_"))
			no_path_acc = 1; 
		if ((bool)par("reissue_rreq_"))
			reissue_rreq = 1;
		if ((bool)par("s_bit_"))
			s_bit = 1;
		hello_ival = par("hello_ival_");

		
		
		

		if ((DYMO_RATELIMIT = (int) par("MaxPktSec"))==-1)
			DYMO_RATELIMIT = 10;
		if (DYMO_RATELIMIT>50)
			DYMO_RATELIMIT=50;
		if ((NET_DIAMETER = (int) par("NetDiameter"))==-1)
			NET_DIAMETER = 10;
		if ((ROUTE_TIMEOUT = (long) par("RouteTimeOut"))==-1)
			ROUTE_TIMEOUT = 5000;
		if ((ROUTE_DELETE_TIMEOUT = (long) par("RouteDeleteTimeOut"))==-1)
			ROUTE_DELETE_TIMEOUT = 2*ROUTE_TIMEOUT;

		if ((RREQ_WAIT_TIME = (long) par("RREQWaitTime"))==-1)
			RREQ_WAIT_TIME = 2000;
		if ((RREQ_TRIES = (int) par("RREQTries"))==-1)
			RREQ_TRIES = 3;
		

		numInterfacesActive=1;

		ipNodeId = new IPAddress(interface80211ptr->ipv4Data()->getIPAddress());

		INIT_DLIST_HEAD(&TQ);
		INIT_DLIST_HEAD(&PENDING_RREQ);
		INIT_DLIST_HEAD(&BLACKLIST);
		INIT_DLIST_HEAD(&NBLIST);

		rtable_init();

		if (hello_ival<=0)
			linkLayerFeeback();
		if ((bool) par("promiscuous"))
			linkPromiscuous();

		norouteBehaviour = par("noRouteBehaviour");
		strcpy(nodeName,getParentModule()->getParentModule()->getFullName());
		dymo_socket_init();
		rtable_init();
		packet_queue_init();
		startDYMOUMAgent();
		is_init=true;
		// Initialize the timer
		scheduleNextEvent();
		ev << "Dymo active" << "\n";

	}
}

/* Destructor for the AODV-UU routing agent */
NS_CLASS ~ DYMOUM()
{
	rtable_destroy();
	cancelAndDelete(sendMessageEvent);
    //log_cleanup();
	if (gateWayAddress)
		delete gateWayAddress;
	if (ipNodeId)
		delete ipNodeId;
	while (!macToIpAdress.empty())
	{
		MacToIpAddress::iterator i = macToIpAdress.begin();
		// delete (*i).second;
		macToIpAdress.erase(i);
	}
}

/*
  Moves pending packets with a certain next hop from the interface
  queue to the packet buffer or simply drops it.
*/


/* Entry-level packet reception */
void NS_CLASS handleMessage (cMessage *msg)
{
	DYMO_element *dymoMsg=NULL;
	IPDatagram * ipDgram=NULL;
	UDPPacket * udpPacket=NULL;
	IPControlInfo *controlInfo=NULL;
	cMessage *msg_aux;
	struct in_addr src_addr;
	struct in_addr dest_addr;
	
	if (is_init==false)
		opp_error ("Aodv has not been initialized ");
	if (msg==sendMessageEvent)
	{
	// timer event
		scheduleNextEvent();
		return;
	}
	/* Handle packet depending on type */
	if (dynamic_cast<ControlManetRouting *>(msg)){
		ControlManetRouting * control =  check_and_cast <ControlManetRouting *> (msg);
		if (control->getOptionCode()== MANET_ROUTE_NOROUTE)
		{
			ipDgram = (IPDatagram*) control->decapsulate();
			EV << "Dymo rec datagram  " << ipDgram->getName() << " with dest=" << ipDgram->getDestAddress().str() << "\n";
			processPacket(ipDgram);   // Data path
		}
		else if (control->getOptionCode()== MANET_ROUTE_UPDATE)
		{
			src_addr.s_addr =control->getSrcAddress().getInt();
			dest_addr.s_addr = control->getDestAddress().getInt();
			rtable_entry_t *src_entry	= rtable_find(src_addr);
			rtable_entry_t *dest_entry	= rtable_find(dest_addr);
			rtable_update_timeout(src_entry);      
			rtable_update_timeout(dest_entry);     
		}
		delete msg;
		scheduleNextEvent();      
		return;
	}
	else if (dynamic_cast<UDPPacket *>(msg))
	{
		udpPacket = check_and_cast<UDPPacket*>(msg);
		if (udpPacket->getDestinationPort()!= DYMO_PORT)
		{
		   delete  msg;
		   scheduleNextEvent();
		   return;
		}
		msg_aux  = udpPacket->decapsulate();

		if (dynamic_cast<DYMO_element  *>(msg_aux))
		{
			controlInfo = check_and_cast<IPControlInfo*>(udpPacket->removeControlInfo());
			dymoMsg = check_and_cast  <DYMO_element *>(msg_aux);
			src_addr.s_addr = controlInfo->getSrcAddr().getInt();
			dymoMsg->setControlInfo(controlInfo);
		}
		else
		{
			delete udpPacket;
			delete msg_aux;
			if (controlInfo!=NULL)
				delete controlInfo;
			scheduleNextEvent();
			return;

		}
		delete udpPacket;
	}
	else
	{
		delete msg;
		scheduleNextEvent();
		return;
	}
	/* Detect routing loops */
	if (src_addr.s_addr == getRouterId())
	{
		dymoMsg->removeControlInfo();
		delete controlInfo;
		delete dymoMsg;
		dymoMsg=NULL;
		scheduleNextEvent();
		return;
	}

	recvDYMOUMPacket(dymoMsg);
	scheduleNextEvent();
}


/* Starts the Dymo routing agent */
int NS_CLASS startDYMOUMAgent()
{

	/* Set up the wait-on-reboot timer */
/*
	if (wait_on_reboot) {
		timer_init(&worb_timer, &NS_CLASS wait_on_reboot_timeout, &wait_on_reboot);
		timer_set_timeout(&worb_timer, DELETE_PERIOD);
		DEBUG(LOG_NOTICE, 0, "In wait on reboot for %d milliseconds.",DELETE_PERIOD);
	}
*/
	/* Schedule the first HELLO */
	//if (!llfeedback && !optimized_hellos)
	hello_init();

	/* Initialize routing table logging */

	/* Initialization complete */
	initialized = 1;
/*
	DEBUG(LOG_DEBUG, 0, "Routing agent with IP = %s : %d started.",
		  ip_to_str(DEV_NR(NS_DEV_NR).ipaddr), DEV_NR(NS_DEV_NR).ipaddr);

	DEBUG(LOG_DEBUG, 0, "Settings:");
	DEBUG(LOG_DEBUG, 0, "unidir_hack %s", unidir_hack ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "rreq_gratuitous %s", rreq_gratuitous ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "expanding_ring_search %s", expanding_ring_search ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "local_repair %s", local_repair ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "receive_n_hellos %s", receive_n_hellos ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "hello_jittering %s", hello_jittering ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "wait_on_reboot %s", wait_on_reboot ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "optimized_hellos %s", optimized_hellos ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "ratelimit %s", ratelimit ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "llfeedback %s", llfeedback ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "internet_gw_mode %s", internet_gw_mode ? "ON" : "OFF");
	DEBUG(LOG_DEBUG, 0, "ACTIVE_ROUTE_TIMEOUT=%d", ACTIVE_ROUTE_TIMEOUT);
	DEBUG(LOG_DEBUG, 0, "TTL_START=%d", TTL_START);
	DEBUG(LOG_DEBUG, 0, "DELETE_PERIOD=%d", DELETE_PERIOD);
*/
	/* Schedule the first timeout */
	scheduleNextEvent();
	return 0;

}



// for use with gateway in the future
IPDatagram * NS_CLASS pkt_encapsulate(IPDatagram *p, IPAddress gateway)
{
	IPDatagram *datagram = new IPDatagram(p->getName());
	datagram->setByteLength(IP_HEADER_BYTES);
	datagram->encapsulate(p);

	// set source and destination address
	datagram->setDestAddress(gateway);

	IPAddress src = p->getSrcAddress();

	// when source address was given, use it; otherwise it'll get the address
	// of the outgoing interface after routing
	// set other fields
	datagram->setDiffServCodePoint(p->getDiffServCodePoint());
	datagram->setIdentification(p->getIdentification());
	datagram->setMoreFragments(false);
	datagram->setDontFragment (p->getDontFragment());
	datagram->setFragmentOffset(0);
	datagram->setTimeToLive(
		   p->getTimeToLive() > 0 ?
		   p->getTimeToLive() :
		   0);

	datagram->setTransportProtocol(IP_PROT_IP);
	return datagram;
}



IPDatagram *NS_CLASS pkt_decapsulate(IPDatagram *p)
{

	if (p->getTransportProtocol() == IP_PROT_IP) {
		IPDatagram *datagram = check_and_cast  <IPDatagram *>(p->decapsulate());
		datagram->setTimeToLive(p->getTimeToLive());
		delete p;
		return datagram;
	}
	return NULL;
}



/*
  Reschedules the timer queue timer to go off at the time of the
  earliest event (so that the timer queue will be investigated then).
  Should be called whenever something might have changed the timer queue.
*/
void NS_CLASS scheduleNextEvent()
{
	struct timeval *timeout;
	double delay;
	simtime_t timer;
	timeout = timer_age_queue();
	if (timeout)
	{
		delay  = (double)(((double)timeout->tv_usec/(double)1000000.0) +(double)timeout->tv_sec);
		timer = simTime()+delay;
		if (sendMessageEvent->isScheduled())
		{
			if (timer < sendMessageEvent->getArrivalTime()) 
			{
				cancelEvent(sendMessageEvent);
				scheduleAt(timer, sendMessageEvent);
			}
		}
		else
		{
			scheduleAt(timer, sendMessageEvent);
		}
	}
}




/*
  Replacement for if_indextoname(), used in routing table logging.
*/
const char *NS_CLASS if_indextoname(int ifindex, char *ifname)
{
 	InterfaceEntry *   ie;
	assert(ifindex >= 0);
	ie = getInterfaceEntry(ifindex);
	return ie->getName();
}


void NS_CLASS getMacAddress(IPDatagram *dgram)
{
	
	if (dgram)
	{
		mac_address macAddressConv;
		cObject * ctrl = dgram->removeControlInfo();
			
		if (ctrl!=NULL)
		{
			Ieee802Ctrl * ctrlmac = check_and_cast<Ieee802Ctrl *> (ctrl);
			memcpy (macAddressConv.address,ctrlmac->getSrc().getAddressBytes(),6);	/* destination eth addr	*/
			// memcpy (&dest,ctrlmac->getDest().getAddressBytes(),6);	/* destination eth addr	*/
			delete ctrl;
			MacToIpAddress::iterator it = macToIpAdress.find(macAddressConv);
			if (it==macToIpAdress.end())
			{
				unsigned int ip_src = dgram->getSrcAddress().getInt();
				macToIpAdress.insert(std::make_pair(macAddressConv,ip_src));
			}
		}
		delete dgram;
	}
}

void NS_CLASS recvDYMOUMPacket(cMessage * msg)
{
	struct in_addr src, dst;
	IPControlInfo *ctrl = check_and_cast<IPControlInfo *>(msg->removeControlInfo());
	IPvXAddress srcAddr = ctrl->getSrcAddr();
	IPvXAddress destAddr = ctrl->getDestAddr();
	src.s_addr = srcAddr.get4().getInt();
	dst.s_addr =  destAddr.get4().getInt();
	DYMO_element  *dymo_msg = check_and_cast<DYMO_element *> (msg);
	if (ctrl)
	{
		getMacAddress (ctrl->removeOrigDatagram());
		delete ctrl;
	}
	InterfaceEntry *   ie;
	for (int i = 0; i <getNumInterfaces(); i++)
	{
		ie = getInterfaceEntry(i);
		IPv4InterfaceData *ipv4data = ie->ipv4Data();
		if (ipv4data->getIPAddress().getInt()== src.s_addr)
		{
			delete   dymo_msg;
			return;
		}
	}
	recv_dymoum_pkt(dymo_msg,src);
}


void NS_CLASS processPacket(IPDatagram * p)
{
	struct in_addr dest_addr, src_addr;
	unsigned int ifindex;
	struct ip_data *ipd = NULL;

	ifindex = NS_IFINDEX;	/* Always use ns interface */
	
	ipd = NULL;			/* No ICMP messaging */
	bool isLocal=false;
	IPAddressVector phops;

	src_addr.s_addr = p->getSrcAddress().getInt();
	dest_addr.s_addr = p->getDestAddress().getInt();
	InterfaceEntry *   ie;
	
	if(!p->getSrcAddress().isUnspecified())
	{
		for (int i = 0; i < getNumInterfaces(); i++)
		{
			ie = getInterfaceEntry (i);
			IPv4InterfaceData *ipv4data = ie->ipv4Data();
			if (ipv4data->getIPAddress().getInt()== src_addr.s_addr)
			{
				isLocal=true;
			}
		}
	}
	else
		isLocal=true;
	ie = getInterfaceEntry(ifindex);
	phops = ie->ipv4Data()->getMulticastGroups();
	IPAddress mcastAdd;
	bool isMcast=false;
	for (unsigned int  i=0;i<phops.size();i++){
		mcastAdd = phops[i];
		if (dest_addr.s_addr == mcastAdd.getInt())
			isMcast=true;
	}
	/* If the packet is not interesting we just let it go through... */
	if (dest_addr.s_addr == DYMO_BROADCAST ||isMcast) {
		send(p,"to_ip");
		return;
	}
	rtable_entry_t *entry	= rtable_find(dest_addr);
	if (!entry || entry->rt_state == RT_INVALID) {
		// If I am the originating node, then a route discovery
		// must be performed
		if (isLocal) {
			packet_queue_add(p, dest_addr);
			route_discovery(dest_addr);
		}
		// Else we must send a RERR message to the source if
		// the route has been previously used
		else {
			struct in_addr addr;
			switch (norouteBehaviour)
			{
			case 3:
				packet_queue_add(p, dest_addr);
				route_discovery(dest_addr);
				break;
			case 2:
				// if (entry && entry->rt_is_used)
				mac_address macAddressConv;

				cObject * ctrl;
				ctrl = p->removeControlInfo();
				if (ctrl!=NULL)
				{
					Ieee802Ctrl * ctrlmac = check_and_cast<Ieee802Ctrl *> (ctrl);
					if (ctrlmac) {
						memcpy (macAddressConv.address,ctrlmac->getSrc().getAddressBytes(),6);	/* destination eth addr	*/
						// memcpy (&dest,ctrlmac->getDest().getAddressBytes(),6);	/* destination eth addr	*/
						delete ctrl;
						MacToIpAddress::iterator it = macToIpAdress.find(macAddressConv);
						if (it!=macToIpAdress.end())
						{
							addr.s_addr = (*it).second;
							rerr_send(dest_addr, 1, entry,addr);
						}
					}
				}
				delete p;
				break;
			case 1:
				rerr_send(dest_addr, NET_DIAMETER, entry);
			default:
			//	icmpAccess.get()->sendErrorMessage(p, ICMP_DESTINATION_UNREACHABLE, 0);
				delete p;
			break;
			}
		}
		scheduleNextEvent();
		return;
	}
	else
	{
		/* DEBUG(LOG_DEBUG, 0, "Sending pkt uid=%d", ch->uid()); */
		send(p,"to_ip");
		/* When forwarding data, make sure we are sending HELLO messages */
		//gettimeofday(&this_host.fwd_time, NULL);
		hello_init();
	}
}


/*
struct dev_info NS_CLASS dev_ifindex (int ifindex)
{
 int index = ifindex2devindex(ifindex);
 return  (this_host.devs[index]);

}

struct dev_info NS_CLASS dev_nr(int n)
{
	return (this_host.devs[n]);
}

int NS_CLASS ifindex2devindex(unsigned int ifindex)
{
  int i;

  for (i = 0; i < this_host.nif; i++)
	if (dev_indices[i] == ifindex)
	  return i;

  return -1;
}
*/
void NS_CLASS processLinkBreak (const cPolymorphic *details)
{
	IPDatagram	*dgram=NULL;
	if (dynamic_cast<IPDatagram *>(const_cast<cPolymorphic*> (details))) 
		dgram = check_and_cast<IPDatagram *>(details);
	else
		return;
	if (hello_ival<=0)
    {
	     	packetFailed(dgram);
	}
}

void NS_CLASS processPromiscuous(const cPolymorphic *details)
{
	Ieee80211DataOrMgmtFrame *frame=NULL;
	if (dynamic_cast<Ieee80211DataOrMgmtFrame *>(const_cast<cPolymorphic*> (details))) 
	{
		mac_address macAddressConv;
		struct in_addr addr;
		frame  = check_and_cast<Ieee80211DataOrMgmtFrame *>(details);
		memcpy (macAddressConv.address,frame->getTransmitterAddress().getAddressBytes(),6);	
		MacToIpAddress::iterator it = macToIpAdress.find(macAddressConv);
		if (it!=macToIpAdress.end())
		{
			rtable_entry_t *entry;
			addr.s_addr = (*it).second;
			entry = rtable_find(addr);
			if (entry)
			{
				rtable_update(entry,			// routing table entry
				addr,	// dest
				addr,	// nxt hop
				NS_DEV_NR,		// iface
				entry->rt_seqnum,			// seqnum
				entry->rt_prefix,		// prefix
				1,	// hop count
				entry->rt_is_gw);		// is gw 
			//rtable_update_timeout(entry);
			}			
		}
		else
			return; // can procces the message, don't know the sender
			// if rrep proccess the packet

		if (!no_path_acc)
		{
			IPDatagram * ip_msg = dynamic_cast<IPDatagram *>(frame->getEncapsulatedMsg());
			if (ip_msg)
			{
				if (ip_msg->getTransportProtocol()==IP_PROT_MANET)
				{
					DYMO_element * dymo_msg = dynamic_cast<DYMO_element *>(ip_msg->getEncapsulatedMsg()->getEncapsulatedMsg());
					if (dymo_msg)
					{
				// check if RREP 
						if ((dymo_msg->type==DYMO_RE_TYPE) && (((RE *) dymo_msg)->a==0))
						{
						//	proccess RREP
							addr.s_addr =ip_msg->getSrcAddress().getInt();
							promiscuous_rrep((RE*)dymo_msg,addr);
						} // end if promiscuous
						//else if (dymo_msg->type==DYMO_RERR_TYPE)
						//{
						//}
					}  // end if dymo msg
				} // end if manet msg
				// refress routes.
				//else
				//{
				//}
			} // end if ipDatagram
		} // end if no_path_acc
	}
}




void NS_CLASS promiscuous_rrep(RE * dymo_re,struct in_addr ip_src)
{
	struct in_addr node_addr;
	rtable_entry_t *entry;


	// check if my address is in the message
	int num_blk = dymo_re->numBlocks();
	for (int i=0;i<num_blk;i++)
		if (dymo_re->re_blocks[i].re_node_addr	== DEV_NR(NS_DEV_NR).ipaddr.s_addr)
			return;
	for (int i=0;i<num_blk;i++)
	{
		//if (dymo_re->re_blocks[i].re_hopcnt+1>2)
		//	continue;
		node_addr.s_addr	= dymo_re->re_blocks[i].re_node_addr;
		entry			= rtable_find(node_addr);
		int seqnum		= ntohl(dymo_re->re_blocks[i].re_node_seqnum);
		struct re_block b;
		memcpy (&b,&dymo_re->re_blocks[i],sizeof(struct re_block));
		b.re_hopcnt+=1;
		int rb_state = re_info_type(&b, entry,0);
		if (rb_state != RB_FRESH)
				continue;
		if (!entry) 
		/*if (entry) 
			rtable_update(
				entry,			// routing table entry
				node_addr,		// dest
				ip_src,			// nxt hop
				NS_DEV_NR,		// iface
				seqnum,			// seqnum
				b.prefix,		// prefix
				b.re_hopcnt,	// hop count
				b.g);		// is gw 
		else
		*/
			rtable_insert(
				node_addr,		// dest
				ip_src,			// nxt hop
				NS_DEV_NR,		// iface
				seqnum,			// seqnum
				b.prefix,		// prefix
				b.re_hopcnt,	// hop count
				b.g);		// is gw
	
	}
	// add this packet
}


/* Called for packets whose delivery fails at the link layer */
void NS_CLASS packetFailed(IPDatagram *dgram)
{
	rtable_entry_t *rt;
	struct in_addr dest_addr, src_addr, next_hop;

	src_addr.s_addr = dgram->getSrcAddress().getInt();
	dest_addr.s_addr = dgram->getDestAddress().getInt();

	/* We don't care about link failures for broadcast or non-data packets */
	if (dgram->getDestAddress().getInt() == IP_BROADCAST ||
		dgram->getDestAddress().getInt() == DYMO_BROADCAST) 
	{
		scheduleNextEvent();
		return;
	}
	ev << "LINK FAILURE for dest=" << dgram->getSrcAddress();
	rt = rtable_find(dest_addr);
	if (rt)
	{
		next_hop.s_addr = rt->rt_nxthop_addr.s_addr;
/*
	DEBUG(LOG_DEBUG, 0, "LINK FAILURE for next_hop=%s dest=%s uid=%d",
	  ip_to_str(next_hop), ip_to_str(dest_addr), ch->uid());
*/
		rtable_expire_timeout_all(next_hop, NS_IFINDEX);
	}
	else
		omnet_chg_rte(dest_addr,dest_addr, dest_addr,0,true);
	scheduleNextEvent();
}


void NS_CLASS finish()
{


    simtime_t t = simTime();
    if (t==0) return;

    if (iswrite)
        return;

    iswrite=true;

    recordScalar("simulated time", t);
    recordScalar("Dymo totalSend ", totalSend);

    recordScalar("Dymo Rreq send", totalRreqSend);
    recordScalar("Dymo Rreq rec", totalRreqRec);

    recordScalar("Dymo Rrep send", totalRrepSend);
    recordScalar("Dymo Rrep rec", totalRrepRec);
/*
    recordScalar("rrep ack send", totalRrepAckSend);
    recordScalar("rrep ack rec", totalRrepAckRec);
*/
    recordScalar("Dymo Rerr send", totalRerrSend);
    recordScalar("Dymo Rerr rec", totalRerrRec);
}


std::string NS_CLASS detailedInfo() const
{
	std::stringstream out;
	
	out << "Node  : "  << *ipNodeId  << "  " << ipNodeId->getInt() << "\n";
	out << "Seq Num  : "  <<this_host.seqnum  << "\n";

  	return out.str();
}


