/**
 * @file PimSplitter.h
 * @brief File contains implementation of PIMSplitter.
 * @date 3.12.2011
 * @author Veronika Rybova
 * @details Splitter is common for all PIM modes. It is used to resent all PIM messages to
 *  correct PIM mode module. It also does work which is same for all modes, e.g.
 *  it send Hello messages and it manages table of PIM interfaces.
 */

#ifndef PIMSPLITTER_H_
#define PIMSPLITTER_H_

#include <omnetpp.h>
#include "PIMPacket_m.h"
#include "PIMTimer_m.h"
#include "IPControlInfo.h"
#include "IPv4InterfaceData.h"
#include "AnsaInterfaceTableAccess.h"
#include "MulticastRoutingTableAccess.h"
#include "RoutingTableAccess.h"
#include "AnsaInterfaceTable.h"
#include "AnsaRoutingTable.h"
#include "NotificationBoard.h"
#include "NotifierConsts.h"
#include "InterfaceStateManager.h"
#include "IPvXAddress.h"
#include "PimNeighborTable.h"
#include "PimInterfaceTable.h"


#define HT 30.0										/**< Hello Timer = 30s. */

/**
 * @brief Class implements PIM Splitter, which splits PIM messages to correct PIM module.
 * @details This module is needed because we cannot distinguish PIM mode on layer 3, all of
 * them have same protocol number (103). PIM Splitter can resend PIM message to correct
 * PIM module according to configuration saved in PimInterfaceTable. Splitter also manages
 * PimNeighborTable.
 */
class PimSplitter : public cSimpleModule, protected INotifiable
{
	private:
		IRoutingTable           	*rt;           	/**< Pointer to routing table. */
		MulticastRoutingTable 		*mrt;			/**< Pointer to multicast routing table. */
	    IInterfaceTable         	*ift;          	/**< Pointer to interface table. */
	    NotificationBoard 			*nb; 		   	/**< Pointer to notification table. */
	    PimInterfaceTable			*pimIft;		/**< Pointer to table of PIM interfaces. */
	    PimNeighborTable			*pimNbt;		/**< Pointer to table of PIM neighbors. */
	    const char *				hostname;      	/**< Router hostname. */

	   void processPIMPkt(PIMPacket *pkt);
	   void processNLTimer(PIMTimer *timer);

	   // methods for Hello packets
	   PIMHello* createHelloPkt(int iftID);
	   void sendHelloPkt();
	   void processHelloPkt(PIMPacket *pkt);

	   // process notification
	   void receiveChangeNotification(int category, const cPolymorphic *details);
	   virtual void newMulticast(IPAddress destAddr, IPAddress srcAddr);
	   void igmpChange(InterfaceEntry *interface);

	   // not in use
	   bool LoadConfigFromXML(const char *filename);

	protected:
		virtual int numInitStages() const  {return 5;}
		virtual void handleMessage(cMessage *msg);
		virtual void initialize(int stage);

	public:
		PimSplitter(){};
};


#endif /* PIMSPLITTER_H_ */


/**
 * @mainpage Multicast Routing Modelling in OMNeT++ Documentation
 *
 * This is programming documentation to thesis Multicast Routing Modelling in OMNeT++ by Veronika Rybova.
 * In this documentation you can see whole C++ programming part of the thesis. You do not find here description
 * of NED files and classes which are autogenerated from files PIMTimer.msg and PIMPacket.msg. The reason is that
 * the output of these files is big amount of classes and documentation would be overwhelmed.
 *
 * The thesis is part of project ANSA, which takes place at the Faculty of Information Technology, Brno University
 * of Technology. For more information visit: https://nes.fit.vutbr.cz/ansa/pmwiki.php
 */
