/**
 * @file AnsaIP.cc
 * @brief File contains extension of class IP, which can work also with multicast data and multicast
 * @author Veronika Rybova
 * @date 10.10.2011
 */


#include <omnetpp.h>
#include "AnsaIP.h"
#include "PimSplitter.h"

Define_Module(AnsaIP);


/**
 * INITIALIZE
 *
 * The method initialize ale structures (tables) which will use.
 *
 * @param stage Stage of initialization.
 */
void AnsaIP::initialize(int stage)
{
	INET_API IP::initialize();

	mrt = MulticastRoutingTableAccess().get();
	nb = NotificationBoardAccess().get();

}

/**
 * HANDLE PACKET FROM NETWORK
 *
 * Extension of method handlePacketFromNetwork() from class IP. It stops deleting of IP Control Info,
 * which will be needed in routeMulticastPacket(). All neccessery info is added to IP Control Info.
 * Datagram with protocol number 103 (IP_PROT_PIM) are sent directly to the PIM module.
 *
 * All part which I added are signed by MYWORK tag.
 *
 * @param datagram Pointer to incoming datagram.
 * @see routeMulticastPacket()
 */
void AnsaIP::handlePacketFromNetwork(IPDatagram *datagram)
{
    //
    // "Prerouting"
    //

    // check for header biterror
    if (datagram->hasBitError())
    {
        // probability of bit error in header = size of header / size of total message
        // (ignore bit error if in payload)
        double relativeHeaderLength = datagram->getHeaderLength() / (double)datagram->getByteLength();
        if (dblrand() <= relativeHeaderLength)
        {
            EV << "bit error found, sending ICMP_PARAMETER_PROBLEM\n";
            icmpAccess.get()->sendErrorMessage(datagram, ICMP_PARAMETER_PROBLEM, 0);
            return;
        }
    }

    // remove control info
    if (datagram->getTransportProtocol()!=IP_PROT_DSR && datagram->getTransportProtocol()!=IP_PROT_MANET && !datagram->getDestAddress().isMulticast() && datagram->getTransportProtocol()!=IP_PROT_PIM)
    {
        delete datagram->removeControlInfo();
    }
    else if (datagram->getMoreFragments())
    	delete datagram->removeControlInfo(); // delete all control message except the last

    //MYWORK Add all neccessery info to the IP Control Info for future use.
    if (datagram->getDestAddress().isMulticast() || datagram->getTransportProtocol() == IP_PROT_PIM)
    {
    	IPControlInfo *ctrl = (IPControlInfo*)(datagram->removeControlInfo());
    	ctrl->setSrcAddr(datagram->getSrcAddress());
    	ctrl->setDestAddr(datagram->getDestAddress());
    	ctrl->setInterfaceId(getSourceInterfaceFrom(datagram)->getInterfaceId());
    	datagram->setControlInfo(ctrl);
    }

    // hop counter decrement; FIXME but not if it will be locally delivered
    datagram->setTimeToLive(datagram->getTimeToLive()-1);

    // send IGMP packet to IGMP module
    if (datagram->getTransportProtocol() == IP_PROT_IGMP)
    {
       	cPacket *packet = decapsulateIP(datagram);
        send(packet, "transportOut", mapping.getOutputGateForProtocol(IP_PROT_IGMP));
	return;
    }

    //MYWORK send PIM packet to PIM module
    if (datagram->getTransportProtocol() == IP_PROT_PIM)
	{
		cPacket *packet = decapsulateIP(datagram);
		send(packet, "transportOut", mapping.getOutputGateForProtocol(IP_PROT_PIM));
	return;
	}
        
    //MYWORK route packet
    if (!datagram->getDestAddress().isMulticast())
        routePacket(datagram, NULL, false,NULL);
    else
        routeMulticastPacket(datagram, NULL, getSourceInterfaceFrom(datagram));
}

/**
 * ROUTE MULTICAST PACKET
 *
 * Extension of method routeMulticastPacket() from class IP. The method checks if data come
 * to RPF interface, if not it sends notification. Multicast data which are sent by this router
 * and has given outgoing interface are sent directly (PIM messages). The method finds route for
 * group. If there is no route, it will be added. Then packet is copied and sent to all outgoing
 * interfaces in route.
 *
 * All part which I added are signed by MYWORK tag.
 *
 * @param datagram Pointer to incoming datagram.
 * @param destIE Pointer to outgoing interface.
 * @param fromIE Pointer to incoming interface.
 * @see routeMulticastPacket()
 */
void AnsaIP::routeMulticastPacket(IPDatagram *datagram, InterfaceEntry *destIE, InterfaceEntry *fromIE)
{
    IPAddress destAddr = datagram->getDestAddress();
    IPAddress srcAddr = datagram->getSrcAddress();
    IPControlInfo *ctrl = (IPControlInfo *) datagram->getControlInfo();
    EV << "Routing multicast datagram `" << datagram->getName() << "' with dest=" << destAddr << "\n";
    MulticastIPRoute *route = mrt->getRouteFor(destAddr, srcAddr);

    numMulticast++;

    // Process datagram only if sent locally or arrived on the shortest
    // route (provided routing table already contains srcAddr) = RPF interface;
    // otherwise discard and continue.
    InterfaceEntry *rpfInt = rt->getInterfaceForDestAddr(datagram->getSrcAddress());
    if (fromIE!=NULL && rpfInt!=NULL && fromIE!=rpfInt)
    {
    	//MYWORK RPF interface has changed
    	/*if (route != NULL && (route->getInIntId() != rpfInt->getInterfaceId()))
    	{
    		EV << "RPF interface has changed" << endl;
    		nb->fireChangeNotification(NF_IPv4_RPF_CHANGE, route);
    	}*/
    	//MYWORK Data come to non-RPF interface
    	if (!rt->isLocalMulticastAddress(destAddr) && !destAddr.isLinkLocalMulticast())
    	{
    		EV << "Data on non-RPF interface" << endl;
    		nb->fireChangeNotification(NF_IPv4_DATA_ON_NONRPF, ctrl);
    		return;
    	}
    	else
    	{
			// FIXME count dropped
			EV << "Packet dropped." << endl;
			delete datagram;
			return;
    	}
    }

    //MYWORK for local traffic to given destination (PIM messages)
    if (fromIE == NULL && destIE != NULL)
    {
		IPDatagram *datagramCopy = (IPDatagram *) datagram->dup();
		datagramCopy->setSrcAddress(destIE->ipv4Data()->getIPAddress());
		fragmentAndSend(datagramCopy, destIE, destAddr);

    	delete datagram;
    	return;
    }

    // if received from the network...
    if (fromIE!=NULL)
    {
    	EV << "Packet was received from the network..." << endl;
        // check for local delivery (multicast assigned to any interface)
        if (rt->isLocalMulticastAddress(destAddr))
        {
        	EV << "isLocalMulticastAddress." << endl;
            IPDatagram *datagramCopy = (IPDatagram *) datagram->dup();

            // FIXME code from the MPLS model: set packet dest address to routerId
            datagramCopy->setDestAddress(rt->getRouterId());
            reassembleAndDeliver(datagramCopy);
        }

        // don't forward if IP forwarding is off
        if (!rt->isIPForwardingEnabled())
        {
        	EV << "IP forwarding is off." << endl;
            delete datagram;
            return;
        }

        // don't forward if dest address is link-scope
        // address is in the range 224.0.0.0 to 224.0.0.255
        if (destAddr.isLinkLocalMulticast())
        {
        	EV << "isLinkLocalMulticast." << endl;
            delete datagram;
            return;
        }
    }

//MYWORK(to the end) now: routing
    EV << "AnsaIP::routeMulticastPacket - Multicast routing." << endl;

    // multicast group is not in multicast routing table and has to be added
    if (route == NULL)
    {
    	EV << "AnsaIP::routeMulticastPacket - Multicast route does not exist, try to add." << endl;
    	nb->fireChangeNotification(NF_IPv4_NEW_MULTICAST, ctrl);
    	delete datagram->removeControlInfo();
    	ctrl = NULL;
    	// read new record
    	route = mrt->getRouteFor(destAddr, srcAddr);
    }

	if (route == NULL)
	{
		EV << "Still do not exist." << endl;
		delete datagram;
		return;
	}

	nb->fireChangeNotification(NF_IPv4_DATA_ON_RPF, route);

	// data won't be sent because there is no outgoing interface and/or route is pruned
    InterfaceVector outInt = route->getOutInt();
    if (outInt.size() == 0 || route->isFlagSet(P))
    {
    	EV << "Route does not have any outgoing interface or it is pruned." << endl;
    	if(ctrl != NULL)
    	{
			if (!route->isFlagSet(A))
				nb->fireChangeNotification(NF_IPv4_DATA_ON_PRUNED_INT, ctrl);
    	}
		delete datagram;
		return;
    }

    // send packet to all outgoing interfaces of route (oilist)
	for (unsigned int i=0; i<outInt.size(); i++)
	{
		// do not send to pruned interface
		if (outInt[i].forwarding == Pruned)
			continue;

		InterfaceEntry *destIE = outInt[i].intPtr;
		IPDatagram *datagramCopy = (IPDatagram *) datagram->dup();

		// set datagram source address if not yet set
		if (datagramCopy->getSrcAddress().isUnspecified())
			datagramCopy->setSrcAddress(destIE->ipv4Data()->getIPAddress());

		// send
		fragmentAndSend(datagramCopy, destIE, destAddr);
	}

	// only copies sent, delete original datagram
	delete datagram;
}

