/*
    Copyright (C) 2012  Stanislav Bárta

    This file is part of Bachelor's thesis: Creating Metadata during 
    Interception of Instant Messaging Communication.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * Soubor:  sprava.c
 * Datum:   16.05.2012
 * Autor:   Stanislav Bárta, xbarta29@stud.fit.vutbr.cz
 * Projekt: BP - tvorba metadat pri odposlechu komunikace v realnem case
 * Popis:   soubor obsahujici funkce pro spravu sezeni a spravu pozadavku na odposlechy
 */

#include "xmpp.h"
#include "sprava.h"

/*
 * otevre spojeni na kterem bude prijimat nove pozadavky na odposlechy
 */
int otevri_spojeni_pro_spravu_sledovani(char *host, char *port)
{
	struct addrinfo hints, *servinfo;

	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;

	if(getaddrinfo(host, port, &hints, &servinfo))
	{
		fprintf(stderr, "chyba pri ziskani informaci o adrese AF\n");
		return -1;
	}
	if((socket_sprava = socket(servinfo->ai_family, servinfo->ai_socktype, servinfo->ai_protocol)) == -1)
	{
		freeaddrinfo(servinfo);
		fprintf(stderr, "chyba pri vytvoreni socketu pro prijem zprav z AF\n");
		return -1;
	}
	if((connect(socket_sprava, servinfo->ai_addr, servinfo->ai_addrlen)) == -1)
	{
		freeaddrinfo(servinfo);
		fprintf(stderr, "chyba pri pripojeni k AF\n");
		return -1;
	}

	freeaddrinfo(servinfo);

	return 0;
}

/*
 * uzavreni spojeni pro spravu odposlechu
 * volano pri ukonceni programu
 */
void uzavri_spojeni_pro_spravu_sledovani(void)
{
	close(socket_sprava);
}

/*
 * inicializace noveho seznamu pozadovanych odposlechu
 */
void inicializace_seznamu_sledovani(struct seznam_sledovani *Seznam)
{
	Seznam->Aktivni = NULL;
	Seznam->Prvni = NULL;
}

/*
 * zpracuje nove prijaty pozadavek na odposlech a upravi podle nej pouzivane seznamy
 */
void zpracuj_novy_pozadavek_na_sledovani(char *pozadavek, struct seznam_sledovani *Seznam, struct seznam_sezeni *Sezeni)
{

	char *akce = NULL;
	char *protokol = NULL;
	char *typ = NULL;
	char *hodnota = NULL;

	int zacatek = 0;
	int i = 0;
	int delka_pozadavku = delka_retezce(pozadavek);

	// typ pozadavku
	while(pozadavek[i] != ';')
	{
		i++;
		if(i > delka_pozadavku)
		{
			return;
		}
	}
	kopie_retezce(pozadavek, &akce, i);
	// posunuti zacatku pro dalsi polozku
	i++;
	zacatek = i;

	// protokol
	while(pozadavek[i] != ';')
	{
		i++;
		if(i > delka_pozadavku)
		{
			free(akce);
			return;
		}
	}
	kopie_retezce(pozadavek+zacatek, &protokol, i-zacatek);
	// posunuti zacatku pro dalsi polozku
	i++;
	zacatek = i;

	// typ
	while(pozadavek[i] != ';')
	{
		i++;
		if(i > delka_pozadavku)
		{
			free(akce);
			free(protokol);
			return;
		}
	}
	kopie_retezce(pozadavek+zacatek, &typ, i-zacatek);
	// posunuti zacatku pro dalsi polozku
	i++;
	zacatek = i;

	// hodnota
	while(pozadavek[i] != '\0')
	{
		i++;
		if(i > delka_pozadavku)
		{
			free(akce);
			free(protokol);
			free(typ);
			return;
		}
	}
	kopie_retezce(pozadavek+zacatek, &hodnota, i-zacatek);


	int int_protokol = NEZNAME;
	if(porovnani_retezce(protokol, "XMPP"))
	{
		int_protokol = XMPP;
	}
	else if(porovnani_retezce(protokol, "IRC"))
	{
		int_protokol = IRC;
	}
	else if(porovnani_retezce(protokol, "OSCAR"))
	{
		int_protokol = OSCAR;
	}

	if(porovnani_retezce(akce, "PRIDAT"))
	{
		if(porovnani_retezce(typ, "IPv4"))
		{
			struct in_addr IP;
			inet_pton(AF_INET, hodnota, &(IP));

			if(!sledovani_IPv4(Seznam, IP, int_protokol))
			{

				vloz_sledovani_IPv4(Seznam, IP, int_protokol);
			}

		}
		else if(porovnani_retezce(typ, "IPv4-ROZSAH"))
		{
			int maska;
			for(int i = 0 ; i < delka_retezce(hodnota) ; i++)
			{
				if(hodnota[i] == '/')
				{
					hodnota[i] = '\0';
					maska = atoi(hodnota+i+1);
				}
			}

			struct in_addr IP;
			inet_pton(AF_INET, hodnota, &(IP));

			// na ziskanou IP adresu se aplikuje maska pro pripad ze nebyla zadana adresa site -> usnadneni dalsich kontrol
			IP.s_addr = ntohl(IP.s_addr);
			IP.s_addr = IP.s_addr >> (32-maska);
			IP.s_addr = IP.s_addr << (32-maska);
			IP.s_addr = htonl(IP.s_addr);

			if(!sledovani_IPv4_rozsah(Seznam, IP, maska, int_protokol))
			{
				vloz_sledovani_IPv4_rozsah(Seznam, IP, maska, int_protokol);
			}
		}
		else if(porovnani_retezce(typ, "ID"))
		{
			if(!sledovani_ID(Seznam, hodnota, int_protokol))
			{
				vloz_sledovani_ID(Seznam, hodnota, int_protokol);
			}
			else
			{
				free(hodnota);
			}
		}
	}
	else if(porovnani_retezce(akce, "ODEBRAT"))
	{

		if(porovnani_retezce(typ, "IPv4"))
		{
			struct in_addr IP;
			inet_pton(AF_INET, hodnota, &(IP));

			zrus_sledovani_IPv4(Seznam, IP, int_protokol);
			deaktivuj_sezeni_IPv4(Seznam, IP, 0, int_protokol, Sezeni);
		}
		else if(porovnani_retezce(typ, "IPv4-ROZSAH"))
		{
			int maska;
			for(int i = 0 ; i < delka_retezce(hodnota) ; i++)
			{
				if(hodnota[i] == '/')
				{
					hodnota[i] = '\0';
					maska = atoi(hodnota+i+1);
				}
			}
			struct in_addr IP;
			inet_pton(AF_INET, hodnota, &(IP));

			// na ziskanou IP adresu se aplikuje maska pro pripad ze nebyla zadana adresa site -> usnadneni dalsich kontrol
			IP.s_addr = ntohl(IP.s_addr);
			IP.s_addr = IP.s_addr >> (32-maska);
			IP.s_addr = IP.s_addr << (32-maska);
			IP.s_addr = htonl(IP.s_addr);

			zrus_sledovani_IPv4_rozsah(Seznam, IP, maska, int_protokol);
			deaktivuj_sezeni_IPv4(Seznam, IP, maska, int_protokol, Sezeni);
		}
		else if(porovnani_retezce(typ, "ID"))
		{
			zrus_sledovani_ID(Seznam, hodnota, int_protokol);
			deaktivuj_sezeni_ID(Seznam, hodnota, int_protokol, Sezeni);

			free(hodnota);
		}
	}
	free(akce);
	free(protokol);

	if(porovnani_retezce(typ , "IPv4") || porovnani_retezce(typ , "IPv4-ROZSAH"))
	{
		free(hodnota);
	}
	free(typ);
}

/*
 * smazani seznamu pozadovanych odposlechu
 */
void smazani_seznamu_sledovani(struct seznam_sledovani *Seznam)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	while(ptr != NULL)
	{
		Seznam->Prvni = ptr->dalsi;

		if(ptr->data.ID != NULL)
		{
			free(ptr->data.ID);
		}
		free(ptr);
		ptr = Seznam->Prvni;
	}
}

/*
 * do seznamu vlozi nove sledovani identifikovane podle IP
 */
void vloz_sledovani_IPv4(struct seznam_sledovani *Seznam, struct in_addr IP, int protokol)
{
	struct sledovany_cil_ptr *ptr = NULL;
	ptr = (struct sledovany_cil_ptr *) malloc(sizeof(struct sledovany_cil_ptr));
	ptr->dalsi = Seznam->Prvni;

	ptr->data.protokol = protokol;
	ptr->data.ID = NULL; // ID neni zadano
	ptr->data.IPv4 = IP;
	for(int i = 0; i < 16; i++){
		ptr->data.IPv6.s6_addr[i] = 0;
	}
	ptr->data.maska = 0;
	Seznam->Prvni = ptr;
}

/*
 * do seznamu vlozi nove sledovani identifikovane podle IP se zadanou maskou
 */
void vloz_sledovani_IPv4_rozsah(struct seznam_sledovani *Seznam, struct in_addr IP, int maska, int protokol)
{
	struct sledovany_cil_ptr *ptr = NULL;
	ptr = (struct sledovany_cil_ptr *) malloc(sizeof(struct sledovany_cil_ptr));
	ptr->dalsi = Seznam->Prvni;

	ptr->data.protokol = protokol;
	ptr->data.ID = NULL; // ID neni zadano
	ptr->data.IPv4 = IP;
	for(int i = 0; i < 16; i++){
		ptr->data.IPv6.s6_addr[i] = 0;
	}
	ptr->data.maska = maska;
	Seznam->Prvni = ptr;
}

/*
 * do seznamu vlozi nove sledovani identifikovane podle ID uzivatele
 */
void vloz_sledovani_ID(struct seznam_sledovani *Seznam, char *ID, int protokol)
{
	struct sledovany_cil_ptr *ptr = NULL;
	ptr = (struct sledovany_cil_ptr *) malloc(sizeof(struct sledovany_cil_ptr));
	ptr->dalsi = Seznam->Prvni;
	Seznam->Prvni = ptr;

	ptr->data.protokol = protokol;
	ptr->data.ID = ID;
	ptr->data.IPv4.s_addr = 0; // IP neni zadana
	for(int i = 0; i < 16; i++){
		ptr->data.IPv6.s6_addr[i] = 0;
	}
	ptr->data.maska = 0;
}

/*
 * ze seznamu odstrani odposlech zadane IP adresy
 */
void zrus_sledovani_IPv4(struct seznam_sledovani *Seznam, struct in_addr IP, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	// seznam je prazdny
	if(ptr == NULL)
	{
		return;
	}

	// kontrola jestli neni maza polozka prvni v seznamu
	if(ptr->data.IPv4.s_addr == IP.s_addr)
	{
		struct sledovany_cil_ptr *mazana_polozka = Seznam->Prvni;

		// upravime ukazatel na prvniho
		Seznam->Prvni = Seznam->Prvni->dalsi;
		free(mazana_polozka);

		return;
	}

	//projdeme dalsi polozky
	while(ptr != NULL)
	{
		// predchozi uz sme zkontorlovali, podivame se jestli je dale co kontrolovat
		if(ptr->dalsi == NULL)
		{
			return;
		}
		// obsah nasledujici polozky seznamu je mazane sezeni
		if(ptr->dalsi->data.IPv4.s_addr == IP.s_addr)
		{
			struct sledovany_cil_ptr *mazana_polozka = ptr->dalsi;
			// nastavime ukazatel aby preskocil mazanou polozku
			ptr->dalsi = ptr->dalsi->dalsi;
			free(mazana_polozka);
		}
		ptr = ptr->dalsi;
	}
}


void zrus_sledovani_IPv4_rozsah(struct seznam_sledovani *Seznam, struct in_addr IP, int maska, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	// seznam je prazdny
	if(ptr == NULL)
	{
		return;
	}

	// kontrola jestli neni maza polozka prvni v seznamu
	if(ptr->data.IPv4.s_addr == IP.s_addr && ptr->data.maska == maska)
	{
		struct sledovany_cil_ptr *mazana_polozka = Seznam->Prvni;

		// upravime ukazatel na prvniho
		Seznam->Prvni = Seznam->Prvni->dalsi;
		free(mazana_polozka);

		return;
	}

	//projdeme dalsi polozky
	while(ptr != NULL)
	{
		// predchozi uz sme zkontorlovali, podivame se jestli je dale co kontrolovat
		if(ptr->dalsi == NULL)
		{
			return;
		}
		// obsah nasledujici polozky seznamu je mazane sezeni
		if(ptr->dalsi->data.IPv4.s_addr == IP.s_addr && ptr->dalsi->data.maska == maska)
		{
			struct sledovany_cil_ptr *mazana_polozka = ptr->dalsi;
			// nastavime ukazatel aby preskocil mazanou polozku
			ptr->dalsi = ptr->dalsi->dalsi;
			free(mazana_polozka);
		}
		ptr = ptr->dalsi;
	}
}

/*
 * ze seznamu odstrani odposlech zadaneho ID uzivatele
 */
void zrus_sledovani_ID(struct seznam_sledovani *Seznam, char *ID, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	// seznam je prazdny
	if(ptr == NULL)
	{
		return;
	}

	if(ptr->data.ID != NULL)
	{
		// kontrola jestli neni maza polozka prvni v seznamu
		if(porovnani_retezce(ptr->data.ID, ID))
		{
			struct sledovany_cil_ptr *mazana_polozka = Seznam->Prvni;
			// upravime ukazatel na prvniho
			Seznam->Prvni = Seznam->Prvni->dalsi;
			free(mazana_polozka->data.ID);
			free(mazana_polozka);

			return;
		}
	}

	//projdeme dalsi polozky
	while(ptr != NULL)
	{
		// predchozi uz sme zkontorlovali, podivame se jestli je dale co kontrolovat
		if(ptr->dalsi == NULL)
		{
			return;
		}
		if(ptr->dalsi->data.ID != NULL)
		{
			// obsah nasledujici polozky seznamu je mazane sezeni
			if(porovnani_retezce(ptr->dalsi->data.ID, ID))
			{
				struct sledovany_cil_ptr *mazana_polozka = ptr->dalsi;
				// nastavime ukazatel aby preskocil mazanou polozku
				ptr->dalsi = ptr->dalsi->dalsi;
				free(mazana_polozka->data.ID);
				free(mazana_polozka);
			}
		}
		ptr = ptr->dalsi;
	}
}

/*
 * vrati 1 (true) pokud je IP adresa mezi sledovanymi jinak vraci 0 (false)
 */
int sledovani_IPv4(struct seznam_sledovani *Seznam, struct in_addr IP, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	if(ptr == NULL)
	{
		return 0;
	}

	while(ptr != NULL)
	{

		if(ptr->data.maska == 0) // sledovani pouze presne IP
		{
			if(ptr->data.IPv4.s_addr == IP.s_addr && ptr->data.protokol == protokol)
			{
				return 1;
			}
		}
		else
		{
			struct in_addr pomocna_IP; 
			pomocna_IP.s_addr = ntohl(IP.s_addr);
			pomocna_IP.s_addr = pomocna_IP.s_addr >> (32-ptr->data.maska);
			pomocna_IP.s_addr = pomocna_IP.s_addr << (32-ptr->data.maska);
			pomocna_IP.s_addr = htonl(pomocna_IP.s_addr);

			if(ptr->data.IPv4.s_addr == pomocna_IP.s_addr && ptr->data.protokol == protokol)
			{
				return 1;
			}
		}

		ptr = ptr->dalsi;
	}

	return 0;
}

/*
 * vrati 1 (true) pokud je IP adresa a odpovidajici maska mezi sledovanymi jinak vraci 0 (false)
 */
int sledovani_IPv4_rozsah(struct seznam_sledovani *Seznam, struct in_addr IP, int maska, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	if(ptr == NULL)
	{
		return 0;
	}

	while(ptr != NULL)
	{

		if(ptr->data.IPv4.s_addr == IP.s_addr && ptr->data.protokol == protokol && ptr->data.maska == maska)
		{
			return 1;
		}
		ptr = ptr->dalsi;
	}

	return 0;
}


/*
 * vrati 1 (true) pokud je ID mezi sledovanymi jinak vraci 0 (false)
 */
int sledovani_ID(struct seznam_sledovani *Seznam, char *ID, int protokol)
{
	struct sledovany_cil_ptr *ptr = Seznam->Prvni;

	if(ptr == NULL)
	{
		return 0;
	}

	while(ptr != NULL)
	{
		if(ID != NULL)
		{
			if(porovnani_retezce(ptr->data.ID, ID) && ptr->data.protokol == protokol)
			{
				return 1;
			}
		}
		ptr = ptr->dalsi;
	}

	return 0;
}

/*
 * Projdou se vsechna sezeni a deaktivuji se rusene odposlechy podle zadane IP
 */
void deaktivuj_sezeni_IPv4(struct seznam_sledovani *Seznam, struct in_addr IP, int maska, int protokol, struct seznam_sezeni *Sezeni)
{
	struct sezeni_ptr *pomocny = Sezeni->Prvni;
	while(pomocny != NULL)
	{
		if(maska == 0)
		{
			if(pomocny->data.protokol == protokol && pomocny->data.ipv4_adresa.s_addr == IP.s_addr)
			{
				char *pomocne_ID;
				int smazat = 0;
				if(protokol == XMPP)
				{
					smazat = 1;
					pomocne_ID = bareJID(pomocny->data.ID_uzivatele);
				
					if(pomocne_ID == NULL)
					{
						pomocne_ID = pomocny->data.ID_uzivatele;
						smazat = 0;
					}
				}
				else
				{
					pomocne_ID = pomocny->data.ID_uzivatele;
				}
				// smazat pouze pokud neni sledovano jeste na zaklade ID
				if(!sledovani_ID(Seznam, pomocne_ID, protokol))
				{
					pomocny->data.aktivni = 0;
					pomocny->data.cil_identifikovan = 0;
				}

				if(smazat)
				{
					free(pomocne_ID);
				}
			}
		}
		else
		{
			// pokud rusime adresy zadane rozsahem, ale zaroven je zadano sledovani samotne ip, pro tuto ip se sezeni nemaze
			if(!sledovani_IPv4(Seznam, IP, protokol))
			{
				struct in_addr pomocna_IP = pomocny->data.ipv4_adresa;
				pomocna_IP.s_addr = ntohl(pomocna_IP.s_addr);
				pomocna_IP.s_addr = pomocna_IP.s_addr >> (32-maska);
				pomocna_IP.s_addr = pomocna_IP.s_addr << (32-maska);
				pomocna_IP.s_addr = htonl(pomocna_IP.s_addr);

				if(pomocny->data.protokol == protokol && pomocna_IP.s_addr == IP.s_addr)
				{
					char *pomocne_ID;
					int smazat = 0;
					if(protokol == XMPP)
					{
						smazat = 1;
						pomocne_ID = bareJID(pomocny->data.ID_uzivatele);
				
						if(pomocne_ID == NULL)
						{
							pomocne_ID = pomocny->data.ID_uzivatele;
							smazat = 0;
						}
					}
					else
					{
						pomocne_ID = pomocny->data.ID_uzivatele;
					}
					// smazat pouze pokud neni sledovano jeste na zaklade ID
					if(!sledovani_ID(Seznam, pomocne_ID, protokol))
					{
						pomocny->data.aktivni = 0;
						pomocny->data.cil_identifikovan = 0;
					}

					if(smazat)
					{
						free(pomocne_ID);
					}
				}
			}
		}
		pomocny = pomocny->dalsi;
	}
}

/*
 * Projdou se vsechna sezeni a deaktivuji se rusene odposlechy podle zadaneho ID
 */
void deaktivuj_sezeni_ID(struct seznam_sledovani *Seznam, char *ID, int protokol, struct seznam_sezeni *Sezeni)
{
	struct sezeni_ptr *pomocny = Sezeni->Prvni;
	while(pomocny != NULL)
	{
		int smazat = 1;
		char *pomocne_ID = bareJID(pomocny->data.ID_uzivatele);
		if(pomocne_ID == NULL)
		{
			pomocne_ID = pomocny->data.ID_uzivatele;
			smazat = 0;
		}

		if(pomocny->data.protokol == protokol && porovnani_retezce(ID, pomocne_ID))
		{
			// smazat pouze pokud neni jeste sledovano na zaklade IP
			if(!sledovani_IPv4(Seznam, pomocny->data.ipv4_adresa, protokol))
			{
				pomocny->data.aktivni = 0;
				pomocny->data.cil_identifikovan = 0;
			}
		}

		if(smazat)
		{
			free(pomocne_ID);
		}

		pomocny = pomocny->dalsi;
	}
}


/******************************* Sprvava sezeni *******************************/

/*
 * inicializuje seznam sezeni pro pozdejsi alokace novych polozek
 */
void inicializace_seznamu_sezeni(struct seznam_sezeni *Seznam)
{
	Seznam->Aktivni = NULL;
	Seznam->Prvni = NULL;
}

/*
 * smazani seznamu sezeni
 */
void smazani_seznamu_sezeni(struct seznam_sezeni *Seznam)
{
	struct sezeni_ptr *ptr = Seznam->Prvni;

	while(ptr != NULL)
	{
		Seznam->Prvni = ptr->dalsi;
		smazani_sezeni(&(ptr->data));
		free(ptr);
		ptr = Seznam->Prvni;
	}
}

/*
 * do seznamu sezeni vlozi novou polozku na prvni misto
 */
void vloz_prvni_sezeni(struct seznam_sezeni *Seznam)
{
	struct sezeni_ptr *ptr = NULL;
	ptr = (struct sezeni_ptr *) malloc(sizeof(struct sezeni_ptr));
	ptr->dalsi = Seznam->Prvni;
	Seznam->Prvni = ptr;

	inicializace_sezeni(&(ptr->data));
}

/*
 * se seznamu sezeni odebere polozku ktera ma jako data rusene sezeni
 */
void odeber_sezeni(struct seznam_sezeni *Seznam, struct sezeni *Sezeni)
{
	struct sezeni_ptr *ptr = Seznam->Prvni;

	// kontrola jestli neni maza polozka prvni v seznamu
	if(&(ptr->data) == Sezeni)
	{
		struct sezeni_ptr *mazana_polozka = Seznam->Prvni;

		// upravime ukazatel na prvniho
		Seznam->Prvni = Seznam->Prvni->dalsi;

		// smazani sezeni
		smazani_sezeni(&(mazana_polozka->data));
		free(mazana_polozka);
		return;
	}

	//projdeme dalsi polozky
	while(ptr != NULL)
	{
		// predchozi uz sme zkontorlovali, podivame se jestli je dale co kontrolovat
		if(ptr->dalsi == NULL)
		{
			return;
		}
		// obsah nasledujici polozky seznamu je mazane sezeni
		if(&(ptr->dalsi->data) == Sezeni)
		{
			struct sezeni_ptr *mazana_polozka = ptr->dalsi;
			// nastavime ukazatel aby preskocil mazanou polozku
			ptr->dalsi = ptr->dalsi->dalsi;

			// smazani sezeni
			smazani_sezeni(&(mazana_polozka->data));
			free(mazana_polozka);

			return;
		}
		ptr = ptr->dalsi;
	}
}

/*
 * inicializace struktury noveho sezeni
 */
void inicializace_sezeni(struct sezeni *Sezeni)
{
	Sezeni->aktivni = 0;
	Sezeni->cil_identifikovan = 0;
	Sezeni->cin = 0; // je prideleno pozdeji
	Sezeni->transportni_protokol = 0; 
	Sezeni->typ_ip = 0;
	Sezeni->ipv4_adresa.s_addr = 0;
	Sezeni->ipv4_adresa_2.s_addr = 0;
	for(int i = 0; i < 16; i++){
		Sezeni->ipv6_adresa.s6_addr[i] = 0;
		Sezeni->ipv6_adresa_2.s6_addr[i] = 0;
	}
	Sezeni->port = 0;
	Sezeni->port_2 = 0;
	Sezeni->protokol = NEZNAME;
	Sezeni->ID_uzivatele = NULL;
	Sezeni->aktualni_stav = 0; // nepripojen
	Sezeni->potreba_dokoncit_prichozi = NULL;
	Sezeni->ocekavat_dalsi_prichozi = 0;
	Sezeni->potreba_dokoncit_odchozi = NULL;
	Sezeni->ocekavat_dalsi_odchozi = 0;
	Sezeni->ukonceno = 0;
}

/*
 * vymaze alokovane polozky v seznamu sezeni
 */
void smazani_sezeni(struct sezeni *Sezeni)
{
	if(Sezeni->ID_uzivatele != NULL)
	{
		free(Sezeni->ID_uzivatele);
	}
	if(Sezeni->potreba_dokoncit_prichozi != NULL)
	{
		free(Sezeni->potreba_dokoncit_prichozi);
	}
	if(Sezeni->potreba_dokoncit_odchozi != NULL)
	{
		free(Sezeni->potreba_dokoncit_odchozi);
	}
}

/*
 * vrati ukazatel na polozku seznamu sezeni ktera odpovida zadane IP adrese a portu
 */
struct sezeni *zapocate_sezeni_IPv4(struct seznam_sezeni *Seznam, int port, struct in_addr ip)
{
	struct sezeni_ptr *pomocny = Seznam->Prvni;
	while(pomocny != NULL)
	{

		if(pomocny->data.port == port && pomocny->data.ipv4_adresa.s_addr == ip.s_addr)
		{
			return &(pomocny->data);
		}
		else
		{
			pomocny = pomocny->dalsi;
		}
	}
	// zadne odpovidajici sezeni neni v seznamu
	return NULL;
}

/*
 * vrati ukazatel na sezeni podle zadane IPv4 adresy
 */
struct sezeni *sezeni_podle_IPv4(struct seznam_sezeni *Seznam, struct in_addr IP)
{
	struct sezeni_ptr *pomocny = Seznam->Prvni;
	while(pomocny != NULL)
	{

		if(pomocny->data.ipv4_adresa.s_addr == IP.s_addr)
		{
			return &(pomocny->data);
		}
		else
		{
			pomocny = pomocny->dalsi;
		}
	}
	// zadne odpovidajici sezeni neni v seznamu
	return NULL;
}

/*
 * vrati ukazatel na sezeni podle zadaneho ID uzivatele
 */
struct sezeni *sezeni_podle_ID(struct seznam_sezeni *Seznam, char *ID)
{
	struct sezeni_ptr *pomocny = Seznam->Prvni;
	while(pomocny != NULL)
	{

		if(porovnani_retezce(pomocny->data.ID_uzivatele, ID))
		{
			return &(pomocny->data);
		}
		else
		{
			pomocny = pomocny->dalsi;
		}
	}
	// zadne odpovidajici sezeni neni v seznamu
	return NULL;
}

