/*
    Copyright (C) 2012  Stanislav Bárta

    This file is part of Bachelor's thesis: Creating Metadata during 
    Interception of Instant Messaging Communication.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * Soubor:  main.c
 * Datum:   16.05.2012
 * Autor:   Stanislav Bárta, xbarta29@stud.fit.vutbr.cz
 * Projekt: BP - tvorba metadat pri odposlechu komunikace v realnem case
 * Popis:   soubor obsahujici funkci main aplikace BP
 *          inicializace funkci knihovny pcap a smycka zachytavani paketu
 *          zachyceni signalu ukonceni aplikace a noveho prochoziho pozadavku z AF
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <pcap.h>
#include <getopt.h>

#include "detekce_protokolu.h"
#include "xmpp.h"
#include "irc.h"
#include "oscar.h"
#include "iri.h"
#include "sprava.h"
#include "my_string.h"

// hodnota jak velky paket maximalne zpracovat -> urceno podle manualu pcap
#define BUFFER 65535

// handler rozhrani pro odposlouchavani
// globalne kvuli moznosti uzavrit handler po ukonceni programu pres sigint
pcap_t *handler_rozhrani;

// pokud jsou pouzivany pouze zname porty
struct bpf_program fp;
char filter_exp[] = "port 5222 or portrange 6665-6669 or portrange 5190-5193";


struct seznam_sezeni Seznam_sezeni;
struct seznam_sledovani Seznam_sledovani;
char msg[1500]; // pro prijem zpravy z AF

/*
 * funkce zavolana po zachyceni signalu sigint (ctrl+c)
 * uzavreni otevrenych spojeni a uvolneni alokovane pameti
 */
void sigint_handler(int sig)
{
	uzavri_spojeni_pro_iri();
	uzavri_spojeni_pro_spravu_sledovani();
	smazani_seznamu_sezeni(&Seznam_sezeni);
	smazani_seznamu_sledovani(&Seznam_sledovani);
	pcap_close(handler_rozhrani);

	exit(EXIT_SUCCESS);
}

/*
 * funkce volana po zachyceni signalu sigio
 * prijme se zaslana zprava a zpracujeme ji
 */
void signal_handler_IO(int status, siginfo_t *ioinfo, void * context)
{
	bzero(msg, sizeof(msg)); // vynulovani zpray pro prijem
	recv(socket_sprava, msg, sizeof(msg), 0); // prijmuti zpravy

	zpracuj_novy_pozadavek_na_sledovani(msg, &Seznam_sledovani, &Seznam_sezeni);
}

/*
 * vycet typu chyb pro identifikovani chyby a vypsani upozorneni
 */
enum typy_chyb
{
	OK,
	CHYBNY_POCET_PARAMETRU,
	CHYBA_V_PARAMETRECH,
	CHYBEJICI_ROZHRANI,
	CHYBA_KOMPILACE_FILTRU,
	CHYBA_APLIKACE_FILTRU
};

/*
 * pole obsahujici chybove zpravy pro predtim definovane kody
 */
const char *CHYBOVE_ZPRAVY[] = {
"Vse OK\n",
"Byl zadan chybny pocet parametru.\n",
"Chybne zadane parametry. Spustenim aplikace s parametrem --help nebo -h ziskate\nseznam parametru nutnych pro spusteni aplikace\n",
"Zadane rozhrani neni dostupne.\n",
"Chyba pri kompilaci filtru pro pcap.\n",
"Chyba pri aplikaci filtru pro pcap.\n"
};

const char *HELP = {
"Aplikaci je treba spustit s parametry:\n"
"\t--af_ip <ip_adresa_administration_function>\n"
"\t--af_port <port_pro_komunikaci_s_administration_function>\n"
"\t--mf_ip <ip_adresa_mediation_function>\n"
"\t--mf_port <port_pro_komunikaci_s_mediation_function>\n"
"\t--interface <nazev_rozhrani_kde_provadet_odposlech>\n"
};

typedef struct
{
	char *rozhrani; // rozhrani na kterem budeme odposlouchavat
	char *AF_IP;
	char *AF_port;
	char *MF_IP;
	char *MF_port;
	int pouze_zname_porty;
	int help;
} Parametry;

/*
 * funkce zkontroluje zadani parametru a nastavi promennou rozhrani na 
 * hodnotu ziskanou z prikazove radky.
 */
int zpracuj_parametry(int argc, char **argv, Parametry *ziskane_parametry)
{
	static struct option long_options[] =
	{
		{"mf_ip",                  required_argument, 0, 0},
		{"mf_port",                required_argument, 0, 0},
		{"af_ip",                  required_argument, 0, 0},
		{"af_port",                required_argument, 0, 0},
		{"interface",              required_argument, 0, 'i'},
		{"pouze_specificke_porty", no_argument, 0, 'p'},
		{"help",                   no_argument, 0, 'h'},
		{0, 0, 0, 0}
	};

	int option_index = 0;
	int c;

	while((c = getopt_long (argc, argv, "i:ph", long_options, &option_index)) != -1)
	{
		switch (c)
		{
			case 0:

				if(porovnani_retezce((char *) (long_options[option_index].name), "mf_ip"))
				{
					ziskane_parametry->MF_IP = optarg;
				}
				else if(porovnani_retezce((char *) (long_options[option_index].name), "mf_port"))
				{
					ziskane_parametry->MF_port = optarg;
				}
				else if(porovnani_retezce((char *) (long_options[option_index].name), "af_ip"))
				{
					ziskane_parametry->AF_IP = optarg;
				}
				else if(porovnani_retezce((char *) (long_options[option_index].name), "af_port"))
				{
					ziskane_parametry->AF_port = optarg;
				}
				break;

			case 'i':
					ziskane_parametry->rozhrani = optarg;
				break;

			case 'p':
					ziskane_parametry->pouze_zname_porty = 1;
				break;

			case 'h':
					ziskane_parametry->help = 1;
				break;

			case '?':
				return CHYBA_V_PARAMETRECH;
				break;

			default:
				return CHYBA_V_PARAMETRECH;
		}
	}

	if(optind < argc)
	{
		return CHYBA_V_PARAMETRECH;
	}

	if((ziskane_parametry->rozhrani == NULL || ziskane_parametry->AF_port == NULL || ziskane_parametry->AF_IP == NULL || ziskane_parametry->MF_port == NULL || ziskane_parametry->MF_IP == NULL) && !ziskane_parametry->help)
	{
		return CHYBA_V_PARAMETRECH;
	}

	return OK;
}


/*
 * callback funkce volana po zachyceni packetu
 */
void ziskat_packet(u_char *args, const struct pcap_pkthdr *header, const u_char *packet)
{
	struct sezeni *aktualni_sezeni = NULL;

	// ziskani dat z packetu
	struct data_packetu obsah = ziskej_obsah(header, packet);

	// kontrola jestli byla zachycena zprava nektereho ze sledovanych protokolu
	if(obsah.typ != NEZNAME)
	{
		// podle IP adresy a potru na strane klienta je nastaveno aktualni sezeni, pripadne je vytvoreno nove
		if(obsah.smer == odchozi)
		{
			if(obsah.verze_ip == IPverze4)
			{
				aktualni_sezeni = zapocate_sezeni_IPv4(&Seznam_sezeni, obsah.port_zdroje, obsah.ip_zdroje);
				// kontrola jestli nebyl zadan pozadavek na odpsolech az po vytvoreni sezeni
				if(aktualni_sezeni != NULL)
				{
					if(sledovani_IPv4(&Seznam_sledovani, aktualni_sezeni->ipv4_adresa, obsah.typ))
					{
						aktualni_sezeni->cil_identifikovan = 1;
					}
				}
			}

			// sezeni jeste nebylo vytvoreno
			if(aktualni_sezeni == NULL)
			{
				// vytvori se nove sezeni a nastavi se mu odpovidajici udaje
				vloz_prvni_sezeni(&Seznam_sezeni);
				Seznam_sezeni.Prvni->data.transportni_protokol = obsah.transportni_protokol;
				if(obsah.verze_ip == IPverze4)
				{
					Seznam_sezeni.Prvni->data.typ_ip = IPverze4;
					Seznam_sezeni.Prvni->data.ipv4_adresa = obsah.ip_zdroje;
					Seznam_sezeni.Prvni->data.ipv4_adresa_2 = obsah.ip_cile;
				}
				Seznam_sezeni.Prvni->data.port_2 = obsah.port_cile;
				Seznam_sezeni.Prvni->data.port = obsah.port_zdroje;
				aktualni_sezeni = &(Seznam_sezeni.Prvni->data);

				if(obsah.typ == XMPP)
				{
					aktualni_sezeni->protokol = XMPP;
				}
				else if(obsah.typ == IRC)
				{
					aktualni_sezeni->protokol = IRC;
				}
				else if(obsah.typ == OSCAR)
				{
					aktualni_sezeni->protokol = OSCAR;
				}

				if(obsah.verze_ip == IPverze4)
				{
					if(sledovani_IPv4(&Seznam_sledovani, aktualni_sezeni->ipv4_adresa, obsah.typ))
					{
						aktualni_sezeni->cil_identifikovan = 1;
					}
				}
			}
		}
		else if(obsah.smer == prichozi)
		{
			if(obsah.verze_ip == IPverze4)
			{
				aktualni_sezeni = zapocate_sezeni_IPv4(&Seznam_sezeni, obsah.port_cile, obsah.ip_cile);
				// kontrola jestli nebyl zadan pozadavek na odpsolech az po vytvoreni sezeni
				if(aktualni_sezeni != NULL)
				{
					if(sledovani_IPv4(&Seznam_sledovani, aktualni_sezeni->ipv4_adresa, obsah.typ))
					{
						aktualni_sezeni->cil_identifikovan = 1;
					}
				}
			}

			// sezeni jeste nebylo vytvoreno
			if(aktualni_sezeni == NULL)
			{
				// vytvori se nove sezeni a nastavi se mu odpovidajici udaje
				vloz_prvni_sezeni(&Seznam_sezeni);
				Seznam_sezeni.Prvni->data.transportni_protokol = obsah.transportni_protokol;
				if(obsah.verze_ip == IPverze4)
				{
					Seznam_sezeni.Prvni->data.typ_ip = IPverze4;
					Seznam_sezeni.Prvni->data.ipv4_adresa = obsah.ip_cile;
					Seznam_sezeni.Prvni->data.ipv4_adresa_2 = obsah.ip_zdroje;
				}
				Seznam_sezeni.Prvni->data.port_2 = obsah.port_zdroje;
				Seznam_sezeni.Prvni->data.port = obsah.port_cile;
				aktualni_sezeni = &(Seznam_sezeni.Prvni->data);

				if(obsah.typ == XMPP)
				{
					aktualni_sezeni->protokol = XMPP;
				}
				else if(obsah.typ == IRC)
				{
					aktualni_sezeni->protokol = IRC;
				}
				else if(obsah.typ == OSCAR)
				{
					aktualni_sezeni->protokol = OSCAR;
				}

				if(obsah.verze_ip == IPverze4)
				{
					if(sledovani_IPv4(&Seznam_sledovani, aktualni_sezeni->ipv4_adresa, obsah.typ))
					{
						aktualni_sezeni->cil_identifikovan = 1;
					}
				}
			}
		}
	}

	// podle typu zjisteneho protokolu zpracujeme obsah
	if(obsah.typ == XMPP)
	{
		zpracuj_xmpp(&obsah, aktualni_sezeni, &Seznam_sledovani);
		free(obsah.data);
		if(aktualni_sezeni->ukonceno)
		{
			odeber_sezeni(&Seznam_sezeni, aktualni_sezeni);
		}
	}
	else if(obsah.typ == IRC)
	{
		zpracuj_irc(&obsah, aktualni_sezeni, &Seznam_sledovani);
		free(obsah.data);
		if(aktualni_sezeni->ukonceno)
		{
			odeber_sezeni(&Seznam_sezeni, aktualni_sezeni);
		}
	}
	else if(obsah.typ == OSCAR)
	{
		zpracuj_oscar(&obsah, aktualni_sezeni, &Seznam_sledovani);
		if(aktualni_sezeni->ukonceno)
		{
			odeber_sezeni(&Seznam_sezeni, aktualni_sezeni);
		}
	}
}

/*
 * hlavni funkce main
 */
int main(int argc, char **argv)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	int chybovy_kod;
	Parametry ziskane_parametry = {NULL, NULL, NULL, NULL, NULL, 0, 0};

	// zpracujeme parametry prikazove radky
	if((chybovy_kod = zpracuj_parametry(argc, argv, &ziskane_parametry)) != OK)
	{
		fprintf(stderr, "%s" ,CHYBOVE_ZPRAVY[chybovy_kod]);
		return EXIT_FAILURE;
	}

	if(ziskane_parametry.help)
	{
		printf("%s", HELP);
		return EXIT_SUCCESS;
	}

	// otevreni spojeni pro zasilani vytvorenych iri zprav
	if(otevri_spojeni_pro_iri(ziskane_parametry.MF_IP, ziskane_parametry.MF_port))
	{
		return EXIT_FAILURE;
	}

	// pripojeni k AF aby bylo mozno zadavat pozadavky na odposlechy
	if(otevri_spojeni_pro_spravu_sledovani(ziskane_parametry.AF_IP, ziskane_parametry.AF_port))
	{
		return EXIT_FAILURE;
	}

	// nastaveni handler funkce pro signal ukonceni pomoci ctrl+c
	signal(SIGINT, sigint_handler);

	// nastaveni akce pri signalu sigio
	struct sigaction saio;

	sigemptyset(&saio.sa_mask);
	saio.sa_flags = SA_SIGINFO;
//	saio.sa_restorer = NULL;
	saio.sa_sigaction = signal_handler_IO;

	sigaction(SIGIO,&saio,NULL);

	// proces bude prijimat signal sigio
	fcntl(socket_sprava, F_SETOWN, getpid());
	// asynchroni prijem zprav z AF
	fcntl(socket_sprava, F_SETFL, FASYNC);

	// inicializace seznamu a pomocnych globalnich promennych
	inicializace_seznamu_sezeni(&Seznam_sezeni);
	inicializace_seznamu_sledovani(&Seznam_sledovani);

	// otevreni rozhrani v promiskuitnim modu pro zachytavani paketu
	handler_rozhrani = pcap_open_live(ziskane_parametry.rozhrani,BUFFER,1,1000,errbuf);

	if(handler_rozhrani == NULL)
	{
		fprintf(stderr, "%s" ,CHYBOVE_ZPRAVY[CHYBEJICI_ROZHRANI]);
		return EXIT_FAILURE;
	}

	if(ziskane_parametry.pouze_zname_porty)
	{
		if(pcap_compile(handler_rozhrani, &fp, filter_exp, 0, 0) == -1)
		{
			fprintf(stderr, "%s" ,CHYBOVE_ZPRAVY[CHYBA_KOMPILACE_FILTRU]);
			return EXIT_FAILURE;
		}

		if(pcap_setfilter(handler_rozhrani, &fp) == -1)
		{
			fprintf(stderr, "%s" ,CHYBOVE_ZPRAVY[CHYBA_APLIKACE_FILTRU]);
			return EXIT_FAILURE;
		}
	}
	pcap_freecode(&fp);
	// smycka zachytavani packetu
	pcap_loop(handler_rozhrani, -1, ziskat_packet, NULL);

	return EXIT_SUCCESS;
}

