/** @file net.cpp
* zdrojovy soubor obsahujici implementaci naslouchani na rozhranich a odesilani ICMPv6 zprav
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <list>
#include <pcap.h>
#include <libnet.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip6.h>
#include <netinet/tcp.h>

#include <signal.h>

#include <string.h> /* pro strncpy */
#include <sys/types.h>
#include <sys/socket.h>

#include "main.h"
#include "config.h"
#include "net.h"
#include "errors.h"
#include "tcp.h"

#define PCAP_SNAPLEN 1538

char pcap_errbuf[PCAP_ERRBUF_SIZE];

/**
 * Soucet dvou 16bitovych cisel v aritmetice jecnickoveho doplnku
 * (vyuzito pro vypocet ICMPv6 checksum)
 * @param checksum prvni cislo a zaroven vysledek
 * @param num druhe cislo
 */
void calculateChk16(uint16_t& checksum, uint16_t& num) {
  union {
    uint16_t sum16[2];
    uint32_t sum32;
  } chkCalc;

  chkCalc.sum32 = 0;
  chkCalc.sum32 = checksum + num;

  uint16_t result = chkCalc.sum16[0] + chkCalc.sum16[1];
  checksum = result;
}


/**
 * Vytvoreni a odeslani prislusne ICMPv6 zpravy
 * @param dest_addr cilova adresa
 * @param ifName nazev rozhrani
 * @param messageType typ zpravy
 */
void sendIcmpMessage(in6_addr dest_addr, string ifName, int messageType, uint8_t hopLimit, uint16_t echo_id, uint16_t echo_seq, TConfig &cfg) {
  pthread_mutex_lock(&cfg.libnet_mutex);

  libnet_t *l;                     // kontext libnetu
  char errbuf[LIBNET_ERRBUF_SIZE]; // chybovy buffer

  if (ifName == "") {
    l = libnet_init (LIBNET_RAW6, NULL, errbuf);
  } else {
    char *c_ifname = const_cast<char*>(ifName.c_str());
    l = libnet_init (LIBNET_RAW6, c_ifname, errbuf);
  }
  if (l == NULL) {
    printError(ECONTEXT);
    return;
  }

  libnet_in6_addr libnetSourceAddress = libnet_get_ipaddr6(l);  // ziskame adresu rozhrani

  // ziskame ukazatel na zdrojovou adresu pretypovany na in6_addr
  in6_addr *sourceAddress = (struct in6_addr *)(&libnetSourceAddress);

  // vytvoreni celeho datagramu
  t_Ip6Icmp6 packet;

  /* ************************* */
  /*  Vytvoreni IPv6 hlavicky  */
  /* ************************* */
  packet.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_flow = 0;
  packet.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_plen = htons(sizeof(struct s_icmpv6Header) + sizeof(union u_icmpv6Body));
  packet.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_nxt = ICMPV6_NEXT_HEADER;
  packet.ipv6_header.ip6_ctlun.ip6_un1.ip6_un1_hlim = hopLimit;
  packet.ipv6_header.ip6_ctlun.ip6_un2_vfc = 0x60;
  packet.ipv6_header.ip6_src = *sourceAddress;
  packet.ipv6_header.ip6_dst = dest_addr;

  /* *************************** */
  /*  Vytvoreni ICMPv6 hlavicky  */
  /* *************************** */
  if (messageType == ICMPV6_UNREACH) {
    packet.icmpv6_header.type = ICMPV6_UNREACH_TYPE;
    packet.icmpv6_header.code = ICMPV6_UNREACH_CODE;
    packet.icmpv6_body.s_generic.body = 0;
  } else if (messageType == ICMPV6_PARAMETER) {
    packet.icmpv6_header.type = ICMPV6_PARAMETER_TYPE;
    packet.icmpv6_header.code = ICMPV6_PARAMETER_CODE;
    packet.icmpv6_body.s_generic.body = 0;
  } else if (messageType == ICMPV6_ECHO) {
    packet.icmpv6_header.type = ICMPV6_ECHO_TYPE;
    packet.icmpv6_header.code = ICMPV6_ECHO_CODE;
    packet.icmpv6_body.s_echo.identifier = htons(echo_id);
    packet.icmpv6_body.s_echo.sequence = htons(echo_seq);
  }
  packet.icmpv6_header.checksum = 0;

  for (unsigned int i = 0; i < 56; i++)
    packet.icmpv6_body.s_echo.data[i] = 0;


  // ============================ Vypocet ICMPv6 checksum ============================
  uint32_t sum = 0;
	uint16_t sum16;

	// -- IPv6 pseudo header --
  sum += uint16_checksum((uint16_t *)(void *)&packet.ipv6_header.ip6_src, 16);
	sum += uint16_checksum((uint16_t *)(void *)&packet.ipv6_header.ip6_dst, 16);
  uint64_t icmpv6_len = htonl(sizeof(struct s_icmpv6Header) + sizeof(union u_icmpv6Body));
  sum += uint16_checksum((uint16_t *)(void *)&icmpv6_len, 4);
  sum += ICMPV6_NEXT_HEADER;

  // -- ICMPv6 zprava --
	sum += uint16_checksum((uint16_t *)(void *)&packet.icmpv6_header, sizeof(struct s_icmpv6Header) + sizeof(union u_icmpv6Body));

  // vypocet jednickoveho doplnku a prevod na 16bitovou reprezentaci
	sum = (sum & 0xFFFF) + (sum >> 16);
	sum = (sum & 0xFFFF) + (sum >> 16);
	sum16 = htons(sum);
	sum16 = ~sum16;
	if (sum16 == 0) {
		sum16 = 0xFFFF;
	}
	packet.icmpv6_header.checksum = sum16;
  // =================================================================================

  int packetLen = sizeof(packet);                           // ziskani delky paketu
  libnet_write_raw_ipv6(l, (u_int8_t*)&packet, packetLen);  // odeslani paketu
  libnet_destroy (l);                                       // zruseni kontextu libnetu

  pthread_mutex_unlock(&cfg.libnet_mutex);
  return;
}


/**
 * Uzavreni vsech rozhrani ze seznamu
 * @param interface_list seznam rozhrani
 */
void closeInterfaces(TConfig cfg) {
  list<TInterface>::iterator i;

  if (!cfg.interface_list.empty()) {
    for(i=cfg.interface_list.begin(); i != cfg.interface_list.end(); ++i) {
      pcap_close((pcap_t*)i->descr);
    }
  }
}


/**
 * Otevreni vsech rozhrani se seznamu, aby na nich bylo mozne naslouchat
 * @param interface_list seznam rozhrani
 * @param read_set mnozina file descriptoru prirazenych rozhranim, funkce ji naplni
 * @param max_fd nejvyssi prideleny file_descriptor - bude vyuzit pri volani select()
 * @return uspech
 */
bool openInterfaces(list<TInterface>& interface_list, fd_set& read_set, int& max_fd) {
  FD_ZERO(&read_set); // mnozinu fild deskriptoru rozhrani nejprve vyprazdnime
  list<TInterface>::iterator i;
  pcap_t *descr;
  int pcap_fd = -1;   // file descriptor aktualne oteviraneho rozhrani
  for(i=interface_list.begin(); i != interface_list.end(); ++i) {
    descr = pcap_open_live(i->name.c_str(), PCAP_SNAPLEN, PROMISC, 0, pcap_errbuf);
    if (descr == NULL) {
      cerr << "Chyba - rozhrani " << i->name << " neni k dipozici!" << endl;
      return false;
    } else {
      pcap_fd = pcap_fileno(descr);     // ziskame file descriptor rozhrani
      if (pcap_fd > max_fd) {           // overeni, zda neni dany deskriptor nejvyssi
        max_fd = pcap_fd;
      }
    }
    FD_SET(pcap_fd, &read_set);         // pridame deskriptor otevreneho rozhrani do mnoziny
    pcap_setnonblock(descr, 1, pcap_errbuf); // nastavime non-blocking rezim
    i->descr = (int*)descr;             // nastavime pcap descriptor
    i->fd = pcap_fd;                    // nastavime fd
    if (verbose()) {
      cout << "Rozhrani: " << i->name;
      if (i->type == IN_IF) {
        cout << " |<-- vstupni" << endl;
      } else {
        cout << " |--> vystupni" << endl;
      }
    }
  }
  return true;
}

/**
 * Zjisti zda jde o link local (nebo link local multicastovou) adresu
 * @param IPv6 adresa
 * @return je typu link local
 */
bool isLinkLocal(in6_addr addr) {
  if ((addr.s6_addr[0] == 0xff && addr.s6_addr[1] == 0x02)
    || (addr.s6_addr[0] == 0xfe && addr.s6_addr[1] == 0x08)) {
    return true;
  } else {
    return false;
  }
}


/**
 * Ziskani MAC adresy daneho rozhrani
 * @param nazev rozhrani
 * @return MAC adresa
*/
libnet_ether_addr getInterfaceMac(string ifName, TConfig &cfg) {
  pthread_mutex_lock(&cfg.libnet_mutex);
  libnet_t *l; // kontext libnetu
  char errbuf[LIBNET_ERRBUF_SIZE];
  libnet_ether_addr mac;
  char *cifName = const_cast<char*>(ifName.c_str());
  for (int octetNum=0; octetNum<=5; octetNum++) {
    mac.ether_addr_octet[octetNum] = 0xFF;
  }
  l = libnet_init (LIBNET_RAW6, cifName, errbuf);
  if (l == NULL) {
    printError(ECONTEXT);
    return mac;
  }
  mac = *(libnet_get_hwaddr(l));

  libnet_destroy(l);
  pthread_mutex_unlock(&cfg.libnet_mutex);

  return mac;
}


/**
 * Ziskani IPv6 adresy daneho rozhrani
 * @param nazev rozhrani
 * @return IPv6 adresa
*/
in6_addr getInterfaceIP(string ifName, TConfig &cfg) {
  pthread_mutex_lock(&cfg.libnet_mutex);
  libnet_t *l; // kontext libnetu
  char errbuf[LIBNET_ERRBUF_SIZE];
  in6_addr ip;
  char *cifName = const_cast<char*>(ifName.c_str());
  for (int octetNum=0; octetNum<=16; octetNum++) {
    ip.s6_addr[octetNum] = 0xFF;
  }
  l = libnet_init (LIBNET_RAW6, cifName, errbuf);
  if (l == NULL) {
    printError(ECONTEXT);
    return ip;
  }
  libnet_in6_addr ln_ip;
  ln_ip = libnet_get_ipaddr6(l);
  ip = *((in6_addr *)(&ln_ip));

  libnet_destroy(l);
  pthread_mutex_unlock(&cfg.libnet_mutex);

  return ip;
}


/**
 * Porovnani dvou MAC adres
 * @param prvni MAC adresa
 * @param druha MAC adresa
 * @return shoda
*/
bool compareMac(libnet_ether_addr& mac1, libnet_ether_addr& mac2) {
  bool match = true;
  for (int octetNum=0; octetNum<=5; octetNum++) {
    if (mac1.ether_addr_octet[octetNum] != mac2.ether_addr_octet[octetNum]) {
      match = false;
      break;
    }
  }
  return match;
}


/**
 * Prevod IPv6 adresy na retezec
 * @param IPv6 adresa
 * @return retezec
 */
string ipv6ToStr(in6_addr addr) {
  char caddr[INET6_ADDRSTRLEN];
  inet_ntop(AF_INET6, &addr, caddr, INET6_ADDRSTRLEN);
  string strAddr = caddr;
  return strAddr;
}
