/** @file config.cpp
* Nacteni konfigurace programu
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <fstream>
#include <sstream>
#include <list>
#include <regex.h>
#include <netinet/ip6.h>
#include <arpa/inet.h>

#include "config.h"
#include "errors.h"
#include "translator.h"

/**
 * Nacteni konfiguracniho souboru
 * @param konfigurace programu
 * @return chybovy kod
 */
int loadConfig(TConfig &cfg) {
  int errNum = OK;
  string line;
  ifstream config_stream (cfg.config_file.c_str(), ios::in);
  if (!config_stream.is_open())
  {
    return EOPENCONFIG;
  }
  const char *pattern_config = "^[[:blank:]]*([a-zA-Z12_]+)[[:blank:]]*=[[:blank:]]*(.+)$";
  const char *pattern_whitespace_only = "^[[:blank:]]*$";
  const char *pattern_comment = "^[[:blank:]]*#.*$";
  regex_t re_config;
  regex_t re_whitespace_only;
  regex_t re_comment;
  if (regcomp( &re_config, pattern_config, REG_EXTENDED) != 0
    || regcomp( &re_whitespace_only, pattern_whitespace_only, REG_EXTENDED) != 0
    || regcomp( &re_comment, pattern_comment, REG_EXTENDED) != 0) { // kompilace reg. vyrazu
    return EREGEX;
  }

  regmatch_t pmatch_config[3];

  // ------ radek konfiguracniho souboru ------
  while (!config_stream.eof())
  {
    getline (config_stream,line); // nacteme radek ze souboru
    if (regexec(&re_config, line.c_str(), 3, pmatch_config, 0) == 0) { // jde o konfiguracni zaznam
      string key = "";
      string value = "";
      key.append(line.c_str()+pmatch_config[1].rm_so, pmatch_config[1].rm_eo-pmatch_config[1].rm_so);
      value.append(line.c_str()+pmatch_config[2].rm_so, pmatch_config[2].rm_eo-pmatch_config[2].rm_so);

      errNum = examineConfigRecord(key, value, cfg); // zpracujeme konfiguracni zaznam
      if (errNum != OK) { // nastala chyba, koncime
        break;
      }

    } else if (regexec(&re_whitespace_only, line.c_str(), 0, NULL, 0) != 0
      && regexec(&re_comment, line.c_str(), 0, NULL, 0) != 0)
    {
      cerr << line << endl; // neznamy zaznam v konfiguracnim souboru
      return ECONFIG;
    }
  }
  // -------------- konec radku --------------

  regfree(&re_config);
  regfree(&re_whitespace_only);
  regfree(&re_comment);

  if (errNum != OK) {
    return errNum;
  }

  // Existuje nejake pravidlo pro preklad?
  if (cfg.rules_list.empty()) {
    return ENORULE;
  }

  // Overeni, zda existuje alespon 1 vstupni+vystupni rozhrani
  if (cfg.interface_list.empty()) {
    return ENOOUTIF;
  } else {
    list<TInterface>::iterator i;
    bool out_exists = false;

    for(i=cfg.interface_list.begin(); i != cfg.interface_list.end(); ++i) {
      if (i->type == OUT_IF) {
        out_exists = true;
      }
    }
    if (!out_exists) {
      return ENOOUTIF;
    }
  }
  return errNum;
}


/**
 * Zpracovani konfiguracniho zaznamu
 * @param klic
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int examineConfigRecord (string &key, string &value, TConfig &cfg) {
  int errNum = OK;
  if (key == "TRANSLATION_RULE") {
    errNum = addTranslateRule(value, cfg);
  } else if (key == "OUT_IF") {
    errNum = addOutIf(value, cfg);
  } else if (key == "SENDER_PREFIX") {
    errNum = setSenderPrefix(value, cfg);
  } else if (key == "SENDER_PORT") {
    errNum = setSenderPort(value, cfg);
  } else if (key == "RECEIVER_PREFIX") {
    errNum = setReceiverPrefix(value, cfg);
  } else if (key == "RECEIVER_PORT") {
    errNum = setReceiverPort(value, cfg);
  } else if (key == "NOISE_AMOUNT") {
    errNum = setNoiseAmount(value, cfg);
  } else if (key == "FAKE_MESSAGE1_FILE") {
    errNum = setFakeMsgFile(1, value, cfg);
  } else if (key == "FAKE_MESSAGE2_FILE") {
    errNum = setFakeMsgFile(2, value, cfg);
  } else if (key == "INTERVAL_NANOSEC") {
    errNum = setNanoInterval(value, cfg);
  } else {
    cerr << key << endl; // neznamy zaznam v konfiguracnim souboru
    errNum = ECONFIG;
  }
  return errNum;
}


/**
 * Pridani pravidla pro preklad
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int addTranslateRule(string &value, TConfig &cfg) {
  const char *pattern = "([a-z0-9]+) ([0-9a-fA-F:]+)/([0-9]+) ([0-9a-fA-F:]+)/([0-9]+)";

  regmatch_t pmatch[6];
  regex_t re;

  TRule rule;
  string inPrefix = "";
  string exPrefix = "";
  string inPrefixLength_str = ""; // pred prevodem na cislo
  string exPrefixLength_str = ""; // pred prevodem na cislo

  if(regcomp( &re, pattern, REG_EXTENDED) != 0) { // kompilace reg. vyrazu
    return EREGEX;
  }

  if (regexec( &re, value.c_str(), 6, pmatch, 0) == 0) { // zkusime nalezt shodu
    rule.inIf.append(value.c_str()+pmatch[1].rm_so, pmatch[1].rm_eo-pmatch[1].rm_so);
    inPrefix.append(value.c_str()+pmatch[2].rm_so, pmatch[2].rm_eo-pmatch[2].rm_so);
    inPrefixLength_str.append(value.c_str()+pmatch[3].rm_so, pmatch[3].rm_eo-pmatch[3].rm_so);
    exPrefix.append(value.c_str()+pmatch[4].rm_so, pmatch[4].rm_eo-pmatch[4].rm_so);
    exPrefixLength_str.append(value.c_str()+pmatch[5].rm_so, pmatch[5].rm_eo-pmatch[5].rm_so);

    // prevod stringove reprezentace adres na typ in6_addr
    if (inet_pton(AF_INET6, inPrefix.c_str(), &(rule.inPrefix)) != 1) {
      cerr << value << endl;
      return EBADPREFIX;
    }
    if (inet_pton(AF_INET6, exPrefix.c_str(), &(rule.exPrefix)) != 1) {
      cerr << value << endl;
      return EBADPREFIX;
    }

    // prevod delky interniho prefixu na cislo
    istringstream isIn(inPrefixLength_str);
    isIn >> rule.inPrefixLength;
    // prevod delky externiho prefixu na cislo
    istringstream isEx(exPrefixLength_str);
    isEx >> rule.exPrefixLength;
    cfg.rules_list.push_back(rule); // pridame dane pravidlo do seznamu

    if (rule.inPrefixLength > MAX_PREFIX_LEN) {
      cerr << value << endl;
      return EPREFIXLEN;
    }
    if (rule.exPrefixLength > MAX_PREFIX_LEN) {
      cerr << value << endl;
      return EPREFIXLEN;
    }

    // Zjistime, zda je rozhrani v seznamu, pokud ne, pridame jej
    list<TInterface>::iterator i;
    bool found = false;
    if (!cfg.interface_list.empty()) {
      for(i=cfg.interface_list.begin(); i != cfg.interface_list.end(); ++i) {
        if (i->name == rule.inIf) {
          found = true;
        }
      }
    }
    if (!found) {
      TInterface interface;
      interface.name = rule.inIf; // nastavime nazev rozhrani
      interface.type = IN_IF;     // jde o vnitrni rozhrani
      interface.fd = -1;          // file descriptor prozatim nastavime na -1
      cfg.interface_list.push_back(interface); // pridame dane zarizeni do seznamu
    }
  } else {
    cerr << value << endl;
    return ECONFIG;
  }
  regfree(&re);
  return OK;
}


/**
 * Pridani vystupniho rozhrani
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int addOutIf(string &value, TConfig &cfg) {
  const char *pattern = "([a-z0-9][a-z0-9]*)";
  regmatch_t pmatch[2];
  regex_t re;
  TInterface interface;
  if(regcomp( &re, pattern, REG_EXTENDED) != 0) { // kompilace reg. vyrazu
    return EREGEX;
  }
  if (regexec( &re, value.c_str(), 2, pmatch, 0) == 0) { // zkusime nalezt shodu
    interface.name.append(value.c_str()+pmatch[1].rm_so, pmatch[1].rm_eo-pmatch[1].rm_so);
    interface.type = OUT_IF; // jde o vnejsi rozhrani
    interface.fd = -1;       // file descriptor prozatim nastavime na -1
    cfg.interface_list.push_back(interface); // pridame dane zarizeni do seznamu
  }
  regfree(&re);
  return OK;
}


/**
 * Nastaveni podminky IPv6 prefixu odesilatele
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setSenderPrefix(string &value, TConfig &cfg) {
  if (value == "ANY") {
    for (unsigned int i = 0; i < 16; i++) {
      cfg.senderPrefix.s6_addr[i] = 0;
    }
    cfg.senderPrefixLen = -1;
    return OK;
  }

  const char *pattern = "([0-9a-fA-F:]+)/([0-9]+)";
  regmatch_t pmatch[3];
  regex_t re;
  if(regcomp( &re, pattern, REG_EXTENDED) != 0) { // kompilace reg. vyrazu
    return EREGEX;
  }

  if (regexec( &re, value.c_str(), 3, pmatch, 0) == 0) { // zkusime nalezt shodu
    string prefix_str = "";
    string prefixLen_str = "";
    in6_addr prefix;
    unsigned int prefixLen = 0;
    prefix_str.append(value.c_str()+pmatch[1].rm_so, pmatch[1].rm_eo-pmatch[1].rm_so);
    prefixLen_str.append(value.c_str()+pmatch[2].rm_so, pmatch[2].rm_eo-pmatch[2].rm_so);

    // prevod stringove reprezentace adres na typ in6_addr
    if (inet_pton(AF_INET6, prefix_str.c_str(), &(prefix)) != 1) {
      cerr << value << endl;
      return EBADPREFIX;
    }

    // prevod delky prefixu na cislo
    istringstream isIn(prefixLen_str);
    isIn >> prefixLen;

    if (prefixLen > MAX_PREFIX_LEN) {
      cerr << value << endl;
      return EPREFIXLEN;
    }
    for (unsigned int i = 0; i < 16; i++) {
      cfg.senderPrefix.s6_addr[i] = prefix.s6_addr[i];
    }
    cfg.senderPrefixLen = prefixLen;
  } else {
    cerr << value << endl;
    return EBADPREFIX;
  }
  regfree(&re);
  return OK;
}


/**
 * Nastaveni podminky portu odesilatele
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setSenderPort(string &value, TConfig &cfg) {
  if (value == "ANY") {
    cfg.senderPort = 0;
    return OK;
  }
  unsigned int port;

  // prevod delky prefixu na cislo
  istringstream isPort(value);
  isPort >> port;

  cfg.senderPort = port;
  return OK;
}

/**
 * Nastaveni podminky IPv6 prefixu prijemce
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setReceiverPrefix(string &value, TConfig &cfg) {
    if (value == "ANY") {
    for (unsigned int i = 0; i < 16; i++) {
      cfg.receiverPrefix.s6_addr[i] = 0;
    }
    cfg.receiverPrefixLen = -1;
    return OK;
  }

  const char *pattern = "([0-9a-fA-F:]+)/([0-9]+)";
  regmatch_t pmatch[3];
  regex_t re;
  if(regcomp( &re, pattern, REG_EXTENDED) != 0) { // kompilace reg. vyrazu
    return EREGEX;
  }

  if (regexec( &re, value.c_str(), 3, pmatch, 0) == 0) { // zkusime nalezt shodu
    string prefix_str = "";
    string prefixLen_str = "";
    in6_addr prefix;
    unsigned int prefixLen = 0;
    prefix_str.append(value.c_str()+pmatch[1].rm_so, pmatch[1].rm_eo-pmatch[1].rm_so);
    prefixLen_str.append(value.c_str()+pmatch[2].rm_so, pmatch[2].rm_eo-pmatch[2].rm_so);

    // prevod stringove reprezentace adres na typ in6_addr
    if (inet_pton(AF_INET6, prefix_str.c_str(), &(prefix)) != 1) {
      cerr << value << endl;
      return EBADPREFIX;
    }

    // prevod delky prefixu na cislo
    istringstream isIn(prefixLen_str);
    isIn >> prefixLen;

    if (prefixLen > MAX_PREFIX_LEN) {
      cerr << value << endl;
      return EPREFIXLEN;
    }

    for (unsigned int i = 0; i < 16; i++) {
      cfg.receiverPrefix.s6_addr[i] = prefix.s6_addr[i];
    }
    cfg.receiverPrefixLen = prefixLen;

  } else {
    cerr << value << endl;
    return EBADPREFIX;
  }

  regfree(&re);
  return OK;
}

/**
 * Nastaveni podminku portu prijemce
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setReceiverPort(string &value, TConfig &cfg) {
    if (value == "ANY") {
    cfg.receiverPort = 0;
    return OK;
  }
  unsigned int port = 0;

  // prevod delky prefixu na cislo
  istringstream isPort(value);
  isPort >> port;

  cfg.receiverPort = port;
  return OK;
}

/**
 * Nastaveni intervalu (v ns) mezi odesilanim jednobytovych paketu
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setNanoInterval(string &value, TConfig &cfg) {
  if (value == "") {
    cfg.pktNanoInterval = 0.0;
    return OK;
  }
  double interval = 0.0;

  // prevod intervalu na double
  istringstream isInterval(value);
  isInterval >> interval;

  cfg.pktNanoInterval = interval;
  return OK;
}

/**
 * Nastaveni mnozstvi generovaneho sumu
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setNoiseAmount(string &value, TConfig &cfg) {
  unsigned int noiseAmount;

  // prevod delky prefixu na cislo
  istringstream isNoiseAmount(value);
  isNoiseAmount >> noiseAmount;

  cfg.noiseAmount = noiseAmount;
  return OK;
}


/**
 * Nastaveni nazvu souboru s podvrzenou zpravou
 * @param cislo zpravy
 * @param hodnota
 * @param konfigurace programu
 * @return chybovy kod
 */
int setFakeMsgFile(int msgNum, string &value, TConfig &cfg) {
  ifstream mfile(value.c_str(), ios::in);
  if (!mfile) {
    cerr << value << endl;
    return EFAKEMSGFILE;
  }

  if (msgNum == 1) {
    cfg.fakeMsgFile1 = value;
  } else if (msgNum == 2) {
    cfg.fakeMsgFile2 = value;
  }
  return OK;
}
