#include "PClines_c.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#define TIME
#define DEBUG

#ifdef DEBUG
    #include <opencv/cv.h>
    #include <opencv/cxcore.h>
    #include <opencv/highgui.h>
#endif


#ifdef TIME
    #define TIME_ITERATIONS 10
#endif

#ifdef _OPENMP
    #include <omp.h>
#endif 

//#define DDA

/********************************************************/
/* swap function                                        */
/********************************************************/
inline void swap(int* a, int* b)
{
    int temp = *a;
    *a = *b;
    *b = temp;
}


/********************************************************/
/* local max function                                   */
/********************************************************/
static int localMaximum(ImageData32S *input_space, int x_coord, int y_coord, long int point_value, unsigned char local_max_radius, int * y_steps)
{
    //check max in local area	 
    for(int i_x = -local_max_radius; i_x <= local_max_radius; i_x++)
    for(int i_y = -local_max_radius; i_y <= local_max_radius; i_y++)
    {
        int o_x = x_coord + i_x;
        int o_y = y_coord + i_y;

        //Moebius strip attach
        if(o_x < 0) 
        { 
            o_x += input_space->width;
            o_y = input_space->height - 1 - o_y;
        }
        else if(o_x >= input_space->width)
        {
            o_x -= input_space->width;
            o_y = input_space->height - 1 - o_y;
        }

        if((i_x > 0 || i_y > 0) && input_space->data[o_y*input_space->width + o_x] >= point_value)
            return 0;
        if(input_space->data[o_y*input_space->width + o_x] > point_value)
            return 0;
        
    }
    return 1;
}


/********************************************************/
/* insert new max into array of maxima                  */
/********************************************************/
static void insertMaxToArray(PCpoint *maxima, long int *max_weight, unsigned int array_size, PCpoint new_max, long int new_value)
{
    int minID = 0;
    int minVal = max_weight[0];

    for(int i=1; i<array_size; i++)
    {
        if (max_weight[i] < minVal)
        {
            minID = i;
            minVal = max_weight[i];
        }
    }

    if(new_value > minVal)
    {
        maxima[minID] = new_max;    
        max_weight[minID] = new_value;
    }
}


/********************************************************/
/* convert angle-rho line parameters to end points      */
/********************************************************/
void getEdgePoints(PCline *line, unsigned int image_width, unsigned int image_height, PCpoint *A, PCpoint *B)
{
    int width_offset = image_width>>1;
    int height_offset = image_height>>1;

    float sin_t = sin(line->angle);
    float cos_t = cos(line->angle);

    if ((line->angle > QUARTER_PI && line->angle < QUARTER_PI*3) ||			//45-135
	(line->angle > (M_PI + QUARTER_PI) && line->angle < (QUARTER_PI + 3*HALF_PI)))	//225-315
    {//horizontal line
        A->x = -width_offset;
        B->x = width_offset;
        A->y = round(-cos_t/sin_t*A->x + line->rho/sin_t + height_offset);
        B->y = round(-cos_t/sin_t*B->x + line->rho/sin_t + height_offset);
		
        A->x += width_offset;
        B->x += width_offset;
    }
    else
    {//vertical line
        A->y = -height_offset;
        B->y = height_offset;
        A->x = round(-sin_t/cos_t*A->y + line->rho/cos_t + width_offset);
        B->x = round(-sin_t/cos_t*B->y + line->rho/cos_t + width_offset);
		
        A->y += height_offset;
        B->y += height_offset;
    }
}


/********************************************************/
/* calculate line from maxima position                  */
/* return number of detected lines                      */
/********************************************************/
static int maximaToLines(PCline *lines, PCpoint *points, unsigned int line_count, SpaceParameters *space_param)
{
    //convert each maxima
    for(int i = 0; i < line_count; i++)
    {
        PCline new_line;
        float u = points[i].x;
        float v = ((float)points[i].y - space_param->vertical_offset)/space_param->scale;
   
        if(u < space_param->distance)
        {
            new_line.angle = atan2(u, space_param->distance - u);
            new_line.rho   = v*space_param->distance*cos(new_line.angle)/(space_param->distance - u);
        }
        else
        {
            new_line.angle = atan2(2*space_param->distance - u, space_param->distance - u);
            new_line.rho   = v*space_param->distance*sin(new_line.angle)/(2*space_param->distance - u);
        }

        if(new_line.angle < 0.f) new_line.angle += 2*M_PI;

        lines[i] = new_line;
    }

    return line_count;
}

/********************************************************/
/* detect maxima in TS space                            */
/* return detected maxima                               */
/********************************************************/
static int detectMaxima(ImageData32S *TSspace, PCpoint *maxima, unsigned int threshold, unsigned int max_count, unsigned int local_max_radius, SpaceParameters *space_param, int * y_steps)
{

    //init array for maxima
    long int max_weight[max_count];
    PCpoint all_max[max_count];
    int sum_of_max = 0;
    
    memset(max_weight, 0, sizeof(long int)*max_count);

#pragma omp parallel for
    //search for maxima
    for(int space_vertical = local_max_radius; space_vertical < TSspace->height - local_max_radius; space_vertical++)
    for(int space_horizontal = 0; space_horizontal < TSspace->width; space_horizontal++)
    {
        //if value is higher then threshold and it's a local maxima, add line  
        long int point_value = TSspace->data[y_steps[space_vertical] + space_horizontal];	 
        if(point_value > threshold && localMaximum(TSspace, space_horizontal, space_vertical, point_value, local_max_radius, y_steps))
        {
            PCpoint new_max;
            new_max.x = space_horizontal;
            new_max.y = space_vertical;
#pragma omp critical
{	
            sum_of_max++;
            insertMaxToArray(all_max, max_weight, max_count, new_max, point_value);
}
        }
    }

    memcpy(maxima, all_max, sizeof(PCpoint)*min(sum_of_max, max_count));

    return min(sum_of_max, max_count);
}

/********************************************************/
/* use bresenham to draw lines                          */
/* for horizontal lines only                            */
/********************************************************/
inline void lineH(int x0, int y0, int x1, int y1, ImageData32S * space, int * y_steps, int weight)
{
#ifdef DDA

	float slope = (float)(y1 - y0)/(space->width/2.f);
	float y_iter = y0 + 0.5f; 
				
	for(int x = x0; x < x1; x++)
	{									
        space->data[y_steps[(int)(y_iter)] + x] += weight;
		y_iter += slope;		
	}

#else

    int deltax = x1 - x0;
    int deltay = abs(y1 - y0);
    float error = deltax/2;
    int ystep = (y0 < y1) ? 1 : -1;

    while(x0 != x1)
    {
        space->data[y_steps[y0] + x0] += weight;

        error -= deltay;
        if (error < 0)
        {
            y0 += ystep;
            error += deltax;
        }        
        x0++;        
    } 

#endif
}

inline void lineHvarStart(int x0, int y0, int x1, int y1, int x_start, int width, ImageData32S * space, int * y_steps, int weight)
{
    int deltax = x1 - x0;
    int deltay = abs(y1 - y0);
    float error = deltax/2;
    int ystep = (y0 < y1) ? 1 : -1;

    int a = (x_start*deltay > deltax/2) ? (x_start*deltay - deltax/2)/deltax + 1 : 0;
    if((a - 1)*deltax == x_start*deltay - deltax/2) a--;
    
    y0 += ystep*a;
    error += (a*deltax - x_start*deltay);

    x0 += x_start;
    x1 = x0 + width;
 
    while(x0 != x1)
    {
        space->data[y_steps[y0] + x0] += weight;

        error -= deltay;
        if (error < 0)
        {
            y0 += ystep;
            error += deltax;
        }        
        x0++;        
    }
}


/********************************************************/
/* use bresenham to draw lines                          */
/* for verical    lines only                            */
/********************************************************/
inline void lineV(int x0, int y0, int x1, int y1, ImageData32S * space, int * y_steps, int weight)
{
#ifdef DDA
    float slope = (space->width/2.f)/(float)(y1 - y0);

	float x_iter = x0 + 0.5f; 
    int step = (y0 < y1) ? 1 : -1;
    slope *= step;
				
	for(int y = y0; y != y1; y+=step)
	{									
        space->data[y_steps[y] + (int)(x_iter)] += weight;
		x_iter += slope;		
	}

#else
    if (y0 > y1)
    {
        swap(&y0, &y1);
        swap(&x0, &x1);
    }

    int delta_y = y1 - y0;
    int delta_x = abs(x1 - x0);
    int xstep = (x0 < x1) ? 1 : -1;

    if(xstep == 1)
    {
        float error = delta_y / 2;
        while((x0 != x1 || y0 != y1) && x0 < space->width)
        {
            space->data[y_steps[y0] + x0] += weight;
    
            error -= delta_x;
            if (error < 0)
            {
                x0++;
                error += delta_y;
            }
            y0++;            
        }
    }
    else
    {
        float error = -delta_y / 2;
        while((x0 != x1 || y0 != y1) && x1  < space->width)
        {
            space->data[y_steps[y1] + x1] += weight;

            error += delta_x;
            if (error > 0)
            {
                x1++;
                error -= delta_y;
            }    
            y1--;
        }
    }
#endif    
}

/************************************************************************************************/
/* Basic line detection using Parallel coordinates.                                             */
/* Input image is processed and for each non zero pixel polyline in TSspace is rasterized. Then */
/* the TS space is searched for local maxima, which represent lines in source image.            */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image                                                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - width, height                                                           */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesStandard(ImageData8U *inputImg, PCline *lines, unsigned int line_count, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
    ImageData32S TSspace;
    TSspace.width  = spaceW&0xfffe;
    TSspace.height = spaceH|1;
    TSspace.data   = (long int*)calloc(TSspace.width*TSspace.height, sizeof(long int));

#ifdef DEBUG  
    fprintf(stderr,"img: width x height: %d %d \n", inputImg->width, inputImg->height);
    fprintf(stderr,"space: width x height: %d %d \n", TSspace.width, TSspace.height);
#endif

    SpaceParameters space_param;
    space_param.distance = TSspace.width>>1;                                                        //distance of parallel axes
    space_param.vertical_offset = (TSspace.height - 1)>>1;                                          //offset in parallel space
    space_param.scale = (float)TSspace.height/(float)(max(inputImg->width, inputImg->height)|1);    //space scale 

#ifdef DEBUG
    fprintf(stderr,"d, ver_off, scale: %d %d %f\n", space_param.distance, space_param.vertical_offset, space_param.scale);
#endif
       
    //fill TSspace                   
    int horizontal_offset = (inputImg->width)>>1;
    int vertical_offset = (inputImg->height)>>1;
            
    int v_steps[TSspace.height];
    for(int i = 0; i < TSspace.height; i++) 
        v_steps[i] = i*TSspace.width;

#ifdef TIME
    int64 time_start, time_end, time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {
        for (int x = 0; x < TSspace.height; x++)
        for (int y = 0; y < TSspace.width; y++)
            TSspace.data[y + x * TSspace.width] = 0;
   
        time_start = cvGetTickCount();
#endif

        for(int img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
        for(int img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
        {
            if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
            {
                int weight = 1;
                 //move input image coordinates to system with origin in the middle of img and scale
                int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);

                int x0 = 0;                     int y0 = img_x;
                int x1 = space_param.distance;  int y1 = img_y;
                int x2 = TSspace.width;         int y2 = TSspace.height - 1 - img_x;

                
                if(abs(y1 - y0) > abs(x1 - x0))
                    lineV(x0, y0, x1, y1, &TSspace, v_steps, weight);
                else
                    lineH(x0, y0, x1, y1, &TSspace, v_steps, weight);
                
                if(abs(y2 - y1) > abs(x2 - x1))
                    lineV(x1, y1, x2, y2, &TSspace, v_steps, weight);
                else
                    lineH(x1, y1, x2, y2, &TSspace, v_steps, weight);                   
            } 
        }

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time accumulation: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

#ifdef DEBUG
    IplImage *space = cvCreateImage(cvSize(TSspace.width, TSspace.height), IPL_DEPTH_32S, 1);
    for (int i = 0; i < TSspace.height; i++)
    for (int j = 0; j < TSspace.width; j++)
        ((int*)(space->imageData + i*space->widthStep))[j] =  TSspace.data[j + i * TSspace.width];
   
    double max_space;
    IplImage *normalized_space = cvCreateImage(cvGetSize(space), IPL_DEPTH_8U, 1);	    
    cvMinMaxLoc(space, 0, &max_space, NULL, NULL, NULL); 
    cvScale(space, normalized_space, 255/max_space, 0);			
    cvShowImage("space", normalized_space);  
    cvSaveImage("space.png", normalized_space, 0);
    cvReleaseImage(&space);
    cvReleaseImage(&normalized_space);        
#endif

    PCpoint points[line_count];
    int total_max, total_lines;

#ifdef TIME
    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif
        //detect maxima
        total_max = detectMaxima(&TSspace, points, threshold, line_count, local_max_radius, &space_param, v_steps);

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time maxima: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif

        //convert maxima to lines
        total_lines = maximaToLines(lines, points, total_max, &space_param);

#ifdef TIME

        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time lines: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

    free(TSspace.data); 

    return total_lines;
}


/************************************************************************************************/
/* Weighted line detection using Parallel coordinates.                                          */
/* Input image is processed and for each non zero pixel polyline with "weight" equals to edge   */
/*  response is resterized in TSspace. Then the TS space is searched for local maxima, which    */
/*  represent lines in source image.                                                            */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image                                                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - width, height                                                           */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesStandardWeighted(ImageData8U *inputImg, PCline *lines, unsigned int line_count, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
    ImageData32S TSspace;
    TSspace.width  = spaceW&0xfffe;
    TSspace.height = spaceH|1;
    TSspace.data   = (long int*)calloc(TSspace.width*TSspace.height, sizeof(long int));

#ifdef DEBUG  
    fprintf(stderr,"img: width x height: %d %d \n", inputImg->width, inputImg->height);
    fprintf(stderr,"space: width x height: %d %d \n", TSspace.width, TSspace.height);
#endif

    SpaceParameters space_param;
    space_param.distance = TSspace.width>>1;                                                        //distance of parallel axes
    space_param.vertical_offset = (TSspace.height - 1)>>1;                                          //offset in parallel space
    space_param.scale = (float)TSspace.height/(float)(max(inputImg->width, inputImg->height)|1);    //space scale 

#ifdef DEBUG
    fprintf(stderr,"d, ver_off, scale: %d %d %f\n", space_param.distance, space_param.vertical_offset, space_param.scale);
#endif
       
    //fill TSspace                   
    int horizontal_offset = (inputImg->width)>>1;
    int vertical_offset = (inputImg->height)>>1;
            
    int v_steps[TSspace.height];
    for(int i = 0; i < TSspace.height; i++) 
        v_steps[i] = i*TSspace.width;

#ifdef TIME
    int64 time_start, time_end, time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {
        for (int x = 0; x < TSspace.height; x++)
        for (int y = 0; y < TSspace.width; y++)
            TSspace.data[y + x * TSspace.width] = 0;
   
        time_start = cvGetTickCount();
#endif

        for(int img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
        for(int img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
        {
            if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
            {
                int weight = inputImg->data[img_vertical*inputImg->width + img_horizontal];
                 //move input image coordinates to system with origin in the middle of img and scale
                int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);

                int x0 = 0;                     int y0 = img_x;
                int x1 = space_param.distance;  int y1 = img_y;
                int x2 = TSspace.width;         int y2 = TSspace.height - 1 - img_x;

                
                if(abs(y1 - y0) > abs(x1 - x0))
                    lineV(x0, y0, x1, y1, &TSspace, v_steps, weight);
                else
                    lineH(x0, y0, x1, y1, &TSspace, v_steps, weight);
                
                if(abs(y2 - y1) > abs(x2 - x1))
                    lineV(x1, y1, x2, y2, &TSspace, v_steps, weight);
                else
                    lineH(x1, y1, x2, y2, &TSspace, v_steps, weight);                   
            } 
        }

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time accumulation: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

#ifdef DEBUG
    IplImage *space = cvCreateImage(cvSize(TSspace.width, TSspace.height), IPL_DEPTH_32S, 1);
    for (int i = 0; i < TSspace.height; i++)
    for (int j = 0; j < TSspace.width; j++)
        ((int*)(space->imageData + i*space->widthStep))[j] =  TSspace.data[j + i * TSspace.width];
   
    double max_space;
    IplImage *normalized_space = cvCreateImage(cvGetSize(space), IPL_DEPTH_8U, 1);	    
    cvMinMaxLoc(space, 0, &max_space, NULL, NULL, NULL); 
    cvScale(space, normalized_space, 255/max_space, 0);			
    cvShowImage("space", normalized_space);  
    cvSaveImage("space.png", normalized_space, 0);
    cvReleaseImage(&space);
    cvReleaseImage(&normalized_space);        
#endif

    PCpoint points[line_count];
    int total_max, total_lines;

#ifdef TIME
    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif
        //detect maxima
        total_max = detectMaxima(&TSspace, points, threshold, line_count, local_max_radius, &space_param, v_steps);

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time maxima: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif

        //convert maxima to lines
        total_lines = maximaToLines(lines, points, total_max, &space_param);

#ifdef TIME

        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time lines: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

    free(TSspace.data); 

    return total_lines;

}


/************************************************************************************************/
/* Line detection using Parallel coordinates with edge orientation estimation.                  */
/* Input image is masked and for each pixel (with non zero corresponding pixel in mask) edge    */
/*  orientation estimation is calculating, corresponding position in TSspace is found and       */
/*  polyline around this point in TSspace is rasterized. Then the TS space is searched for      */
/*  local maxima, which represent lines in source image.                                        */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image with edge orientation estimation: <0, 256) ... <0, PI)              */
/* - image mask with resulution equals to source image size                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - radius in pixels rasterized around estimation, width, height            */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesEdgeOrientation(ImageData32F *gradientImg, ImageData8U *maskImg, PCline *lines, unsigned int line_count, float estimated_radius, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius)
{
    if(gradientImg->width !=maskImg->width || gradientImg->height != maskImg->height)
    {
        fprintf(stderr, "\nImage and mask size not match!\n");
        return 0; 
    }
 
    ImageData32S TSspace;
    TSspace.width  = spaceW&0xfffe;

    int radius = round(estimated_radius/M_PI*TSspace.width);
    if(radius*2 >= TSspace.width>>1) 
    {
       fprintf(stderr,"\n >NORMAL< radius: %d size: %d\n", radius, TSspace.width>>1);
       return pcLinesStandard(maskImg, lines, line_count, spaceW, spaceH, threshold, local_max_radius);    
    }
    fprintf(stderr,"radius: %d size: %d\n", radius, TSspace.width>>1);

    TSspace.height = spaceH|1;
    TSspace.data   = (long int*)calloc(TSspace.width*TSspace.height, sizeof(long int));

#ifdef DEBUG  
    fprintf(stderr,"img: width x height: %d %d \n", gradientImg->width, gradientImg->height);
    fprintf(stderr,"space: width x height: %d %d \n", TSspace.width, TSspace.height);
#endif

    SpaceParameters space_param;
    space_param.distance = TSspace.width>>1;                                                        //distance of parallel axes
    space_param.vertical_offset = (TSspace.height - 1)>>1;                                          //offset in parallel space
    space_param.scale = (float)TSspace.height/(float)(max(gradientImg->width, gradientImg->height)|1);    //space scale 

#ifdef DEBUG
    fprintf(stderr,"d, ver_off, scale: %d %d %f\n", space_param.distance, space_param.vertical_offset, space_param.scale);
#endif
       
    //fill TSspace                   
    int horizontal_offset = (gradientImg->width)>>1;
    int vertical_offset = (gradientImg->height)>>1;
            
    int v_steps[TSspace.height];
    for(int i = 0; i < TSspace.height; i++) 
        v_steps[i] = i*TSspace.width;

#ifdef TIME
    int64 time_start, time_end, time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {
        for (int x = 0; x < TSspace.height; x++)
        for (int y = 0; y < TSspace.width; y++)
            TSspace.data[y + x * TSspace.width] = 0;
   
        time_start = cvGetTickCount();
#endif

        for(int img_vertical = 0; img_vertical < gradientImg->height; img_vertical++)
        for(int img_horizontal = 0; img_horizontal < gradientImg->width; img_horizontal++)
        {
            if(maskImg->data[img_vertical*maskImg->width + img_horizontal] > 0)
            {
                float orientation = gradientImg->data[img_vertical*gradientImg->width + img_horizontal];
                int u_estimation = (int)(space_param.distance/ (-1.f*sgn(M_PI/2 - orientation) - tan(orientation)) + space_param.distance + 0.5f);
               
                int left_border  = u_estimation - radius;
                int right_border = u_estimation + radius; 	    
            
                int weight = 1;

                int x0 = -1;  int y0;
                int x1 = -1;  int y1;
                int x2 = -1;  int y2;
                int x3 = -1;  int y3;

                 //move input image coordinates to system with origin in the middle of img and scale
                int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);

                int img_ix = TSspace.height - 1 - img_x;

                if(left_border >= 0 && left_border <= space_param.distance && right_border <= space_param.distance)
                {
                    //whole line in S
                    x0 = left_border;           y0 = (int)(img_x + (float)(img_y - img_x)*x0/space_param.distance + 0.5f);
                    x1 = right_border;          y1 = (int)(img_x + (float)(img_y - img_x)*x1/space_param.distance + 0.5f);
                }               
                else if(right_border <= TSspace.width && left_border >= space_param.distance && right_border >= space_param.distance)
                {
                    //whole line in T
                    x0 = left_border;           y0 = (int)(2*img_y - img_ix + (float)((img_ix - img_y)*x0)/space_param.distance + 0.5f);
                    x1 = right_border;          y1 = (int)(2*img_y - img_ix + (float)((img_ix - img_y)*x1)/space_param.distance + 0.5f);                 
                }
                else if(left_border <= space_param.distance && right_border >= space_param.distance)
                {
                     //in S and T
                     x0 = left_border;           y0 = (int)(img_x + (float)(img_y - img_x)*x0/space_param.distance + 0.5f);
                     x1 = space_param.distance;  y1 = img_y;
                     x2 = space_param.distance;  y2 = img_y;
                     x3 = right_border;          y3 = (int)(2*img_y - img_ix + (float)((img_ix - img_y)*x3)/space_param.distance + 0.5f);                               
                }
                else if(left_border < 0)
                {
                    //through the left axes
                    x0 = 0;                             y0 = img_x;
                    x1 = right_border;                  y1 = (int)(img_x + (float)(img_y - img_x)*x1/space_param.distance + 0.5f);              
                    x2 = TSspace.width + left_border;   y2 = (int)(2*img_y - img_ix + (float)((img_ix - img_y)*x2)/space_param.distance + 0.5f);
                    x3 = TSspace.width;                 y3 = img_ix;                    
                }
                else
                {
                    //through the right axes
                    x0 = 0;                             y0 = img_x;
                    x1 = right_border - TSspace.width;  y1 = (int)(img_x + (float)(img_y - img_x)*x1/space_param.distance + 0.5f);              
                    x2 = left_border;                   y2 = (int)(2*img_y - img_ix + (float)((img_ix - img_y)*x2)/space_param.distance + 0.5f);
                    x3 = TSspace.width;                 y3 = img_ix;
                }

                if(x0 > -1)
                {
                    if(abs(y1 - y0) > abs(x1 - x0))
                        lineV(x0, y0, x1, y1, &TSspace, v_steps, weight);
                    else
                        lineH(x0, y0, x1, y1, &TSspace, v_steps, weight);                    
                }

                if(x2 > -1)
                {
                    if(abs(y3 - y2) > abs(x3 - x2))
                        lineV(x2, y2, x3, y3, &TSspace, v_steps, weight);
                    else
                        lineH(x2, y2, x3, y3, &TSspace, v_steps, weight);                   
                }
            } 
        }

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time accumulation: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

#ifdef DEBUG
    IplImage *space = cvCreateImage(cvSize(TSspace.width, TSspace.height), IPL_DEPTH_32S, 1);
    for (int i = 0; i < TSspace.height; i++)
    for (int j = 0; j < TSspace.width; j++)
        ((int*)(space->imageData + i*space->widthStep))[j] =  TSspace.data[j + i * TSspace.width];
   
    double max_space;
    IplImage *normalized_space = cvCreateImage(cvGetSize(space), IPL_DEPTH_8U, 1);	    
    cvMinMaxLoc(space, 0, &max_space, NULL, NULL, NULL); 
    cvScale(space, normalized_space, 255/max_space, 0);			
    cvShowImage("space", normalized_space);  
    cvSaveImage("space.png", normalized_space, 0);
    cvReleaseImage(&space);
    cvReleaseImage(&normalized_space);        
#endif

    PCpoint points[line_count];
    int total_max, total_lines;

#ifdef TIME
    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif
        //detect maxima
        total_max = detectMaxima(&TSspace, points, threshold, line_count, local_max_radius, &space_param, v_steps);

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time maxima: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif

        //convert maxima to lines
        total_lines = maximaToLines(lines, points, total_max, &space_param);

#ifdef TIME

        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time lines: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

    free(TSspace.data); 

    return total_lines;

}

/************************************************************************************************/
/* Basic line detection using Parallel coordinates and parallelized by OpenMP                   */
/* Input image is processed and for each non zero pixel polyline in TSspace is rasterized. Then */
/* the TS space is searched for local maxima, which represent lines in source image.            */
/*                                                                                              */
/* input:                                                                                       */
/* - grayscale source image                                                                     */
/* - array to store output lines and their count                                                */
/* - space parameters - width, height                                                           */
/* - threshold for maxima detection and radius for local maxima search                          */
/*                                                                                              */
/* output:                                                                                      */
/* - array filled with detected lines                                                           */
/*                                                                                              */
/* return                                                                                       */
/* - number of detected lines                                                                   */
/************************************************************************************************/
int pcLinesStandardParallel(ImageData8U *inputImg, PCline *lines, unsigned int line_count, unsigned int spaceW, unsigned int spaceH, unsigned int threshold, unsigned int local_max_radius, int num_threads)
{
    ImageData32S TSspace;

#ifdef _OPENMP
	omp_set_num_threads(num_threads);
    int width = spaceW/2/num_threads;
    TSspace.width  = width*num_threads*2;
#else
    TSspace.width  = spaceW&0xfffe;
#endif

    TSspace.height = spaceH|1;
    TSspace.data   = (long int*)calloc(TSspace.width*TSspace.height, sizeof(long int));

#ifdef DEBUG  
    fprintf(stderr,"img: width x height: %d %d \n", inputImg->width, inputImg->height);
    fprintf(stderr,"space: width x height: %d %d \n", TSspace.width, TSspace.height);
#endif

    SpaceParameters space_param;
    space_param.distance = TSspace.width>>1;                                                        //distance of parallel axes
    space_param.vertical_offset = (TSspace.height - 1)>>1;                                          //offset in parallel space
    space_param.scale = (float)TSspace.height/(float)(max(inputImg->width, inputImg->height)|1);    //space scale 

#ifdef DEBUG
    fprintf(stderr,"d, ver_off, scale: %d %d %f\n", space_param.distance, space_param.vertical_offset, space_param.scale);
#endif
       
    //fill TSspace                   
    int horizontal_offset = (inputImg->width)>>1;
    int vertical_offset = (inputImg->height)>>1;
            
    int v_steps[TSspace.height];
    for(int i = 0; i < TSspace.height; i++) 
        v_steps[i] = i*TSspace.width;


#ifdef TIME
    int64 time_start, time_end, time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {
        for (int x = 0; x < TSspace.height; x++)
        for (int y = 0; y < TSspace.width; y++)
            TSspace.data[y + x * TSspace.width] = 0;
   
        time_start = cvGetTickCount();
#endif


#ifdef _OPENMP
    #pragma omp parallel
    {
        int thread = omp_get_thread_num();
        int count = omp_get_num_threads();
       
        if(count == 2)
        {
            for(int img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
            for(int img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
            {
                if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
                {
                    int weight = 1;
                     //move input image coordinates to system with origin in the middle of img and scale
                    int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                    int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);

                    int x0 = 0;                     int y0 = img_x;
                    int x1 = space_param.distance;  int y1 = img_y;
                    int x2 = TSspace.width;         int y2 = TSspace.height - 1 - img_x;

                    if(thread)
                    {
                        if(abs(y1 - y0) > abs(x1 - x0))
                            lineV(x0, y0, x1, y1, &TSspace, v_steps, weight);
                        else 
                            lineH(x0, y0, x1, y1, &TSspace, v_steps, weight);
                    }
                    else
                    {
                        if(abs(y2 - y1) > abs(x2 - x1)) 
                            lineV(x1, y1, x2, y2, &TSspace, v_steps, weight);
                        else
                            lineH(x1, y1, x2, y2, &TSspace, v_steps, weight); 
                    }
                }
            }
        }
        else //lineHvarStart(int x0, int y0, int x1, int y1, int x_start, int x_end, ImageData32S * space, int * y_steps, int weight)
        { 
            int width = space_param.distance/count;
           
            for(int img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
            for(int img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
            {
                if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
                {
                     //move input image coordinates to system with origin in the middle of img and scale
                    int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                    int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);


                    int x0 = 0;                     int y0 = img_x;
                    int x1 = space_param.distance;  int y1 = img_y;
                    int x2 = TSspace.width;         int y2 = TSspace.height - 1 - img_x;

                    lineHvarStart(x0, y0, x1, y1, thread*width, width, &TSspace, v_steps, 1);
                    lineHvarStart(x1, y1, x2, y2, thread*width, width, &TSspace, v_steps, 1);
                }
            }
        }
    }
#else
        for(int img_vertical = 0; img_vertical < inputImg->height; img_vertical++)
        for(int img_horizontal = 0; img_horizontal < inputImg->width; img_horizontal++)
        {
            if(inputImg->data[img_vertical*inputImg->width + img_horizontal] > 0)
            {
                int weight = 1;
                 //move input image coordinates to system with origin in the middle of img and scale
                int img_x = (int)((img_horizontal - horizontal_offset)*space_param.scale + space_param.vertical_offset + 0.5f);
                int img_y = (int)((img_vertical   - vertical_offset)*space_param.scale + space_param.vertical_offset + 0.5f);

                int x0 = 0;                     int y0 = img_x;
                int x1 = space_param.distance;  int y1 = img_y;
                int x2 = TSspace.width;         int y2 = TSspace.height - 1 - img_x;

                
                if(abs(y1 - y0) > abs(x1 - x0))
                    lineV(x0, y0, x1, y1, &TSspace, v_steps, weight);
                else
                    lineH(x0, y0, x1, y1, &TSspace, v_steps, weight);
                
                if(abs(y2 - y1) > abs(x2 - x1))
                    lineV(x1, y1, x2, y2, &TSspace, v_steps, weight);
                else
                    lineH(x1, y1, x2, y2, &TSspace, v_steps, weight);                  

            } 
        }
#endif
    

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time accumulation: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

#ifdef DEBUG
    IplImage *space = cvCreateImage(cvSize(TSspace.width, TSspace.height), IPL_DEPTH_32S, 1);
    for (int i = 0; i < TSspace.height; i++)
    for (int j = 0; j < TSspace.width; j++)
        ((int*)(space->imageData + i*space->widthStep))[j] =  TSspace.data[j + i * TSspace.width];
   
    double max_space;
    IplImage *normalized_space = cvCreateImage(cvGetSize(space), IPL_DEPTH_8U, 1);	    
    cvMinMaxLoc(space, 0, &max_space, NULL, NULL, NULL); 
    cvScale(space, normalized_space, 255/max_space, 0);			
    cvShowImage("space", normalized_space);  
    cvSaveImage("space.png", normalized_space, 0);
    cvReleaseImage(&space);
    cvReleaseImage(&normalized_space);        
#endif

    PCpoint points[line_count];
    int total_max, total_lines;

#ifdef TIME
    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif
        //detect maxima
        total_max = detectMaxima(&TSspace, points, threshold, line_count, local_max_radius, &space_param, v_steps);

#ifdef TIME
        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time maxima: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

    time_sum = 0;
    for(int i = 0; i<TIME_ITERATIONS; i++)
    {          
        time_start = cvGetTickCount();
#endif

        //convert maxima to lines
        total_lines = maximaToLines(lines, points, total_max, &space_param);

#ifdef TIME

        time_end = cvGetTickCount();
        time_sum += time_end - time_start;
    }
    fprintf(stderr, "time lines: %f \n", time_sum/(cvGetTickFrequency()*1.0e6)/TIME_ITERATIONS);

#endif

    free(TSspace.data); 

    return total_lines;
}
               
 
