#include <cstdlib>
#include <cstdio>
#include <opencv/cv.h>
#include <opencv/cxcore.h>
#include <opencv/highgui.h>

#include "PClines_c.h"

enum {STANDARD, WEIGHTED, SOBEL, PARALLEL};

#define SOBEL_THRESH      100
#define LOCAL_MAX_RADIUS  5
#define BEST_LINES        10 

/********************************************************/
/* sobel                                                */
/********************************************************/
void sobelResponse(IplImage *input, ImageData8U *data)
{	

 for(int y = 0; y < input->height; y++)
 {
  for(int x = 0; x < input->width; x++)
  {
   const bool XL = (x > 0);
   const bool XR = (x < input->width - 1);
   const bool YT = (y > 0);
   const bool YB = (y < input->height - 1);

   unsigned char AA = ((unsigned char*)input->imageData)[x-XL + (y-YT)*input->widthStep];
   unsigned char AB = ((unsigned char*)input->imageData)[x	   + (y-YT)*input->widthStep];
   unsigned char AC = ((unsigned char*)input->imageData)[x+XR + (y-YT)*input->widthStep];
   unsigned char BA = ((unsigned char*)input->imageData)[x-XL + (y)   *input->widthStep];
   unsigned char BC = ((unsigned char*)input->imageData)[x+XR + (y)   *input->widthStep];
   unsigned char CA = ((unsigned char*)input->imageData)[x-XL + (y+YB)*input->widthStep];
   unsigned char CB = ((unsigned char*)input->imageData)[x	   + (y+YB)*input->widthStep];
   unsigned char CC = ((unsigned char*)input->imageData)[x+XR + (y+YB)*input->widthStep];

   int h = - AA - 2*AB - AC + CA + 2*CB + CC;
   int v = - AA - 2*BA - CA + AC + 2*BC + CC;
			
   float sobel = float(max(abs(v)/4,abs(h)/4));

   if(sobel > SOBEL_THRESH )
   {
    data->data[y*input->width + x] = int(sobel);
   }
   else
   {
    data->data[y*input->width + x] = 0;
   }
  }
 }
}


void sobelGrad(IplImage *input, ImageData8U *mask, ImageData32F *data)
{	

 for(int y = 0; y < input->height; y++)
 {
  for(int x = 0; x < input->width; x++)
  {
   const bool XL = (x > 0);
   const bool XR = (x < input->width - 1);
   const bool YT = (y > 0);
   const bool YB = (y < input->height - 1);

   unsigned char AA = ((unsigned char*)input->imageData)[x-XL + (y-YT)*input->widthStep];
   unsigned char AB = ((unsigned char*)input->imageData)[x	   + (y-YT)*input->widthStep];
   unsigned char AC = ((unsigned char*)input->imageData)[x+XR + (y-YT)*input->widthStep];
   unsigned char BA = ((unsigned char*)input->imageData)[x-XL + (y)   *input->widthStep];
   unsigned char BC = ((unsigned char*)input->imageData)[x+XR + (y)   *input->widthStep];
   unsigned char CA = ((unsigned char*)input->imageData)[x-XL + (y+YB)*input->widthStep];
   unsigned char CB = ((unsigned char*)input->imageData)[x	   + (y+YB)*input->widthStep];
   unsigned char CC = ((unsigned char*)input->imageData)[x+XR + (y+YB)*input->widthStep];

   int h = - AA - 2*AB - AC + CA + 2*CB + CC;
   int v = - AA - 2*BA - CA + AC + 2*BC + CC;
			
   float sobel = float(max(abs(v)/4,abs(h)/4));

   if(sobel > SOBEL_THRESH )
   {
	float grad = (float)atan2((float)h,(float)v) + (float)CV_PI;					
	grad = fmodf(grad, (float)CV_PI);

	data->data[y*input->width + x] = grad;
    mask->data[y*input->width + x] = 1;

   }
   else
   {
    data->data[y*input->width + x] = 0;
    mask->data[y*input->width + x] = 0;
   }
  }
 }    
}

/********************************************************/
/* main function                                        */
/********************************************************/
int main(int argc, char** argv)
{
 if(argc < 2) 
 { 
  printf("\n run ./pclines image [mode] \n where mode is 0 - standard PC, 1 - weighted PC, 2 - sobel estimation (default standard) \n");
  return 0;
 }

 IplImage *rgb = cvLoadImage(argv[1], CV_LOAD_IMAGE_COLOR);
 IplImage *src = cvLoadImage(argv[1], CV_LOAD_IMAGE_GRAYSCALE);

 int mode;
 if(argc >= 3)
 {
  mode = atoi(argv[2]);
 }
 else
 {
  mode = 0;
 }
  
 PCline lines[BEST_LINES];
 
 ImageData8U image, mask;
 image.width = src->width;
 image.height = src->height;
 image.data = (unsigned char*)malloc(src->width*src->height*sizeof(unsigned char));

 ImageData32F gradient;
 gradient.width = src->width;
 gradient.height = src->height;
 gradient.data = (float*)malloc(src->width*src->height*sizeof(float));

 unsigned int space_height = max(src->height, src->width);
 unsigned int space_width  = space_height/2;
 unsigned int threshold    = space_height/16;

 int line_count = 0;

 ImageData8U image2;
 image2.width = src->width;
 image2.height = src->height;
 image2.data = (unsigned char*)malloc(image2.width*image2.height*sizeof(unsigned char));
 for(int y = 0; y < src->height; y++)
  for(int x = 0; x < src->width; x++)
    image2.data[y*image2.width + x] = ((unsigned char*)src->imageData)[x + y*src->widthStep];


 switch (mode)
 {
  case WEIGHTED:
   fprintf(stderr,"weighted \n");
   sobelResponse(src, &image);
   line_count = pcLinesStandardWeighted(&image, lines, BEST_LINES, space_width, space_height, threshold*50, LOCAL_MAX_RADIUS);

  break;
 
  case SOBEL:
   mask.data = (unsigned char*)malloc(src->width*src->height*sizeof(unsigned char));
   mask.width = src->width;
   mask.height = src->height;
   sobelGrad(src, &mask, &gradient);
   line_count = pcLinesEdgeOrientation(&gradient, &mask, lines, BEST_LINES, M_PI/32, space_width, space_height, threshold, LOCAL_MAX_RADIUS);
  break;

    case PARALLEL:
   fprintf(stderr,"parallel \n");
   sobelResponse(src, &image);    
   line_count = pcLinesStandardParallel(&image, lines, BEST_LINES, space_width, space_height, threshold, LOCAL_MAX_RADIUS, 2);

   //line_count = pcLinesStandard(&image2, lines, BEST_LINES, space_width/2, space_height, threshold, LOCAL_MAX_RADIUS);
  break; 
  
  case STANDARD:
  default:
   fprintf(stderr,"normal \n");
   sobelResponse(src, &image);    
   line_count = pcLinesStandard(&image, lines, BEST_LINES, space_width, space_height, threshold, LOCAL_MAX_RADIUS);
   //line_count = pcLinesStandard(&image2, lines, BEST_LINES, space_width/2, space_height, threshold, LOCAL_MAX_RADIUS);
  break;  


 }

 for(int i=0; i<min(line_count, BEST_LINES); i++)
 {
  printf("\n %d theta rho: %f %f", i+1, lines[i].angle/CV_PI*180.f, lines[i].rho); 
  PCpoint A, B;
  getEdgePoints(&lines[i], src->width, src->height, &A, &B);
  //cvLine(rgb, cvPoint(A.x, A.y), cvPoint(B.x, B.y), CV_RGB(0,0,0), 3);
  cvLine(rgb, cvPoint(A.x, A.y), cvPoint(B.x, B.y), CV_RGB(255,0,0));
 }

 cvShowImage("PClines", rgb);
 cvSaveImage("PClines.png", rgb);
 cvShowImage("src", src);

 cvWaitKey(0);
 if(mode == SOBEL) 
 {
  free(mask.data);
 } 

 free(image.data);
 free(image2.data);
 free(gradient.data);
 cvReleaseImage(&src);
 cvReleaseImage(&rgb);
 return 0;
}
/********************************************************/
/* EOF                                                  */
/********************************************************/
