/* mbr.c - Source file: Handling the Master boot record (MBR).
 * Author: Vlastimil Kosar <ikosar@fit.vutbr.cz>
 */

#include <stdint.h>
#include "spi.h"
#include "mbr.h"
#include "sd.h"
#include "errors.h"
#include "compose.h"

/*
 * Read MBR from SD card, parse it and locate partition which contains image of linux.
 * Params:
 *   info - partition structure
 * Returns: 1 if everything is OK, 0 if error happend.
 */
uint32_t read_mbr(t_mbr_info *info, t_sd_info *sd_info)
{
    // Cycle counter
    uint8_t i;
    // Indicates presence of at least one partition
    uint8_t has_partition = 0;
    // Sector loaded from SD card.
    t_sd_block sector;
    
    // Read first sector from SD card - MBR and check if sector was loaded correctly
    if (read_sd_block_512B(0, &sector, sd_info) == 0)
    {
        error(ERR_SD_READ);
        return 0;
    }
    
    // Check if loaded sector is MBR - last two bytes must be little-endian number 0xAA55
    if (sector[510] != 0x55 || sector[511] != 0xAA)
    {
        error(ERR_MBR_NOT_PRESENT);
        return 0;
    }
    
    // Read partitions
    for (i = 0; i < 4; i++)
    {
        // Address of partition first sector in LBA
        uint32_t begin_addr = 446 + i * 16 + 8;
        // Address of partition size in sectors
        uint32_t size_addr = 446 + i * 16 + 12;
        // Address of partition type
        uint32_t fs_addr = 446 + i * 16 + 4;
        
        // read data into structures
        (*info)[3-i].start = compose_32(&sector[begin_addr], LITTLE, LITTLE);
        (*info)[3-i].size = compose_32(&sector[size_addr], LITTLE, LITTLE);
        (*info)[3-i].valid = ((*info)[3-i].size > 0) ? 1 : 0;
        (*info)[3-i].fs = sector[fs_addr];
        
        // update info about existence of at least one partition
        if ((*info)[3-i].valid == 1)
        {
            has_partition = 1;
        }
    }
    
    // Check if some partition exist
    if (has_partition == 0)
    {
        error(ERR_MBR_NO_PARTITION);
        return 0;
    }
    
    return has_partition;
}
