/* mb_hw.c - Source file: Hardware specific functions
 *           such as POST and application start.
 * Author: Vlastimil Kosar <ikosar@fit.vutbr.cz>
 */

#include <stdint.h>
#include <stdio.h>
#include <xil_cache.h>
#include <xil_testmem.h>
#include "parameters.h"
#include "boot_global_defs.h"
#include "mb_hw.h"
#include "xparameters.h"
#include "errors.h"
#include "xstatus.h"

/*
 *  Run power on self test and inits.
 *  Returns: 1 if all is OK
 *           0 if error occured
 */
uint32_t post()
{
	// Enable caches
#if XPAR_MICROBLAZE_0_USE_ICACHE
    Xil_ICacheInvalidate();
    Xil_ICacheEnable();
#endif

#if XPAR_MICROBLAZE_0_USE_DCACHE
    Xil_DCacheInvalidate();
    Xil_DCacheEnable();
#endif

#ifdef MEMORY_TEST
    // if memory test is required, run it
#ifdef DEBUG
    xil_printf("Memory test of DDR2 SDRAM ... ");
#endif /* DEBUG */
    if (Xil_TestMem32((uint32_t *)XPAR_MCB_DDR2_S0_AXI_BASEADDR , ((XPAR_MCB_DDR2_S0_AXI_HIGHADDR + 1) - (XPAR_MCB_DDR2_S0_AXI_BASEADDR + MEM_PROTECT_BYTES)) >> 4, 0, XIL_TESTMEM_ALLMEMTESTS) != XST_SUCCESS)
    {
    	error(ERR_MEM_TEST);
    	return 0;
    }
#ifdef DEBUG
    else
    {
    	xil_printf("PASSED\r\n");
    }
#endif /* DEBUG */
#endif /* MEMORY_TEST */
	return 1;
}

/*
 * Run loaded application.
 * Params:
 * address - memory map info.
 */
void kick_and_pray(t_run_info *address)
{
	// Disable caches
#if XPAR_MICROBLAZE_0_USE_DCACHE
    Xil_DCacheDisable();
    Xil_DCacheInvalidate();
#endif

#if XPAR_MICROBLAZE_0_USE_ICACHE
    Xil_ICacheDisable();
    Xil_ICacheInvalidate();
#endif

    // Kick
    // Run standalone application
    if (address->type == APPLICATION)
    {
#ifdef DEBUG
    	xil_printf("\r\nKick & pray: %x\r\n", address->image_addres);
#endif
    	((void(*)(void))(address->image_addres))();
    }
    // Run linux
    else
    {
    	 void (*linux) (uint8_t *, uint32_t, uint32_t);
    	 linux = (void (*)(uint8_t *, uint32_t, uint32_t))address->image_addres;
    	 linux(address->cmdline, address->initramfs_address, address->fdtb_address);
    }
    // and pray
}
