/*
 * fifo.h - Header file: Implements fifo like data structure for efficent SREC
 *          extraction from blocks.
 * Author: Vlastimil kosar
 */

typedef struct
{
    uint8_t data[1200]; // data array
    uint16_t start;     // first byte of valid data
    uint16_t end;       // first free position in data
} t_fifo;

/*
 * Init the fifo like structure.
 * Params:
 * fifo - Fifo control structure.
 */
void fifo_init(t_fifo *fifo);

/*
 * Is fifo length smaller than 512B?
 * Params:
 * fifo - Fifo control structure.
 * Returns: 1 if fifo->end - fifo->start < 512B
 *          0 otherwise
 */
uint8_t is_block_free(t_fifo *fifo);

/*
 * Add block to fifo.
 * Params:
 * fifo  - Fifo control structure.
 * block - 512B block 
 * Returns: 1 if all is OK
 *          0 if error occured
 */
uint8_t add_block(t_fifo *fifo, t_sd_block block);

/*
 * Get SREC line.
 * Params:
 * fifo - Fifo control structure.
 * line - SREC line. Max size: SREC_MAX_BYTES.
 * Returns: 1 if all is OK
 *          0 if error occured
 */
uint8_t get_srec_line(t_fifo *fifo, uint8_t *line);

/*
 * Rotate fifo->start to 0.
 * Params:
 *  fifo - Fifo control structure.
 */
void rotate(t_fifo *fifo);

// The fast path
/*
 * Get address of 512B block of fifo memory for fast load of data.
 * Params:
 *   fifo - fifo control structure.
 * Returns address of the 512B block.
 */
inline uint8_t* get_blk_address(t_fifo *fifo);

/*
 * Update counter values uppon successfull load of SD block.
 * Params:
 *   fifo - fifo control structure.
 */
inline void add_block_fast(t_fifo *fifo);

/*
 * Get SREC line fast.
 * Params:
 * fifo - Fifo control structure.
 * line - pointer to SREC line. Max size: SREC_MAX_BYTES.
 * Returns: 1 if all is OK
 *          0 if error occured
 *          2 if ended prematurely - can happen in last block - practically in corect SREC file imposible as reading will be ended before.
 */
uint8_t get_srec_line_fast(t_fifo *fifo, uint8_t **line);

