/*
 * fifo.c - Header file: Implements fifo like data structure for efficent SREC
 *          extraction from blocks.
 * Author: Vlastimil kosar
 */

#include <stdint.h>
#include "boot_global_defs.h"
#include "spi.h"
#include "sd.h"
#include "fifo.h"
#include "srec.h"

/*
 * Init the fifo like structure.
 * Params:
 * fifo - Fifo control structure.
 */
void fifo_init(t_fifo *fifo)
{
    fifo->start = 0;
    fifo->end = 0;
}

/*
 * Is fifo length smaller than 600B?
 * Params:
 * fifo - Fifo control structure.
 * Returns: 1 if fifo->end - fifo->start < 600B
 *          0 otherwise
 */
uint8_t is_block_free(t_fifo *fifo)
{
    if (fifo->end - fifo->start < 600)
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

/*
 * Get address of 512B block of fifo memory for fast load of data.
 * Params:
 *   fifo - fifo control structure.
 * Returns address of the 512B block.
 */
inline uint8_t* get_blk_address(t_fifo *fifo)
{
    return &(fifo->data[fifo->end]);
}

/*
 * Update counter values uppon successfull load of SD block.
 * Params:
 *   fifo - fifo control structure.
 */
inline void add_block_fast(t_fifo *fifo)
{
    fifo->end = fifo->end + 512;
}

/*
 * Get SREC line fast.
 * Params:
 * fifo - Fifo control structure.
 * line - pointer to SREC line. Max size: SREC_MAX_BYTES.
 * Returns: 1 if all is OK
 *          0 if error occured
 *          2 if ended prematurely - can happen in last block - practically in corect SREC file imposible as reading will be ended before.
 */
uint8_t get_srec_line_fast(t_fifo *fifo, uint8_t **line)
{
    uint8_t c;
    int count = 0;
    uint16_t i;

    for (i = fifo->start; i < fifo->end; i++)
    {
        c = fifo->data[i];
        
        // Check end of SREC line
        if (c == 0xD)
        {
            // Eat up the 0xA too and move to the next char
            i = i + 2;
            
            *line = &(fifo->data[fifo->start]);
            
            // Update fifo start
            fifo->start = i;
            
            return 1;
        }

        count++;
        if (count > SREC_MAX_BYTES) 
            return 0;
    }
    return 2;
}

/*
 * Add block to fifo.
 * Params:
 * fifo  - Fifo control structure.
 * block - 512B block 
 * Returns: 1 if all is OK
 *          0 if error occured
 */
uint8_t add_block(t_fifo *fifo, t_sd_block block)
{
	uint32_t i;

    if (1536 - (fifo->end - fifo->start) < 512)
    {
        return 0;
    }
    
    for (i = 0; i < 512; i++)
    {
        fifo->data[fifo->end + i] = block[i];
    }
    fifo->end = fifo->end + 512;
    
    return 1;
}

/*
 * Get SREC line.
 * Params:
 * fifo - Fifo control structure.
 * line - SREC line. Max size: SREC_MAX_BYTES.
 * Returns: 1 if all is OK
 *          0 if error occured
 *          2 if ended prematurely - can happen in last block - practically in corect SREC file imposible as reading will be ended before.
 */
uint8_t get_srec_line(t_fifo *fifo, uint8_t *line)
{
    uint8_t c;
    int count = 0;
    uint16_t i;

    for (i = fifo->start; i < fifo->end; i++)
    {
        c = fifo->data[i];
        
        // Check end of SREC line
        if (c == 0xD)
        {
            // Eat up the 0xA too and move to the next char
            i = i + 2;
            
            // Update fifo start
            fifo->start = i;
            
            return 1;
        }
        *line++ = c;
        count++;
        if (count > SREC_MAX_BYTES) 
            return 0;
    }
    return 2;
}

/*
 * Rotate fifo->start to 0.
 * Params:
 *  fifo - Fifo control structure.
 */
void rotate(t_fifo *fifo)
{
	uint16_t i;

    // move data
    for (i = fifo->start; i < fifo->end; i++)
    {
        fifo->data[i - fifo->start] = fifo->data[i];
    }
    fifo->end = i - fifo->start;
    fifo->start = 0;
}
