proc swapp_get_name {} {
    return "uSonda SD Card Bootloader";
}

proc swapp_get_description {} {
    return "Advanced bootloader for loading SREC images from SD card. This program assumes that you have an SREC image loaded into SD card already. The program also assumes that the target SREC image is an application for this processor that does not overlap the bootloader and resides in separate physical memory in the hardware. Typically this application is initialized into BRAM so that it bootloads the SREC image when the FPGA is powered up.
    
Features:
    * Physical layer of SD, SDHC and SDXC cards supported.
    * FAT16 and FAT32 file system supported.
    * Optional memory test supported.
    * Optional loading of XXTEA encrypted SREC images supported.
    * Only 8.3 filenames supported.
    * Highly parametrizable.

Don't forget to modify parameters.h to set your options and file name!
    
Optionaly modify the ld scripts to configure place where your memory is located and where do you want to locate the stack and heap!

NOTE: The 16kB of required BRAM memory is valid only for minimal configuration with stack and heap in DDR SDRAM, full configuration may take at least 32kB of BRAM!
";
}

proc check_standalone_os {} {
    set oslist [xget_sw_modules "type" "os"];

    if { [llength $oslist] != 1 } {
        return 0;
    }
    set os [lindex $oslist 0];

    if { $os != "standalone" } {
        error "This bootloader is supported only on the Standalone Board Support Package.";
    }
}

proc generate_stdout_config { fid } {
    set stdout [xget_sw_module_parameter "standalone" "STDOUT"];

    # if stdout is uartlite, we don't have to generate anything
    set stdout_type [xget_ip_attribute "type" $stdout];

    if { [regexp -nocase "uartlite" $stdout_type] || [string match -nocase "mdm" $stdout_type] } {
        return;
    } elseif { [regexp -nocase "uart16550" $stdout_type] } {
	# mention that we have a 16550
        puts $fid "#define STDOUT_IS_16550";

        # and note down its base address
	set prefix "XPAR_";
	set postfix "_BASEADDR";
	set stdout_baseaddr_macro $prefix$stdout$postfix;
	set stdout_baseaddr_macro [string toupper $stdout_baseaddr_macro];
	puts $fid "#define STDOUT_BASEADDR $stdout_baseaddr_macro";
    }
}

proc get_os {} {
    set oslist [xget_sw_modules "type" "os"];
    set os [lindex $oslist 0];

    if { $os == "" } {
        error "No Operating System specified in the Board Support Package.";
    }
    
    return $os;
}

proc get_stdout {} {
    set os [get_os];
    set stdout [xget_sw_module_parameter $os "STDOUT"];
    return $stdout;
}

proc check_stdout_hw {} {
    set uartlites [xget_ips "type" "uartlite"];
    if { [llength $uartlites] == 0 } {
        # we do not have an uartlite
	set uart16550s [xget_ips "type" "uart16550"];
	if { [llength $uart16550s] == 0 } {      
	    # Check for MDM-Uart peripheral. The MDM would be listed as a peripheral
	    # only if it has a UART interface. So no further check is required
	    set mdmlist [xget_ips "type" "mdm"]
	    if { [llength $mdmlist] == 0 } {
		error "This application requires a Uart IP in the hardware."
	    }
	}
    }
}

proc check_stdout_sw {} {
    set stdout [get_stdout];
    if { $stdout == "none" } {
        error "The STDOUT parameter is not set on the OS. The bootloader requires STDOUT to be set."
    }
}

proc get_mem_size { memlist } {
    return [lindex $memlist 4];
}

proc require_memory {memsize} {
    set imemlist [xget_memory_ranges "access_type" "I"];
    set idmemlist [xget_memory_ranges "access_type" "ID"];
    set dmemlist [xget_memory_ranges "access_type" "D"];

    set memlist [concat $imemlist $idmemlist $dmemlist];

    while { [llength $memlist] > 3 } {
        set mem [lrange $memlist 0 4];
        set memlist [lreplace $memlist 0 4];

        if { [get_mem_size $mem] >= $memsize } {
            return 1;
        }
    }

    error "This application requires atleast $memsize bytes of memory.";
}

proc swapp_is_supported_hw {} {
    # bootloader is supported only for microblaze
    set proc_instance [xget_processor_name];
    set proc_type [xget_ip_attribute "type" $proc_instance];

    if { ($proc_type != "microblaze")} {
	error "This application is supported only for MicroBlaze processors";
    }

    # check for uart peripheral
    check_stdout_hw;

    # we require atleast 16KB memory
    require_memory "16384";
}

proc swapp_is_supported_sw {} {
    # check for standalone OS
    check_standalone_os;

    # check for stdout being set
    check_stdout_sw;
}


proc swapp_generate {} {
    # cleanup this file for writing
    set fid [open "platform_config.h" "w+"];
    puts $fid "#ifndef __PLATFORM_CONFIG_H_";
    puts $fid "#define __PLATFORM_CONFIG_H_\n";

    # if we have a uart16550 as stdout, then generate some config for that
    generate_stdout_config $fid;

    puts $fid "#endif";
    close $fid;
}

proc get_mem_name { memlist } {
    return [lindex $memlist 0];
}

proc get_mem_type { memlist } {
    return [lindex $memlist 1];
}

proc get_mem_ip { memlist } {
    return [lindex $memlist 2];
}

proc get_mem_base { memlist } {
    return [format "%08x" [lindex $memlist 3]];
}

proc get_mem_size { memlist } {
    return [lindex $memlist 4];
}

proc extract_bram_memories { memlist } {
    set bram_mems [];
    set l [llength $memlist];
    while { [llength $memlist] > 3 } {
        set mem [lrange $memlist 0 4];
        set memlist [lreplace $memlist 0 4];

        if { [get_mem_type $mem] == "BRAM" } {
            set bram_mems [concat $bram_mems $mem];
        }
    }

    return $bram_mems;
}

proc get_program_code_memory {} {
    set imemlist [xget_memory_ranges "access_type" "I"];
    set idmemlist [xget_memory_ranges "access_type" "ID"];

    set memlist [concat $imemlist $idmemlist];
    set codemem [extract_bram_memories $memlist];
    return $codemem;
}

proc get_program_data_memory {} {
    set dmemlist [xget_memory_ranges "access_type" "D"];
    set idmemlist [xget_memory_ranges "access_type" "ID"];

    set memlist [concat $dmemlist $idmemlist];
    set datamem [extract_bram_memories $memlist];
    return $datamem;
}

proc swapp_get_linker_constraints {} {
    set code_memory [get_mem_name [get_program_code_memory]];
    set data_memory [get_mem_name [get_program_data_memory]];

    # set code & data memory to point to bram
    # no need for vectors section (affects PPC linker scripts only)
    # no need for heap
    return "code_memory $code_memory data_memory $data_memory vector_section no heap 0 stack 3072";
}
