
#include <ros/ros.h>
#include <sensor_msgs/LaserScan.h>
#include <sensor_msgs/Image.h>
#include <sensor_msgs/CameraInfo.h>
#include <sensor_msgs/PointCloud2.h>
#include <nav_msgs/Odometry.h>
#include <std_msgs/UInt32.h>

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <iterator>

#include <boost/algorithm/string.hpp>
#include <rosbag/bag.h>
#include <tf/tfMessage.h>

#include <message_filters/subscriber.h>
#include <message_filters/time_synchronizer.h>
#include <message_filters/synchronizer.h>
#include <message_filters/sync_policies/approximate_time.h>

#include <termios.h>

using namespace std;
using namespace sensor_msgs;
using namespace rosbag;
using namespace message_filters;

Bag writeb;
int counter = 0;
int nth = 1;

//reference time
//ros::Time ref;

class TerminalIO
{
public:
    //! Default constructor calls setup() automatically.
    TerminalIO() { setup(); }

    //! Destructor restores the original terminal settings.
    ~TerminalIO() { restore(); }

    //! Reads a single character from stdin.
    //! - Returns EOF if there is no pending character on the input.
    int readCharFromStdin();

protected:
    //! Original terminal flags that should be restored
    termios m_orig_flags;

    //! Helper file descriptors
    fd_set m_stdin_fdset;
    int m_maxfd;

    //! Setup terminal so that reading of a single character is a non-blocking
    //! operation (so called noncannonical mode).
    void setup();

    //! Restore original settings
    void restore();
};

void syncCallback(const std_msgs::UInt32ConstPtr& msg)
{
	//cout << "t: " << msg->data << " " << ros::Time::now() << endl;
	writeb.write("/sync", ros::Time::now(), msg);
}

void scanCallback(const sensor_msgs::LaserScanConstPtr& msg)
{
	writeb.write("/scan", msg->header.stamp, msg);
}

void tfCallback(const tf::tfMessageConstPtr& msg)
{
	//write to bag
	writeb.write("/tf", ros::Time::now(),msg);
}

void kinectCallback( const sensor_msgs::ImageConstPtr& dep, const sensor_msgs::ImageConstPtr& img, const CameraInfoConstPtr& dep_info,
			const CameraInfoConstPtr& img_info )
{
	if(counter == 0)
	{
		writeb.write("/camera/depth/image", dep->header.stamp, dep);
		writeb.write("/camera/rgb/image_color", img->header.stamp, img);
		writeb.write("/camera/depth/camera_info", dep_info->header.stamp, dep_info);
		writeb.write("/camera/rgb/camera_info", img_info->header.stamp, img_info);
	}	

	counter ++;
	if(counter == nth)
		counter = 0;
}

int main( int argc, char** argv )
{
	//this only so we can get rostime
	ros::init(argc, argv, "but_kinect_recorder");
	ros::NodeHandle n;

	if( argc < 4 )
	{
		cout << "please specify parameters correctly - recorder <bagfile name> <rec laser - 1/0> <record every Nth frame>" << endl;
		return 0;
	}

	cout << "Recording to " << argv[1] << endl;
	
	nth = atoi(argv[3]);
	cout << "Recording every " << nth << "th frame" << endl;

	bool rec_laser = atoi(argv[2]);
	if( rec_laser )
		cout << "Laser recording too" << endl;

	writeb.open( argv[1], bagmode::Write );

	ros::Rate rate(50);
	
	message_filters::Subscriber<Image> depth_sub(n, "/camera/depth/image", 10);
	message_filters::Subscriber<Image> image_sub(n, "/camera/rgb/image_color", 10);
	message_filters::Subscriber<CameraInfo> depth_info_sub(n, "/camera/depth/camera_info", 10);
	message_filters::Subscriber<CameraInfo> image_info_sub(n, "/camera/rgb/camera_info", 10);

	typedef sync_policies::ApproximateTime<Image, Image, CameraInfo, CameraInfo> MySyncPolicy;
	Synchronizer<MySyncPolicy> sync(MySyncPolicy(10), depth_sub, image_sub, depth_info_sub, image_info_sub);
	sync.registerCallback(boost::bind(&kinectCallback, _1, _2, _3, _4));

	if( rec_laser )
		ros::Subscriber scan_sub = n.subscribe("/scan", 10, scanCallback);

	ros::Subscriber tf_sub = n.subscribe("/tf", 10, tfCallback);

	ros::Subscriber sync_sub = n.subscribe("/sync", 10, syncCallback);

	TerminalIO term;
    	bool bPaused = true, bStep = false;
    	bool finish = false;

	//ref = ros::Time::now();
	cout << "PAUSED, press SPACE to begin recording" << endl;

	while(!finish)
	{

		// test if any key was pressed
		int key = term.readCharFromStdin();
		switch( key )
		{
			case ' ':
				bPaused = !bPaused;
				if(!bPaused)
					cout << "recording..." << endl;
				else
					cout << "paused..." << endl;
				break;

			case 's':
				bPaused = true;
				bStep = true;
				break;
			case 'x':
				finish = true;
				break;
		}

		if( !bPaused || bStep )
		{
			bStep = false;

			ros::spinOnce();
			rate.sleep();
		}
	}
	
	writeb.close();
}

/*****************************************************************************
 *
 */

void TerminalIO::setup()
{
    const int fd = fileno(stdin);
    termios flags;
    tcgetattr(fd, &m_orig_flags);
    flags = m_orig_flags;
    flags.c_lflag &= ~ICANON;      // set raw (unset canonical modes)
    flags.c_cc[VMIN]  = 0;         // i.e. min 1 char for blocking, 0 chars for non-blocking
    flags.c_cc[VTIME] = 0;         // block if waiting for char
    tcsetattr(fd, TCSANOW, &flags);

    FD_ZERO(&m_stdin_fdset);
    FD_SET(fd, &m_stdin_fdset);
    m_maxfd = fd + 1;
}


void TerminalIO::restore()
{
    const int fd = fileno(stdin);
    tcsetattr(fd, TCSANOW, &m_orig_flags);
}


int TerminalIO::readCharFromStdin()
{
    fd_set testfd = m_stdin_fdset;

    timeval tv;
    tv.tv_sec  = 0;
    tv.tv_usec = 0;
    if( select(m_maxfd, &testfd, NULL, NULL, &tv) <= 0 )
    {
        return EOF;
    }

    return getc(stdin);
}
