/*
								+---------------------------------+
								|                                 |
								|   ***   Directory lookup  ***   |
								|                                 |
								|  Copyright   -tHE SWINe- 2006  |
								|                                 |
								|              Dir.h              |
								|                                 |
								+---------------------------------+
*/

/**
 *	@file Dir.h
 *	@author -tHE SWINe-
 *	@brief basic directory lookup and file info functions
 *
 *	@date 2007-07-03
 *
 *	added filename allocation, deallocation and copy functions to TFileInfo
 *
 *	added constructors to TFileInfo::TTime
 *
 *	added TFileInfo::b_valid so it's clear wheter call to CDirectory::Get_FirstFile
 *	or CDirectory::Get_NextFile functions failed or wheter there are no more files in
 *	the directory (as was identified by the same return value before)
 *
 *	merged source code of linux CDirectory and win32 CDirectory together as it shares
 *	substantial sections
 *
 *	created Dir.cpp
 *
 *	added CDirTraversal class
 *
 *	@date 2007-07-09
 *
 *	added TFileInfo copy-constructor TFileInfo(const TFileInfo &r_t_other)
 *	changed CDirTraversal's stack to queue so directory traversal is not reversed
 *
 *	@date 2008-02-21
 *
 *	fixed Win32 implementation of CDir::Get_FirstFile where it didn't detect
 *	failure properly and sometimes it could return invalid file data
 *
 *	@date 2008-02-27
 *
 *	fixend inconsistency between linux and windows implementation (in linux, TFileInfo used
 *	to contain bare filenames only, in windows it contained full paths. full paths are now
 *	used in both implementations). note this error made CDirTraversal fail under linux.
 *
 *	@date 2008-03-04
 *
 *	now using Integer.h header
 *
 *	@date 2008-08-08
 *
 *	added \#ifdef for windows 64
 *
 *	@date 2008-10-25
 *
 *	added CDirectory::path_Separator
 *
 *	@date 2008-11-21
 *
 *	added CDirTraversal::Traverse2, enabling file listener to interrupt directory traversal
 *
 *	added parameter b_recurse_subdirectories to both original CDirTraversal::Traverse() and
 *	new CDirTraversal::Traverse2() functions.
 *
 *	cleared-up CDirTraversal class documentation comment, regarding recursive traversal
 *	without recursion (ie. using explicit stack, placed on heap)
 *
 *	@date 2008-12-11
 *
 *	added TFileInfo::n_Size64() for greater conveniency
 *
 *	@date 2008-12-22
 *
 *	removed some g++ warnings
 *
 *	@date 2009-05-23
 *
 *	removed all instances of std::vector::reserve and replaced them by stl_ut::Reserve_*
 *
 *	@date 2009-10-20
 *
 *	fixed some warnings when compiling under VC 2005, implemented "Security
 *	Enhancements in the CRT" for VC 2008. compare against MyProjects_2009-10-19_
 *
 *	@date 2010-01-07
 *
 *	renamed TFileInfo::p_s_Filename() to TFileInfo::p_s_Path(), added TFileInfo::p_s_FileName()
 *	and TFileInfo::p_s_Extension()
 *
 *	added TFileInfo::GetInfo() which updates info about file from the filesystem
 *
 *	added TFileInfo constructor with filename, which enables TFileInfo being used
 *	much like File in java
 *
 *	added TFileInfo::b_exists, which changes behavior of CDirectory::Get_FirstFile()
 *		and CDirectory::Get_NextFile(), which used TFileInfo::b_valid to indicate whether
 *		there are any more files in the directory. access to TFileInfo::b_valid is now limited
 *		only trough TFileInfo::b_Valid() function, so the old incompatible code wouldn't
 *		compile.
 *
 *	added TFileInfo::n_mode
 *
 *	@date 2010-08-10
 *
 *	added code, handling correct construction of TFileInfo for root directories
 *	via constructor / GetInfo() under windows.
 *
 *	@note note creation / last access / last modification times
 *		are not available for root directories under windows
 *
 *	@date 2010-10-28
 *
 *	fixed minor memory leak, caused by not closing FindFirstFile() handles on windows systems.
 *	it didn't manifest as memory leak as memory got allocated on system side, it just made
 *	pagefile grow on frequent use.
 *
 *	@date 2010-10-29
 *
 *	Unified windows detection macro to "#if defined(_WIN32) || defined(_WIN64)".
 *
 */

#ifndef __DIR_INCLUDED
#define __DIR_INCLUDED

#include "NewFix.h"

#if defined(_WIN32) || defined (_WIN64)
#include <windows.h>
#include <string.h>
#include <io.h>
#else // _WIN32, _WIN64
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#endif // _WIN32, _WIN64
#include <time.h>
#include <string>

#include "Integer.h"
#include "StlUtils.h"

/**
 *	@def PRIsizeB
 *
 *	@brief simple macro for printing file sizes in user-readable format
 *
 *	This macro is used in conjunction with PRIsizeBparams.
 */
#define PRIsizeB "%.*f %s"

/**
 *	@def PRIsizeBparams
 *
 *	@brief simple macro for expanding size in bytes to number of fractional
 *		digits, value and unit
 *
 *	This is used with PRIsizeB for printing sizes. Usage is as follows:
 *
 *@code
 *	void Test()
 *	{
 *		printf("%d = " PRIsizeB "B\n", 256, PRIsizeBparams(256));
 *		printf("%d = " PRIsizeB "B\n", 1024 + 256, PRIsizeBparams(1024 + 256));
 *		printf("%d = " PRIsizeB "B\n", 1048576, PRIsizeBparams(1048576));
 *	}
 *
 *	output:
 *
 *	256 = 256 B
 *	1280 = 1.25 kB
 *	1048576 = 1.00 MB
 *@endcode
 *
 *	More extended version of this macro is PRIsizeBparamsExt.
 *
 *	@param[in] s is size in bytes
 */
#define PRIsizeBparams(s) (((s) < 1024)? 0 : 2), (((s) < 1024)? (s) : ((s) < 1048576)? \
	(s) / 1024.0 : ((s) < 1073741824)? (s) / 1048576.0 : ((s) < 1099511627776)? \
	(s) / 1073741824.0 : (s) / 1099511627776.0), (((s) < 1024)? "" : ((s) < 1048576)? "k" : \
	((s) < 1073741824)? "M" : ((s) < 1099511627776)? "G" : "T")

/**
 *	@def PRIsizeBparamsExt
 *
 *	@brief simple macro for expanding size in bytes to number of fractional
 *		digits, value and unit
 *
 *	This is used with PRIsizeB for printing sizes. Usage is as follows:
 *
 *@code
 *	void Test()
 *	{
 *		printf("%d = " PRIsizeB "B\n", 256, PRIsizeBparamsExt(256, 1, 2, false));
 *		printf("%d = " PRIsizeB "B\n", 256, PRIsizeBparamsExt(256, 0, 2, true));
 *		printf("%d = " PRIsizeB "B\n", 1048576, PRIsizeBparamsExt(1048576, 0, 3, false));
 *	}
 *
 *	output:
 *
 *	256 = 256.0 B
 *	256 = 0.25 kB
 *	1048576 = 1.000 MB
 *@endcode
 *
 *	More extended version of this macro is PRIsizeBparamsExt.
 *
 *	@param[in] s is size in bytes
 *	@param[in] byteDecNum is number of fractional digits when printing size in bytes
 *	@param[in] decNum is number of fractional digits when printing size not in bytes
 *	@param[in] forceKB is flag, controlling wheter to allow output in bytes (false),
 *		or to force output in kilobytes (true); does not have any effect if s >= 1024
 */
#define PRIsizeBparamsExt(s,byteDecNum,decNum,forceKB) (((s) < 1024 && !forceKB)? \
	byteDecNum : decNum), (((s) < 1024 && !forceKB)? (s) : ((s) < 1048576)? (s) / 1024.0 : \
	((s) < 1073741824)? (s) / 1048576.0 : ((s) < 1099511627776)? (s) / 1073741824.0 : \
	(s) / 1099511627776.0), (((s) < 1024 && !forceKB)? "" : ((s) < 1048576)? "k" : \
	((s) < 1073741824)? "M" : ((s) < 1099511627776)? "G" : "T")

/**
 *	@brief simple file-info structure
 *
 *	Keeps memory for it's filename
 *
 *	@todo Rewrite this so filename is stored in std::string, get rid of unnecessary
 *		memory management code.
 */
struct TFileInfo {
private:
	bool b_valid; /**< are data inside this structure valid? */

public:
	/**
	 *	@brief determines wheter contents of this structure are valid
	 *
	 *	@return Returns true in case this structure contains valid
	 *		information about a file, otherwise returns false.
	 *
	 *	@note All file info's should be valid, even if the file doesn't exist
	 *		(b_exists is not set then). In case this function returns false,
	 *		it's most likely program error (not enough memory, ...).
	 */
	inline bool b_Valid() const
	{
		return b_valid;
	}

	bool b_exists; /**< does specified file / directory exist? @note in case file doesn't exist, value of the following fields are undefined */
	bool b_directory; /**< file / directory flag */

	std::string s_filename; /**< file name */

	/**
	 *	@brief access flag names
	 */
	enum {
		flag_Read = 1,							/**< general reading privilege */
		flag_Write = 2,							/**< general writing privilege */
		flag_Execute = 4,						/**< general execute privilege */

		flag_owner_Read = flag_Read,			/**< owner reading privilege */
		flag_owner_Write = flag_Write,			/**< owner writing privilege */
		flag_owner_Execute = flag_Execute,		/**< owner execute privilege */

		flag_group_Read = flag_Read << 3,		/**< group reading privilege */
		flag_group_Write = flag_Write << 3,		/**< group writing privilege */
		flag_group_Execute = flag_Execute << 3,	/**< group execute privilege */

		flag_other_Read = flag_Read << 6,		/**< other reading privilege */
		flag_other_Write = flag_Write << 6,		/**< other writing privilege */
		flag_other_Execute = flag_Execute << 6	/**< other execute privilege */
	};

	int n_mode; /**< combination of flag_[Read|Write] @note this field is only for files, it is undefined for directories */
	int n_flags; /**< access flags (combination of flag_[owner|group|other]_[Read|Write|Execute]) */
	uint32_t n_size_lo; /**< filesize - 32 low bits */
	uint32_t n_size_hi; /**< filesize - 32 high bits */

	/**
	 *	@brief gets file size
	 *
	 *	@return Returns 64-bit file size.
	 */
	inline uint64_t n_Size64() const
	{
		return (uint64_t(n_size_hi) << 32) | n_size_lo;
	}

	/**
	 *	@brief very simple time structure (unification for linux / windows)
	 *
	 *	@note This was derived from windows' SYSTEMTIME structure, see it's
	 *		documentation on MSDN for more details.
	 *
	 *	@todo Supply conversions back to struct FILETIME / struct tm.
	 */
	struct TTime {
		short n_month;	/**< month index (1 - 12) */
		short n_day;	/**< month day index (1 - 31) */
		short n_year;	/**< year (1601 - 30827) */
		short n_hour;	/**< hour (0 - 23) */
		short n_minute;	/**< minute (0 - 59) */
		short n_second;	/**< second (0 - 59) */

		/**
		 *	@brief default constructor
		 *
		 *	Sets all members to -1.
		 */
		inline TTime()
			:n_month(-1), n_day(-1), n_year(-1), n_hour(-1), n_minute(-1), n_second(-1)
		{}

#if defined(_WIN32) || defined (_WIN64)
		/**
		 *	@brief (win32) conversion constructor
		 *
		 *	Takes <tt>struct SYSTEMTIME</tt> as input, always succeeds.
		 *
		 *	@param[in] r_t_time is input time
		 */
		TTime(const SYSTEMTIME &r_t_time);

		/**
		 *	@brief (win32) conversion constructor
		 *
		 *	Takes <tt>struct FILETIME</tt> as input.
		 *
		 *	@param[in] r_t_time is input time
		 *
		 *	@note FILETIME needs to be converted to SYSTEMTIME first, in case conversion
		 *		fails, all members are set to -1.
		 */
		TTime(const FILETIME &r_t_time);
#else // _WIN32, _WIN64
		/**
		 *	@brief (unix) conversion constructor
		 *
		 *	Takes <tt>struct tm</tt> as input, always succeeds
		 *
		 *	@param[in] p_time is input time
		 */
		TTime(const tm *p_time);
#endif // _WIN32, _WIN64

		/**
		 *	@brief equality operator
		 *
		 *	@param[in] r_t_time is time to be compared to
		 *
		 *	@return Returns true in case this is equal to r_t_time, otherwise false.
		 *
		 *	@todo Write operator < as well. Implement the other operators as well? Or does STL do it?
		 */
		bool operator ==(const TTime &r_t_time) const;
	};

	/**
	 *	@brief file time type names
	 */
	enum {
		time_Creation = 0,	/**< time of file creation */
		time_LastAccess,	/**< time of last access to a file */
		time_LastWrite		/**< time of last modification */
	};

	TTime p_time[3]; /**< file time (use time_[Creation|LastAccess|LastWrite] as index) */

	/**
	 *	@brief default constructor
	 *
	 *	Doesn't initialize anything but file name, creates invalid file info.
	 *
	 *	@todo Write constructor with file name specification (get file info for specific file,
	 *		eg. for "c:\blah.txt", so TFileInfo could be used without CDirectory)
	 */
	inline TFileInfo()
		:b_valid(false)
	{}

	/**
	 *	@brief default constructor
	 *
	 *	Copies filename and fills the structure with information about the file. In case the file
	 *	doesn't exist, b_valid is set, but b_exists isn't.
	 */
	TFileInfo(const char *p_s_filename);

	/**
	 *	@brief copy constructor
	 *
	 *	Copies information about file r_other to this. Note there may not be enough
	 *		memory for file name, check if b_valid is set.
	 *
	 *	@param r_other is file info to be copied
	 */
	inline TFileInfo(const TFileInfo &r_other)
	{
		*this = r_other; // copy values
	}

	/**
	 *	@brief gets file name
	 *
	 *	@return Returns filename as null-terminated string
	 *		  or 0 in case no filename was specified yet.
	 *
	 *	@todo Figure-out the best way to support unicode here.
	 */
	const char *p_s_FileName() const;

	/**
	 *	@brief gets file extension
	 *
	 *	@return Returns file extension as null-terminated string,
	 *		or an empty string in case there's no extension.
	 *
	 *	@todo Figure-out the best way to support unicode here.
	 */
	const char *p_s_Extension() const;

	/**
	 *	@brief gets file path
	 *
	 *	@return Returns file path (including filename) as null-terminated string,
	 *		  or 0 in case no filename was specified yet.
	 *
	 *	@todo Figure-out the best way to support unicode here.
	 */
	inline const char *p_s_Path() const
	{
		return s_filename.c_str();
	}

	/**
	 *	@brief marks the file as non-existent (but valid)
	 */
	inline void SetNoFile()
	{
		b_valid = true;
		b_exists = false;
	}

	/**
	 *	@brief marks the file information as invalid (to mark error)
	 */
	inline void SetInvalid()
	{
		b_valid = false;
	}

	/**
	 *	@brief sets new filename
	 *
	 *	Sets filename to p_s_new_filename.
	 *
	 *	@param[in] p_s_new_filename is null-terminated string, containing new file name
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note Note p_s_new_filename can be 0 (sets this filename to 0 as well).
	 */
	bool SetFilename(const char *p_s_new_filename);

	/**
	 *	@brief copy-operator
	 *
	 *	Copies contents of r_t_file_info to this.
	 *
	 *	@param[in] r_t_file_info is file info to be copied
	 *
	 *	@return Returns true on success, false in case there
	 *		was not enough memory to hold filename string.
	 */
	bool operator =(const TFileInfo &r_t_file_info);

	/**
	 *	@brief generates name for temporary file
	 *
	 *	Creates name for temporary file in default system temp path.
	 *		If the function succeeds, it creates an empty file, so it
	 *		may be considered thread-safe (no two threads can create the same file).
	 *
	 *	@param[out] r_s_temp_file_name will contain unique temporary file name
	 *	@param[in] p_s_app_id is short application identifier,
	 *		which will be part of temp filename (may not be empty)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note p_s_app_id might get truncated if it's too long (3 characters is usual length)
	 */
	static bool Get_TempFileName(std::string &r_s_temp_file_name, const char *p_s_app_id = "tmp");

#if defined(_WIN32) || defined (_WIN64)
	/**
	 *	@brief updates information about the file
	 *
	 *	Calls OS api to get information about the file (specified by p_s_Path()),
	 *	and fills member variables of the structure.
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note Success is even if the file doesn't exist.
	 */
	bool GetInfo();

	/**
	 *	@brief updates information about the file
	 *
	 *	Calls OS api to get information about the file (specified by p_s_Path()),
	 *	and fills member variables of the structure. This overload is optimized for
	 *	getting file information from existing WIN32_FIND_DATA structure (when iterating
	 *	trough files in a directory).
	 *
	 *	@param[in] r_t_find_data contains information about the file (not filename)
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note Success is even if the file doesn't exist.
	 */
	bool GetInfo(const WIN32_FIND_DATAA &r_t_find_data);
#else // _WIN32, _WIN64
	/**
	 *	@brief updates information about the file
	 *
	 *	Calls OS api to get information about the file (specified by p_s_Path()),
	 *	and fills member variables of the structure.
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note Success is even if the file doesn't exist.
	 */
	bool GetInfo();
#endif // _WIN32, _WIN64
};

/**
 *	@brief simple directory class
 *
 *	This class allows for very basic access to directory's files (sub-directories).
 */
class CDirectory {
public:
	/**
	 *	@brief path separator character
	 */
	enum {
#if defined(_WIN32) || defined (_WIN64)
		path_Separator = '\\'	/**< windows path separator character */
#else // _WIN32, _WIN64
		path_Separator = '/'	/**< unix path separator character */
#endif // _WIN32, _WIN64
	};

private:
	char *m_p_s_dir_name;

#if defined(_WIN32) || defined (_WIN64)
	HANDLE m_h_prev_file;
#else // _WIN32, _WIN64
    DIR *m_p_dir;
#endif // _WIN32, _WIN64

public:
	/**
	 *	@brief default constructor
	 *
	 *	Creates handle to directory with address p_s_dir_name. Call p_s_Name() to find
	 *		out wheter there was enough memory for string copy, otherwise this object
	 *		can't be used to find files (sub-directories).
	 *
	 *	@param[in] p_s_dir_name is directory name (such as "c:\blah" or
	 *		"c:\blah\" or ".\blah" or ".\blah\" or "blah" or "blah\")
	 *
	 *	@note p_s_dir_name is copied and can be freed right after constructor returns
	 *
	 *	@todo Rewrite this so name is stored in std::string, get rid of unnecessary
	 *		memory management code.
	 */
	CDirectory(const char *p_s_dir_name);

	/**
	 *	@brief destructor
	 */
	~CDirectory();

	/**
	 *	@brief gets directory name
	 *
	 *	@return Returns directory name as passed to constructor.
	 *	@return Returns 0 in case there was not enough memory for string copy.
	 */
	inline const char *p_s_Name() const
	{
		return m_p_s_dir_name;
	}

	/**
	 *	@brief finds a file
	 *
	 *	Gets first file in the directory, copies it's data to r_t_file. In case the
	 *		directory doesn't have any files, r_t_file.b_exists is set to false.
	 *
	 *	@param[out] r_t_file on return contains information about found file
	 *
	 *	@return Returns true on success (even if directory is empty - that is marked by
	 *		r_t_file.b_valid), false on failure (possibly inaccesible / non-existant
	 *		directory).
	 *
	 *	@note r_t_file.b_valid is always true, even if the file doesn't exist (aparts
	 *		from when the function fails).
	 */
	bool Get_FirstFile(TFileInfo &r_t_file);

	/**
	 *	@brief finds a file
	 *
	 *	Gets next file in the directory, copies it's data to r_t_file. In case the
	 *		directory doesn't have any more files, r_t_file.b_exists is set to false.
	 *	In case Get_FirstFile() was not called prior calling Get_NextFile, it's called
	 *		automatically.
	 *
	 *	@param[out] r_t_file on return contains information about found file 
	 *
	 *	@return Returns true on success (even if directory does not contain (more)
	 *		files - that is marked by r_t_file.b_valid), false on failure (possibly
	 *		inaccesible / non-existant directory).
	 *
	 *	@note In order to get file list for this folder again, Get_FirstFile()
	 *		must be called (r_t_file contents are only written to, not read from).
	 *	@note r_t_file.b_valid is always true, even if the file doesn't exist (aparts
	 *		from when the function fails).
	 */
	bool Get_NextFile(TFileInfo &r_t_file);

private:
#if defined(_WIN32) || defined (_WIN64)
	bool GetFileInfo(const WIN32_FIND_DATAA &r_t_find_data, TFileInfo &r_t_file);
#endif // _WIN32, _WIN64
};

/**
 *	@brief simple recursive directory traversal
 *
 *	Implemented in non-recursive manner (explicit stack), so it wouldn't cause
 *		stack overflow when traversing complex directory trees. To show directory
 *		listing, use:
 *@code
 *	void OnFile(const TFileInfo &r_t_file)
 *	{
 *		printf((r_t_file.b_directory)? "[%s]\n" : "%s\n", r_t_file.p_s_Path());
 *		// directories printed in brackets
 *	}
 *
 *	int main(int n_arg_num, const char **p_arg_list)
 *	{
 *		_ASSERTE(n_arg_num == 2);
 *		return !CDirTraversal::Traverse(p_arg_list[1], OnFile);
 *		// traverse directory from commandline
 *	}
 *@endcode
 */
class CDirTraversal {
public:
	/**
	 *	@brief traverses directory
	 *
	 *	Traverses directory p_s_dir_name, all files found are passed to r_file_listener.
	 *		If b_recurse_subdirectories is set, subdirectories are traversed as well.
	 *		All files and subdirectories in a directory are passed to r_file_listener before
	 *		traversing into any of subdirectories (if b_recurse_subdirectories is set,
	 *		that is).
	 *
	 *	@param[in] p_s_dir_name is name of directory to be traversed
	 *	@param[in] r_file_listener is CFileListener object; CFileListener is file listener
	 *		functor, must implement void operator () with a single <tt>const TFileInfo &</tt>
	 *		parameter.
	 *	@param[in] b_recurse_subdirectories is directory recursion flag
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note This doesn't give file listener option to early terminate the traversal.
	 *		That is implemented in Traverse2().
	 *	@note Files, passed to file listener are always valid (TFileInfo::b_valid is true).
	 */
	template <class CFileListener>
	static bool Traverse(const char *p_s_dir_name,
		CFileListener &r_file_listener, bool b_recurse_subdirectories = true)
	{
		CDirectory *p_dir_info;
		if(!(p_dir_info = new(std::nothrow) CDirectory(p_s_dir_name)))
			return false;
		// prepare the first directory handle

		std::vector<CDirectory*> m_dir_queue;
		stl_ut::Reserve_N(m_dir_queue, 16);
		if(!m_dir_queue.capacity())
			return false;
		m_dir_queue.push_back(p_dir_info);
		// put it into the queue

		while(!m_dir_queue.empty()) {
			CDirectory *p_dir = m_dir_queue.back();
			m_dir_queue.erase(m_dir_queue.end() - 1);
			// fetch a single record from top of the queue

			unsigned long n_pos = m_dir_queue.size();

			TFileInfo t_file_info;
			if(!p_dir->Get_FirstFile(t_file_info)) {
				std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
				delete p_dir;
				return false; // t_odo - free the pair data
			}
			while(/*t_file_info.b_valid &&*/ t_file_info.b_exists) {
				_ASSERTE(t_file_info.b_Valid());
				r_file_listener((const TFileInfo&)t_file_info);
				// pass file data to listener
			
				if(b_recurse_subdirectories && t_file_info.b_directory) {
					CDirectory *p_subdir_info;
					if(!stl_ut::Reserve_1More(m_dir_queue) ||
					   !(p_subdir_info = new(std::nothrow) CDirectory(t_file_info.p_s_Path()))) {
						std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
						delete p_dir;
						return false;
					}
					// make sure there's enough space in the queue,
					// prepare the subdirectory handle

					m_dir_queue.insert(m_dir_queue.begin() + n_pos, p_subdir_info);
				}
				// add the directory to the list to recurse

				if(!p_dir->Get_NextFile(t_file_info)) {
					std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
					delete p_dir;
					return false;
				}
				// get next file info
			}
			delete p_dir;
		}

		return true;
	}

	/**
	 *	@brief traverses directory
	 *
	 *	Traverses directory p_s_dir_name, all files found are passed to r_file_listener.
	 *		If b_recurse_subdirectories is set, subdirectories are traversed as well.
	 *		All files and subdirectories in a directory are passed to r_file_listener before
	 *		traversing into any of subdirectories (if b_recurse_subdirectories is set,
	 *		that is).
	 *
	 *	@param[in] p_s_dir_name is name of directory to be traversed
	 *	@param[in] r_file_listener is CFileListener object; CFileListener is file listener
	 *		functor, must implement bool operator () with a single <tt>const TFileInfo &</tt>
	 *		parameter. In case it returns true, traversal continues, in case it returns false,
	 *		Traverse2() returns false immediately.
	 *	@param[in] b_recurse_subdirectories is directory recursion flag
	 *
	 *	@return Returns true on success, false on failure.
	 *
	 *	@note Files, passed to file listener are always valid (TFileInfo::b_valid is true).
	 */
	template <class CFileListener>
	static bool Traverse2(const char *p_s_dir_name,
		CFileListener &r_file_listener, bool b_recurse_subdirectories = true)
	{
		CDirectory *p_dir_info;
		if(!(p_dir_info = new(std::nothrow) CDirectory(p_s_dir_name)))
			return false;
		// prepare the first directory handle

		std::vector<CDirectory*> m_dir_queue;
		stl_ut::Reserve_N(m_dir_queue, 16);
		if(!m_dir_queue.capacity())
			return false;
		m_dir_queue.push_back(p_dir_info);
		// put it into the queue

		while(!m_dir_queue.empty()) {
			CDirectory *p_dir = m_dir_queue.back();
			m_dir_queue.erase(m_dir_queue.end() - 1);
			// fetch a single record from top of the queue

			size_t n_pos = m_dir_queue.size();

			TFileInfo t_file_info;
			if(!p_dir->Get_FirstFile(t_file_info)) {
				std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
				delete p_dir;
				return false; // t_odo - free the pair data
			}
			while(/*t_file_info.b_valid &&*/ t_file_info.b_exists) {
				_ASSERTE(t_file_info.b_Valid());
				if(!r_file_listener((const TFileInfo&)t_file_info)) {
					std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
					delete p_dir;
					return false;
				}
				// pass file data to listener
			
				if(b_recurse_subdirectories && t_file_info.b_directory) {
					CDirectory *p_subdir_info;
					if(!stl_ut::Reserve_1More(m_dir_queue) ||
					   !(p_subdir_info = new(std::nothrow) CDirectory(t_file_info.p_s_Path()))) {
						std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
						delete p_dir;
						return false;
					}
					// make sure there's enough space in the queue,
					// prepare the subdirectory handle

					m_dir_queue.insert(m_dir_queue.begin() + n_pos, p_subdir_info);
				}
				// add the directory to the list to recurse

				if(!p_dir->Get_NextFile(t_file_info)) {
					std::for_each(m_dir_queue.begin(), m_dir_queue.end(), DeleteDir);
					delete p_dir;
					return false;
				}
				// get next file info
			}
			delete p_dir;
		}

		return true;
	}

private:
	static inline void DeleteDir(CDirectory *p_dir)
	{
		delete p_dir;
	}
};

#endif //__DIR_INCLUDED
