#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include "platform.h"
#include "test.h"


#ifdef STATIC_DATA
const extern size_t binfile_s1;
const extern size_t binfile_s2;
const extern size_t binfile_s3;
extern char binfile_1[];
extern char binfile_2[];
extern char binfile_3[];
#ifdef FILTER_CAN_REMOVE
const extern size_t binfile_s4;
extern char binfile_4[];
#endif
#endif 

#define skipSpace(cfg) { while(isspace(*cfg)) cfg++; }
#define nextWord(cfg)  { while(!isspace(*cfg)) cfg++; while(isspace(*cfg)) cfg++; }

// testing Bloom filter
struct filter_t filter;

// the length of key to insert/query
unsigned int key_length;

char * config;
int config_length;

// data to be copied from files to memory...
uint32_t *input1, *input2, *input3;
// ... and their length
long int length1, length2, length3;

/*
 * test_config loads configuration from text file
 * and initializes Bloom filter structure.
 */
void test_config() {

  unsigned int hash_blocks;
  uint32_t *hash_seeds;
  unsigned int field_elements;
	
  char * cfg = config;
  skipSpace(cfg);
  if(sscanf(cfg, "hash_blocks=%d", &hash_blocks) != 1) exit(2);

  hash_seeds = malloc(sizeof(uint32_t)*hash_blocks);
  if (hash_seeds == NULL) exit(1);

  for (unsigned int i = 0; i < hash_blocks; i++) {
    nextWord(cfg);
    if ((sscanf(cfg, "%X", &hash_seeds[i])) != 1) exit(5);
  }
  nextWord(cfg);
  if (sscanf(cfg, "field_elements=%d", &field_elements) != 1) exit(3);

  nextWord(cfg);
  if (sscanf(cfg, "key_length=%d", &key_length) != 1) exit(4);

  if (!filter_init(&filter, hash_blocks, hash_seeds, field_elements))
    exit(5);
}

/*
 * test_load() copies data from input files to allocated memory buffers.
 */
void test_load(int argc, char *argv[]) {

#ifdef STATIC_DATA
  config = binfile_1;
  config_length = binfile_s1;

  input1 = (uint32_t*) binfile_2;
  length1 = binfile_s2;
  input2 = (uint32_t*) binfile_3;
  length2 = binfile_s3;
#ifdef FILTER_CAN_REMOVE
  input3 = (uint32_t*) binfile_4;
  length3 = binfile_s4;
#endif

#else
  if(argc != ARGC_REQ) {
    fprintf(stderr, "Usage: %s config_file items_file query_file%s\n", argv[0], (ARGC_REQ == 5) ? " removefile":"");
    exit(EXIT_FAILURE);
  }

  FILE *input = fopen(argv[1], "r");
  if (input==NULL) exit(4);
  fseek(input, 0, SEEK_END);
  config_length = ftell(input);
  config = (char*) malloc(config_length);
  fseek(input, 0, SEEK_SET);
  fread(config, 1, config_length, input);
  fclose(input);

  input = fopen(argv[2], "r");
  if (input==NULL) exit(4);
  fseek(input, 0, SEEK_END);
  length1 = ftell(input);
  input1 = (uint32_t*) malloc(length1);
  fseek(input, 0, SEEK_SET);
  fread(input1, 1, length1, input);
  fclose(input);

  input = fopen(argv[3], "r");
  if (input==NULL) exit(4);
  fseek(input, 0, SEEK_END);
  length2 = ftell(input);
  input2 = (uint32_t*) malloc(length2);
  fseek(input, 0, SEEK_SET);
  fread(input2, 1, length2, input);
  fclose(input);

#ifdef FILTER_CAN_REMOVE
  input = fopen(argv[4], "r");
  if (input==NULL) exit(4);
  fseek(input, 0, SEEK_END);
  length3 = ftell(input);
  input3 = (uint32_t*) malloc(length3);
  fseek(input, 0, SEEK_SET);
  fread(input3, 1, length3, input);
  fclose(input);
#endif // FILTER_CAN_REMOVE
#endif // STATIC_DATA

  open_output();
}

/*
 * test_run does the test.
 * Data are read from memory buffers.
 */
void test_run() {

  uint32_t out;

  timer_start();

  // inserting items into the Bloom filter

  for (uint32_t *i = input1;
       i < (input1+length1/sizeof(uint32_t));
       i += key_length) {
	 trace();
    filter_add(&filter, i, key_length);
  }

  timer_print();
  timer_start();

  // querying the Bloom filter

  for (uint32_t *i = input2;
       i < (input2+length2/sizeof(uint32_t));
       i += key_length) {
	  trace();
      out = filter_query(&filter, i, key_length);
	  print_output(out);
  }

  timer_print();
  timer_start();

#ifdef FILTER_CAN_REMOVE
  // removing items from the Bloom filter

  for (uint32_t *i = input3;
       i < (input3+length3/sizeof(uint32_t));
       i += key_length) {
	 trace();
    filter_remove(&filter, i, key_length);
  }

  timer_print();
  timer_start();

  // querying the Bloom filter

  for (uint32_t *i = input2;
       i < (input2+length2/sizeof(uint32_t));
       i += key_length) {
	  trace();
    // output2[output2_length++] =
      out = filter_query(&filter, i, key_length);
	  print_output(out);
  }

  timer_print();
#endif // FILTER_CAN_REMOVE

}


/*
 * test_clear deallocates resources.
 */
void test_clear() {

#ifndef STATIC_DATA
#ifdef FILTER_CAN_REMOVE
  free(input3);
#endif
  free(input2);
  free(input1);
  free(config);
#endif
  free(filter.hash_seeds);

  filter_deinit(&filter);

  close_output();
}

int test_main(int argc, char *argv[]) {

  test_load(argc, argv);	// load input data
  test_config();			// load config
  test_run();				// test
  test_clear();				// clean up

  return 0;
}
