#include "jenkins.h"
#include "cbf4.h"
#include <stddef.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>

bool cbf4_init(struct cbf4_t *filter,
               unsigned int hash_functions,
               uint32_t *hash_seeds,
               unsigned int field_elements) {

  filter->hash_functions = hash_functions;
  filter->hash_seeds = hash_seeds;

  filter->field_elements = field_elements;
  /* allocate & clear 4-bit counters */
  filter->field = (unsigned char*) malloc(field_elements/2);
  if (filter->field == NULL)
    return false;
  memset(filter->field, 0, field_elements/2);

  return true;

}

void cbf4_deinit(struct cbf4_t *filter) {

  // deallocate counters
  free(filter->field);

}

/* inline */
void cbf4_add(struct cbf4_t *filter,
              const uint32_t *key,
              size_t key_length) {

  for (unsigned int i = 0; i < filter->hash_functions; i++) {
    // get the index to the bit field
    unsigned int index = hash_jenkins(key, key_length, filter->hash_seeds[i])
      % filter->field_elements;
    // increment appropriate counter
    if (index&1) {
      // the counter in higher nibble
      // do not overflow
      //if ((filter->field[index/2]&0xf0) != 0xf0)
      if ((filter->field[index>>1]&0xf0) != 0xf0)
        //filter->field[index/2] += 0x10;
        filter->field[index>>1] += 0x10;
    } else {
      // the counter in lower nibble
      // do not overflow
      //if ((filter->field[index/2]&0x0f) != 0x0f)
      if ((filter->field[index>>1]&0x0f) != 0x0f)
        //filter->field[index/2] += 0x01;
        filter->field[index>>1] += 0x01;
    }
  }

}

/* inline */
void cbf4_remove(struct cbf4_t *filter,
                 const uint32_t *key,
                 size_t key_length) {

  for (unsigned int i = 0; i < filter->hash_functions; i++) {
    // get the index to the bit field
    unsigned int index = hash_jenkins(key, key_length, filter->hash_seeds[i])
      % filter->field_elements;
    // decrement appropriate counter
    if (index&1) {
      // the counter in higher nibble
      // do not underflow
      //if ((filter->field[index/2]&0xf0) != 0)
      if ((filter->field[index>>1]&0xf0) != 0)
        //filter->field[index/2] -= 0x10;
        filter->field[index>>1] -= 0x10;
    } else {
      // the counter in lower nibble
      // do not underflow
      //if ((filter->field[index/2]&0x0f) != 0)
      if ((filter->field[index>>1]&0x0f) != 0)
        //filter->field[index/2] -= 0x01;
        filter->field[index>>1] -= 0x01;
    }
  }

}

/* inline */
bool cbf4_query(struct cbf4_t *filter,
                const uint32_t *key,
                size_t key_length) {

  for (unsigned int i = 0; i < filter->hash_functions; i++) {
    // get the index to the bit field
    unsigned int index = hash_jenkins(key, key_length, filter->hash_seeds[i])
      % filter->field_elements;
    // if any of indexed counters is zero, the element is not in the filter
    if (index&1) {
      // the counter in higher nibble
      //if ((filter->field[index/2]&0xf0) == 0)
      if ((filter->field[index>>1]&0xf0) == 0)
        return false;
    } else {
      // the counter in lower nibble
      //if ((filter->field[index/2]&0x0f) == 0)
      if ((filter->field[index>>1]&0x0f) == 0)
        return false;
    }
  }

  return true;

}
