library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity DFSM_LOGIC%$% is
   generic(
      SYMBOL_WIDTH      : integer := %$%;
      NFSM_INPUT_WIDTH  : integer := %$%;
      NFSM_OUTPUT_WIDTH : integer := %$%;
      STATE_WIDTH       : integer := %$%;
      MEMORY_ADDR_WIDTH : integer := %$%;
      RULES             : integer := %$%
   );
   port(
      -- common signals
      CLK            : in  std_logic;
      RESET          : in  std_logic;
      
      -- symbol input interface
      DATA           :  in std_logic_vector(SYMBOL_WIDTH - 1 downto 0);
      SOF            :  in std_logic;
      EOF            :  in std_logic;
      SRC_RDY        :  in std_logic;
      DST_RDY        : out std_logic;
      --SYMBOL         : in  std_logic_vector(SYMBOL_WIDTH - 1 downto 0);
      --SYMBOL_VLD     : in  std_logic;
      --SYMBOL_ACK     : out std_logic;
           
      -- input from NFSM
      INPUT_STATES   : in  std_logic_vector(NFSM_INPUT_WIDTH - 1 downto 0);
      
      -- output to NFSM
      OUTPUT_STATES  : out std_logic_vector(NFSM_OUTPUT_WIDTH -1 downto 0);
      
      -- output data interface
      BITMAP         : out std_logic_vector(RULES - 1 downto 0);
      VLD            : out std_logic;
      ACK            :  in std_logic
   );
end entity DFSM_LOGIC%$%;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of DFSM_LOGIC%$% is
   signal addr                   : std_logic_vector(MEMORY_ADDR_WIDTH - 1 downto 0));
   signal rq                     : std_logic;
   signal n                      : std_logic;
   signal f                      : std_logic;
   signal v                      : std_logic;
   signal target                 : std_logic_vector(STATE_WIDTH - 1 downto 0));
   signal symbol                 : std_logic_vector(SYMBOL_WIDTH - 1 downto 0));
   signal nfsm_state_binary      : std_logic_vector(log2(NFSM_INPUT_WIDTH) - 1 downto 0);
   signal nfsm_state_binary_vld  : std_logic;
   signal input_state_muxed      : std_logic_vector(STATE_WIDTH - 1 downto 0);
   signal output_state           : std_logic_vector(STATE_WIDTH - 1 downto 0);
   signal output_state_vld       : std_logic;
   signal reg_symbol             : std_logic_vector(SYMBOL_WIDTH - 1 downto 0);
   signal move_vld               : std_logic;
   signal target                 : std_logic;
   signal reg_target             : std_logic;
   signal output                 : std_logic_vector(NFSM_OUTPUT_WIDTH -1 downto 0);
   signal bitmap_in              : std_logic_vector(RULES - 1 downto 0);
   signal local_reset            : std_logic;
   signal we                     : std_logic;
   signal rdy                    : std_logic;
   signal vld_internal           : std_logic;
   signal set                    : std_logic;
   %$%   
begin

    local_reset <= RESET or ACK;
    we <= SRC_RDY and rdy;
    DST_RDY <= rdy;
    VLD <= vld_internal;
    set <= SRC_RDY and EOF and rdy;
    rdy <= not vld_internal;
    
    end_reg: process(CLK)
    begin
        if (CLK'event and CLK = '1') then
            if (local_reset = '1') then
                vld_internal <= '0';
            else
                if set = '1' then
                    vld_internal <= '1';
                end if;
            end if;
        end if;
    end process end_reg;
  
   one_hot_to_binary: entity work.ENCODER%$%
   port map(
      INPUT   => INPUT_STATES,
      OUTPUT  => nfsm_state_binary,
      VLD     => nfsm_state_binary_vld
   );
     
   mux_in_states: process(nfsm_state_binary, nfsm_state_binary_vld, output_state, output_state_vld)
   begin
      if (nfsm_state_binary_vld = '1' and output_state_vld = '0') then
         input_state_muxed <= nfsm_state_binary;
      else
         input_state_muxed <= output_state;
      end if;
   end process mux_in_states;
   
   memory_i: entity work.DFSM_MEMORY%$%
   port map(
      CLK         => CLK,
      RESET       => RESET,
      MEMORY_ADDR => addr,
      MEMORY_RQ   => rq,
      TARGET      => target,
      SYMBOL      => symbol,
      N           => n,
      F           => f,
      V           => v
   );
   
   addr(SYMBOL_WIDTH - 1 downto 0) <= input_state_muxed(SYMBOL_WIDTH - 1 downto 0) xor DATA;
   addr(STATE_WIDTH - 1 downto SYMBOL_WIDTH) <= input_state_muxed(STATE_WIDTH - 1 downto SYMBOL_WIDTH);
   rq   <= we and output_state_vld;
   
   symbol_reg: process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         reg_symbol <= DATA;
      end if;
   end process symbol_reg;
   
   cmp_symbol: process(reg_symbol, MEMORY_DATA)
   begin
      if (reg_symbol = symbol) then
         move_vld <= '1';
      else
         move_vld <= '0';
      end if;
   end process;
      
   output_state_vld <= not n and not f and v and move_vld;
   
   OUTPUT_STATE
   decoder_i: entity work,DECODER%$%
   port map(
      INPUT => target,
      OUTPUT => output
   );
   
   gen_and: for i in 0 to NFSM_OUTPUT_WIDTH - 1 generate
      OUTPUT_STATES(i) <= output(i) and n;
   end generate gen_and; 
   
   output_state <= target;
   
   bitmap: entity work.FINAL_BITMAP
   generic map(
      DATA_WIDTH => RULES
   )
   port map(
      CLK    => CLK,
      RESET  => local_reset,
      SET    => bitmap_in,
      BITMAP => BITMAP
   );
   
   %$%
end architecture full;