#! /usr/bin/env python2.6
###############################################################################
#  ptrn_test.py: Module for PATTERN MATCH - test module
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vlastimil Kosar <ikosar@fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$


import unittest
import nfa_data
import nfa_parser
import sym_char
import b_symbol
import sym_char_class
import msfm_parser
import b_dfa
import b_state
import b_nfa
import copy
import b_automaton

class BasicTests(unittest.TestCase):
    """Tests Basic properties of the pattern matching part of the library"""
    
    def testLoadSave(self):
        """Tries to save and load empty object of nfa_data"""
        failed = 0
        try:
            Test = nfa_data.nfa_data()
            Test.SaveToFile("Test")
            Test.LoadFromFile("Test") 
        except Exception:
            failed = 1
        self.assert_(failed == 0)

    def testParser(self):
        """Tries to initialize Parser object"""
        Parser = nfa_parser.nfa_parser()

    def testSaveLoadData(self):
        """Tries to save and load nfa_data containing automata from test.msf"""
        pass
    
    def testShowEmpty(self):
        """ Tries to graph out empty nfa_data"""
        failed = 0
        try:
            Test = nfa_data.nfa_data()
            self.assert_(Test.Show("test_graph.dot") == False)
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testShow(self):
        """ Tries to graph out nfa_data"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            Test.set_text("/abc[d-f]/")
            nfa = Test.get_nfa()
            self.assert_(nfa.Show("test_graph.dot") == True)
        except None:#Exception:
            failed = 1
        self.assert_(failed == 0)
    
    def testBSymCharCreate(self):
        """Tries to create object of type b_Sym_char"""
        failed = 0
        try:
            Test = sym_char.b_Sym_char("a", "a")
        except Exception:
            failed = 1
        self.assert_(failed == 0)
    
    def testBSymCharAccept(self):
        """Tries to accept char from text string"""
        failed = 0
        try:
            Test = sym_char.b_Sym_char("a", "a")
            try:
                if Test.accept("abcd") != "bcd":
                    failed = 1
            except b_symbol.accept_exception:
                failed = 1
            
            try:
                if Test.accept("bbcd") != "bbcd":
                    failed = 1
            except b_symbol.accept_exception:
                failed = 0
                
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testBSymCharCollision(self):
        """Tries to compute collision of char symbol with set of symbols"""
        failed = 0
        try:
            Test = sym_char.b_Sym_char("a", "a")
            Test2 = sym_char.b_Sym_char("a", "a")
            Test3 = sym_char.b_Sym_char("b", "b")
            Test4 = sym_char.b_Sym_char("c", "c")
            set1 = set([Test2, Test3])
            set2 = set([Test3, Test4])
            if Test.collision(set1) == False:
                failed = 1
            if Test.collision(set2) == True:
                failed = 1
        except None:
                failed = 1
        self.assert_(failed == 0)
        
    def testBSymCharClassAccept(self):
        """Tries to accept char class from text string"""
        failed = 0
        try:
            Test = sym_char_class.b_Sym_char_class("a", set(["a", "b"]))
            try:
                if Test.accept("abcd") != "bcd":
                    failed = 1
                if Test.accept("bcd") != "cd":
                    failed = 1
            except b_symbol.accept_exception:
                failed = 1
            
            try:
                if Test.accept("cbcd") != "cbcd":
                    failed = 1
            except b_symbol.accept_exception:
                failed = 0
                
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testBSymCharClassCollision(self):
        """Tries to compute collision of char class symbol with set of symbols"""
        failed = 0
        try:
            Test = sym_char_class.b_Sym_char_class("chc1" ,set(["a", "b"]))
            Test2 = sym_char_class.b_Sym_char_class("chc2" ,set(["d", "b"]))
            Test3 = sym_char_class.b_Sym_char_class("chc3" ,set(["d", "c"]))
            set1 = set([Test2])
            set2 = set([Test3])
            if Test.collision(set1) == False:
                failed = 1
            if Test.collision(set2) == True:
                failed = 1
        except None:
                failed = 1
        self.assert_(failed == 0)
        
    def testBSymCharClassCreate(self):
        """Tries to create object of type b_Sym_char_class"""
        failed = 0
        try:
            Test = sym_char_class.b_Sym_char_class("chc1", set(["a", "b"]))
        except Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserCreate(self):
        """Tries to create object of type msfm_parser"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
        except Exception:
            failed = 1
        self.assert_(failed == 0)
    
    def testMsfmParserSetText(self):
        """Tries to set text to object of type msfm_parser"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            Test.set_text("/abc[d-f]/")
        except Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserLoadFile(self):
        """Tries to load file to object of type msfm_parser"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            Test.load_file("msfm_test_file.txt")
        except Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserSetMultilineText(self):
        """Tries to set multiline text to object of type msfm_parser"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            Test.set_text("/abc[d-f]/\n/abc/")
        except Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserParse(self):
        """Tries to parse RE"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            
            Test.set_text("/abc[d-f]/")
            
            self.assert_(Test.get_nfa() != None)
                           
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserParseNonExisting(self):
        """Tries to parse RE while __text is None"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            
            if (Test.get_nfa() != None):
                failed = 1
                           
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserParseNextLine(self):
        """Tries to test behavior of msfm_parser method next_line"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            
            if (Test.next_line() != False):
                failed = 1
                
            Test.set_text("/abc[d-f]/")
            if (Test.next_line() != False):
                failed = 1
            
            Test.set_text("/abc[d-f]/\n/abc/")
            if (Test.next_line() != True):
                failed = 1
                           
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserParseMove2Line(self):
        """Tries to test behavior of msfm_parser method move_to_line"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            
            self.assert_(Test.move_to_line(0) == False)
            #if (Test.move_to_line(0) != False):
            #    failed = 1
                
            Test.set_text("/abc[d-f]/")
            self.assert_(Test.move_to_line(1) == False)
            #if (Test.move_to_line(1) != False):
            #    failed = 1
            
            Test.set_text("/abc[d-f]/\n/abc/")
            self.assert_(Test.move_to_line(1) == True)
           # if (Test.move_to_line(1) != True):
           #     failed = 1
            
            Test.set_text("/abc[d-f]/\n/abc/")
            self.assert_(Test.move_to_line(-1) == False)
           # if (Test.move_to_line(-1) != False):
           #     failed = 1
                           
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testMsfmParserParseNumLines(self):
        """Tries to test behavior of msfm_parser method num_lines"""
        failed = 0
        try:
            Test = msfm_parser.msfm_parser()
            
            if (Test.num_lines() != 0):
                failed = 1
                
            Test.set_text("/abc[d-f]/")
            if (Test.num_lines() != 1):
                failed = 1
            
            Test.set_text("/abc[d-f]/\n/abc/")
            if (Test.num_lines() != 2):
                failed = 1
                           
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testDeterminisationBasic1(self):
        """ Tries to perform Determinisation test Basic 1 """
        failed = 0
        try:
            dfa = b_dfa.b_dfa()
            nfa0 = nfa_data.nfa_data()
            state1 = b_state.b_State(0,0)
            state2 = b_state.b_State(1,0)
            state3 = b_state.b_State(2,0)
            state4 = b_state.b_State(3,1)
            nfa0.states[0] = state1
            nfa0.states[1] = state2
            nfa0.states[2] = state3
            nfa0.states[3] = state4
            Symbol = sym_char.b_Sym_char("a", "a", 0)
            nfa0.alphabet[0] = Symbol
            nfa0.start = 0
            nfa0.final.add(3)
            trans1 = (0,0,1)
            trans2 = (0,0,2)
            trans3 = (1,0,3)
            trans4 = (2,0,3)
            nfa0.transitions.add(trans1)
            nfa0.transitions.add(trans2)
            nfa0.transitions.add(trans3)
            nfa0.transitions.add(trans4)
            nfa0.Flags["Epsilon Free"] = True
            dfa.create_from_nfa_data(nfa0)   
            dfa.determinise()
            nfa = dfa.get_automaton()

            self.assert_(nfa.start == 0)
            self.assert_(len(nfa.final) == 1)
            self.assert_(len(nfa.alphabet) == 1)
            self.assert_(len(nfa.states) == 3)
            self.assert_(len(nfa.transitions) == 2)
            
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testSymCharCMP(self):
        """Tries to test behavior of sym_char methods __eq__ and __neq__"""
        failed = 0
        try:
            s1 = sym_char.b_Sym_char("a","a",0)
            s2 = sym_char.b_Sym_char("a","a",1)
            s3 = sym_char.b_Sym_char("b","b",2)
            
            self.assert_((s1 == s2) == True)
            self.assert_((s1 == s3) == False)
            self.assert_((s1 != s2) == False)
            self.assert_((s1 != s3) == True)
                                                   
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testSymCharHash(self):
        """Tries to test behavior of sym_char method __hash__"""
        failed = 0
        try:
            s1 = sym_char.b_Sym_char("a","a",0)
            s2 = sym_char.b_Sym_char("a","a",1)
            s3 = sym_char.b_Sym_char("b","b",2)
            
            symDict = dict()
            symDict[s1] = s1.get_id()
            symDict[s3] = s3.get_id()
            
            self.assert_(symDict.has_key(s2) == True)
            
            symDict[s2] = s2.get_id()
            
            
            self.assert_((symDict[s1] == s2.get_id()) == True)
            self.assert_(len(symDict) == 2)
                                                   
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testSymCharClassCMP(self):
        """Tries to test behavior of sym_char_class methods __eq__ and __neq__"""
        failed = 0
        try:
            s1 = sym_char_class.b_Sym_char_class("ab", set(["a", "b"]),0)
            s2 = sym_char_class.b_Sym_char_class("ab", set(["a", "b"]),1)
            s3 = sym_char_class.b_Sym_char_class("bc", set(["b", "c"]),2)
            
            self.assert_((s1 == s2) == True)
            self.assert_((s1 == s3) == False)
            self.assert_((s1 != s2) == False)
            self.assert_((s1 != s3) == True)
                                                   
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testSymCharClassHash(self):
        """Tries to test behavior of sym_char_class method __hash__"""
        failed = 0
        try:
            s1 = sym_char_class.b_Sym_char_class("ab", set(["a", "b"]),0)
            s2 = sym_char_class.b_Sym_char_class("ab", set(["a", "b"]),1)
            s3 = sym_char_class.b_Sym_char_class("bc", set(["b", "c"]),2)
            
            symDict = dict()
            symDict[s1] = s1.get_id()
            symDict[s3] = s3.get_id()
            
            self.assert_(symDict.has_key(s2) == True)
            
            symDict[s2] = s2.get_id()
            
            
            self.assert_((symDict[s1] == s2.get_id()) == True)
            self.assert_(len(symDict) == 2)
                                                   
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testJoin1(self):
        """Tries to join 2 same automata """
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            Test1 = msfm_parser.msfm_parser()
            
            Test0.set_text("/abc[d-f]/")
            Test1.set_text("/abc[d-f]/")
            
            nfa0 = Test0.get_nfa()
            nfa1 = Test1.get_nfa()

            bnfa = b_automaton.b_Automaton()
            bnfa.create_from_nfa_data(nfa0)
            bnfa.join(nfa1)
            nfa = bnfa.get_automaton()
            
            self.assert_(len(nfa.alphabet) == len(nfa1.alphabet))
            self.assert_(len(nfa.states) == len(nfa0.states) + len(nfa1.states))
            self.assert_(len(nfa.transitions) == len(nfa0.transitions) + len(nfa1.transitions) + 1)
            self.assert_(len(nfa.final) == len(nfa0.final) + len(nfa1.final))
            self.assert_(nfa.Flags["Deterministic"] == False)
            self.assert_(nfa.Flags["Epsilon Free"] == False)
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testJoin2(self):
        """Tries to join 2 different automata """
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            Test1 = msfm_parser.msfm_parser()
            
            Test0.set_text("/^abc[d-f]/")
            Test1.set_text("/^9812(5464)|(446p*)/")
            
            nfa0 = Test0.get_nfa()
            nfa1 = Test1.get_nfa()

            bnfa = b_automaton.b_Automaton()
            bnfa.create_from_nfa_data(nfa0)
            bnfa.join(nfa1)
            nfa = bnfa.get_automaton()
            
            self.assert_(len(nfa.alphabet) != len(nfa1.alphabet))
            self.assert_(len(nfa.alphabet) == len(nfa0.alphabet) + len(nfa1.alphabet) - 1)
            self.assert_(len(nfa.states) == len(nfa0.states) + len(nfa1.states))
            self.assert_(len(nfa.transitions) == len(nfa0.transitions) + len(nfa1.transitions) + 1)
            self.assert_(len(nfa.final) == len(nfa0.final) + len(nfa1.final))
            self.assert_(nfa.Flags["Deterministic"] == False)
            self.assert_(nfa.Flags["Epsilon Free"] == False)
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testJoin3(self):
        """Tries to join 2 automata with modify_reg_exp_num == True """
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            Test1 = msfm_parser.msfm_parser()
            
            Test0.set_text("/abc[d-f]/")
            Test1.set_text("/abc[d-f]/")
            
            nfa0 = Test0.get_nfa()
            nfa1 = Test1.get_nfa()

            bnfa = b_automaton.b_Automaton()
            bnfa.create_from_nfa_data(nfa0)
            bnfa.join(nfa1, True)
            nfa = bnfa.get_automaton()
            
            rnums = set()
            for i in nfa.final:
                rnums.add(nfa.states[i]._rnum)
            
            self.assert_(len(rnums) == 2)
            
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testJoin4(self):
        """Tries to join 2 automata with modify_reg_exp_num == False """
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            Test1 = msfm_parser.msfm_parser()
            
            Test0.set_text("/abc[d-f]/")
            Test1.set_text("/abc[d-f]/")
            
            nfa0 = Test0.get_nfa()
            nfa1 = Test1.get_nfa()

            bnfa = b_automaton.b_Automaton()
            bnfa.create_from_nfa_data(nfa0)
            bnfa.join(nfa1, False)
            nfa = bnfa.get_automaton()
            
            rnums = set()
            for i in nfa.final:
                rnums.add(nfa.states[i]._rnum)
            
            self.assert_(len(rnums) == 1)
            
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
        
    def testRemoveEpsilons(self):
        """Tries to remove epsilon transitions from automaton"""
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            Test1 = msfm_parser.msfm_parser()
            
            Test0.set_text("/abc[d-f]/")
            Test1.set_text("/9812(5464)|(446p*)/")
            
            nfa0 = Test0.get_nfa()
            nfa1 = Test1.get_nfa()

            bnfa = b_automaton.b_Automaton()
            bnfa.create_from_nfa_data(nfa0)
            bnfa.join(nfa1, False)
            bnfa.remove_epsilons()
            nfa = bnfa.get_automaton()
                 
            epsnums = 0 
            for trans in nfa.transitions:
                if trans[1] == -1:
                    epsnums += 1
            
            self.assert_(epsnums == 0)
            
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)
            
    def testRemoveIUB(self):
        """ Tries to remove isolated, unreachable and blind states """
        failed = 0
        try:
            bnfa = b_automaton.b_Automaton()
            state1 = b_state.b_State(0,1)
            state2 = b_state.b_State(1,0)
            state3 = b_state.b_State(2,0)
            nfa0 = nfa_data.nfa_data()
            
            nfa0.states[0] = state1
            nfa0.states[1] = state2
            nfa0.states[2] = state3
            Symbol = sym_char.b_Sym_char("a", "a", 0)
            nfa0.alphabet[0] = Symbol
            nfa0.start = 0
            nfa0.final.add(0)
            trans1 = (1,0,2)
            trans2 = (0,0,0)
            nfa0.transitions.add(trans1)
            nfa0.transitions.add(trans2)
            bnfa.create_from_nfa_data(nfa0)
            bnfa.remove_unreachable()
            nfa = bnfa.get_automaton()
            
            self.assert_(len(nfa.final) == 1)
            self.assert_(len(nfa.states) == 1)
            self.assert_(len(nfa.transitions) == 1)
            
        except None:
            failed = 1
        self.assert_(failed == 0)
        
    def testCreateByParser(self):
        """Tries to create b_Automaton object using create_by_parser method"""
        failed = 0
        try:
            Test0 = msfm_parser.msfm_parser()
            
            Test0.set_text("/^abc[d-f]/\n/^9812(5464)|(446p*)/")
            
            bnfa = b_automaton.b_Automaton()
            self.assert_(bnfa.create_by_parser(Test0) == True)
            
            #bnfa.remove_epsilons()
            
            nfa = bnfa.get_automaton()
            self.assert_(len(nfa.states) == 31)
            self.assert_(len(nfa.alphabet) == 13)
            self.assert_(len(nfa.transitions) == 33)
            self.assert_(len(nfa.final) == 2)
                        
        except None: #Exception:
            failed = 1
        self.assert_(failed == 0)

if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(BasicTests)
    unittest.TextTestRunner(verbosity=2).run(suite)

###############################################################################
# End of File ptrn_test.py                                                    #
###############################################################################