/**
 * \file pcre2nfa.cpp
 * \brief Functions to transform pcre into nfa
 * \author Andrej Hank <xhanka00@liberouter.org> 
 * \date 2007
 * 
 * Copyright (C) 2007 CESNET
 * 
 * $Id: pcre2nfa.cpp,v 1.12 2007/10/21 20:30:18 xhanka00 Exp $
 *
 */

#include "pcre2nfa.hpp"
__RCSID("$Id: pcre2nfa.cpp,v 1.12 2007/10/21 20:30:18 xhanka00 Exp $");
#define DEBUG_HEADER "pcre2nfa"

using namespace std;

// local fuctions
//int insertSymbolRecord(tSymbolTable &table, tSymbolTableRecord record);
bool compareSymbolRecords(tSymbolTableRecord first, tSymbolTableRecord second);
void printSymbolTableRecord(ostream &fout, tSymbolTableRecord *with, bool epsMove, bool dot);
bool addModifier(char ch, tModifiers &modifiers);
	
/**
 * \brief Initialize map container
 * \param map Map to initialize
 * \param labels Labels to be set
 * \param count Number of items to set
 */
void initPcreParser(map<string, tCharClass>& map, const string labels[], int count) {
	for(int i = 0; i < count; i++) {
		map[labels[i]] = (tCharClass)i;
	}
}

/** 
 * \brief Create transition
 * \param au Automata
 * \param from State from
 * \param to State to
 * \param with Symbol table record to perform move with
 * \param epsMove Is epsilon move
 * \return Success of creation
 */
bool mkMove(tAutomata *au, int from, int to, tSymbolTableRecord *with, bool epsMove) {
	tMove move;
	move.stateFrom = from;
	move.stateTo = to;
	move.epsMove = epsMove;

	// check table record
	if(!epsMove) {
		if(!with) {
			DEBUG ("Symbol Record missing" << endl);
			return false;
		}
		// insert SymbolRecord
		move.symbolTableIndex = insertSymbolRecord(au->symbolTable, *with);
	}

	// insert move
	au->moves.push_back(move);
	
	// print contents
	DEBUG ("-> New move from " << from << " to " << to);
	if(debug)
		printSymbolTableRecord(cout, with, epsMove, false);

	return true;
}

/**
 * \brief	Add new state
 *
 */
void addState(tAutomata *au, bool isEndState, int * currentState) {
	if(isEndState) {
		tEndState endState;
		//endState.ruleID.push_back();
		endState.stateNo = au->stateCount;
		au->endStates.push_back(endState);
	}

	// mark end state map
	au->mapEndStates[au->stateCount] = isEndState;

	au->stateCount++;
	*currentState = au->stateCount - 1;
	DEBUG("* New state " << au->stateCount - 1 << endl);
}

/* -------------------------------------------------------------------------- */
/** \brief automata2ps Export nfa to postscript picture
 * 
 * \param regex Exported regular expresion
 * \param automata FSM created from regexp
 * \param psfile File to export
 * 
 * \return 
 * 	- true if successfully created
 * 	- false if failed
 */
/* -------------------------------------------------------------------------- */
bool automata2ps(string regex, tAutomata *automata, string psfile) {
	string tmpFileName = TMP_DOT_FILE;
	ofstream fdot(tmpFileName.c_str());
	vector<tMove>::iterator moveiter;
	vector<tEndState>::iterator enditer;
	moveiter = automata->moves.begin();
	enditer = automata->endStates.begin();

	fdot << "digraph \"" << regex << "\" {" << endl;
	fdot << "   rankdir=LR;" << endl;
	fdot << "   size=\"8,5\""<< endl;
	fdot << "node [shape = doublecircle];" << endl;

	// end states
	enditer = automata->endStates.begin();
	for (; enditer != automata->endStates.end(); enditer++)
		fdot << enditer->stateNo << ";" << endl;

	fdot << "node [shape = circle];" << endl;

	// moves
	moveiter = automata->moves.begin();
	for (; moveiter != automata->moves.end(); moveiter++) {
		fdot << moveiter->stateFrom << " -> " << moveiter->stateTo;
		fdot << " [ label = \"";
		tSymbolTableRecord *with;
		if(moveiter->epsMove)
			with = NULL;
		else
			with = &(automata->symbolTable.at(moveiter->symbolTableIndex));

		printSymbolTableRecord(fdot, with, moveiter->epsMove, true);

		fdot << " \" ];" << endl;
	}

	fdot << "}" << endl;
	fdot.close();

	//string command = "dot -Tps " + tmpFileName + " >" + psfile;
	return true;//!system(command.c_str());
}
/**
 * \brief 	Add Grouping Record
 */
void addGroupingRecord(vector<tGroupingRecord> &stack, int start, bool endStateCreated, int end) {
	tGroupingRecord record = { start, endStateCreated, end };
	stack.push_back(record);
	DEBUG("() gouping record added: " << start << " " << endStateCreated << " " << end << endl);
}
void removeGroupingRecord(vector<tGroupingRecord> &stack, tGroupingRecord &lastGroup) {
	lastGroup = stack.back();
	stack.pop_back();
	DEBUG("() grouping record removed, lastGroup from: " << lastGroup.startState << " to: " << lastGroup.endState << endl);
}
/* -------------------------------------------------------------------------- */
/** \brief groupStart Mark beginning of group into grouping stack
 * 
 * \param au Automaton
 * \param groupingStack Stack to write
 * \param currentState Current state
 */
/* -------------------------------------------------------------------------- */
void groupStart(tAutomata *au, vector<tGroupingRecord> &groupingStack, int * currentState) {
	// redundant state with epsilon move before every group needed - without
	// it caused bug when OR "|" followed left parenthesis "("
	mkMove(au, *currentState, au->stateCount, NULL, true);
	addState(au, false, currentState); // group start
	
	// mark beginning of group
	addGroupingRecord(groupingStack, au->stateCount - 1, false, -1);
	DEBUG("() group starts @ " << au->stateCount - 1 << endl);
}
/* -------------------------------------------------------------------------- */
/** \brief groupEnd Mark group end into grouping stack and in last group record,
 * create necessary state if needed 
 * 
 * \param au Automaton 
 * \param groupingStack Grouping stack
 * \param currentState Current state
 * \param lastGroup Last group created
 */
/* -------------------------------------------------------------------------- */
void groupEnd(tAutomata *au, vector<tGroupingRecord> &groupingStack, int * currentState, tGroupingRecord &lastGroup) {
	tGroupingRecord group = groupingStack.back();

	// end of group - more branches - if group end state created, create move into
	// created end state, add new state with epsilon move after whole ended
	// group and adjust grouping record
	if(group.endStateCreated) { 
		mkMove(au, *currentState, group.endState, NULL, true);
		mkMove(au, group.endState, au->stateCount, NULL, true);
		addState(au, false, currentState);
		group.endState = *currentState;
		groupingStack.pop_back();
		groupingStack.push_back(group);
	} else { // end fo group state not created - only one branch - crate new end state
		mkMove(au, *currentState, au->stateCount, NULL, true);
		addState(au, false, currentState); // group end
		// adjust tGroupingRecord
		group.endStateCreated = true;
		group.endState = au->stateCount - 1;
		groupingStack.pop_back();
		groupingStack.push_back(group);
	}
	DEBUG("() group end, current set to " << *currentState << endl);
	// remove tGroupingRecord
	removeGroupingRecord(groupingStack, lastGroup);
}
/* -------------------------------------------------------------------------- */
/** \brief groupOr Group branches, end previous branch
 * 
 * \param au Automaton
 * \param groupingStack Grouping stack
 * \param currentState Current state
 */
/* -------------------------------------------------------------------------- */
void groupOr(tAutomata *au, vector<tGroupingRecord> &groupingStack, int * currentState) {
	tGroupingRecord group = groupingStack.back();

	// end of group branch 
	if(group.endStateCreated) { // just crate move into crated state
		mkMove(au, *currentState, group.endState, NULL, true);
	} else { // end fo group state not created - create new state
		mkMove(au, *currentState, au->stateCount, NULL, true);
		addState(au, false, currentState); 

		// adjust tGroupingRecord
		group.endStateCreated = true;
		group.endState = au->stateCount - 1;
		groupingStack.pop_back();
		groupingStack.push_back(group);
		
		DEBUG("() group branch, end state " << au->stateCount - 1 << " created" << endl);
	}
	
	// set current state at beginning of group
	*currentState = group.startState;
	DEBUG("() group branch, current set to " << *currentState << endl);
}

/* -------------------------------------------------------------------------- */
/**
 * \brief Create epsilon move which skips state/group
 * 
 * \param au Automaton
 * \param lastGroup Last group
 * \param currentState Current state
 */
/* -------------------------------------------------------------------------- */
void quantityZero(tAutomata *au, tGroupingRecord &lastGroup, int currentState) {
	if(currentState == lastGroup.endState) {
		// after group end, count with whole group
		mkMove(au, lastGroup.startState, lastGroup.endState, NULL, true);
		DEBUG("? quantity ZERO - from " << lastGroup.startState << " to " << lastGroup.endState << endl);
	} else {
		mkMove(au, currentState - 1, currentState, NULL, true);
		DEBUG("? quantity ZERO - from " << currentState - 1 << " to " << currentState << endl);
	}
}

/* -------------------------------------------------------------------------- */
/**
 * \brief Create epsilon move which returns into state/beginning of group
 * 
 * \param au Automaton
 * \param checkGroup Group or only state
 * \param lastGroup Last group
 * \param currentState Current state
 */
/* -------------------------------------------------------------------------- */
void quantityMore(tAutomata *au, bool checkGroup, tGroupingRecord &lastGroup, int currentState) {
	if(checkGroup && currentState == lastGroup.endState) {
		// after group end, count with whole group
		mkMove(au, lastGroup.endState, lastGroup.startState, NULL, true);
		DEBUG("* quantity MORE - from " << lastGroup.endState << " to " << lastGroup.startState << endl);
	} else {
		// with characters from last move
		mkMove(au, currentState, currentState, &(au->symbolTable.at(au->moves.back().symbolTableIndex)), false);
		DEBUG("* quantity MORE - from " << currentState - 1 << " to " << currentState - 1 << endl);
	}
}

/* -------------------------------------------------------------------------- */
/** \brief repeatPart Add to automaton part of automaton, used with repeating of
 * groups
 *
 * Example (abc){2,4}
 * 
 * \param au Automaton
 * \param stateFrom Beginning of repeating
 * \param stateTo End of repeating
 * \param compulsoryTimes Compulsory times of repetition
 * \param optionalTimes Optional times of repetition
 * \param currentState Current state
 * 
 * \return 
 * 	- true if successfully added 
 * 	- false if failed
 */
/* -------------------------------------------------------------------------- */
bool repeatPart(tAutomata *au, int stateFrom, int stateTo, int compulsoryTimes, int optionalTimes, int *currentState) {
	DEBUG("{} repeating " << compulsoryTimes << "-" << optionalTimes << "times, from: " << stateFrom << " to: " << stateTo << endl);
	int stateDifference = stateTo - stateFrom;

	// first move of repeating
	vector<tMove>::iterator firstMove = au->moves.end() + 1;
	// last move of repeating
	vector<tMove>::iterator lastMove = au->moves.end() + 1;

	// find first move into initial state
	for(vector<tMove>::iterator moveiter = au->moves.begin(); moveiter != au->moves.end(); moveiter++) {	
		if(moveiter->stateFrom == stateFrom) {
			firstMove = moveiter;
			break;
		}
	}
	if(firstMove == (au->moves.end() + 1))
		return false;
	
	// find last move into end state
	for(vector<tMove>::iterator moveiter = au->moves.end(); moveiter != firstMove; moveiter--) {	
		if(moveiter->stateTo == stateTo) {
			lastMove = moveiter;
			break;
		}
	}
	if(lastMove == (au->moves.end() + 1))
		return false;

	int times = (compulsoryTimes > optionalTimes)? compulsoryTimes : optionalTimes;
	for(int i = 1; i <= times; i++) {
		// now generate exactly same states and moves
		for(vector<tMove>::iterator moveiter = firstMove; moveiter != lastMove; moveiter++) {	
			if(moveiter->epsMove)
				mkMove(au, moveiter->stateFrom + (i * stateDifference), moveiter->stateTo + (i * stateDifference), NULL, true);
			else {
				tSymbolTableRecord with = (au->symbolTable.at(moveiter->symbolTableIndex));
				mkMove(au, moveiter->stateFrom + (i * stateDifference), moveiter->stateTo + (i * stateDifference), &with, false);
			}
		}
		// last state in repeat
		if(lastMove->epsMove)
			mkMove(au, lastMove->stateFrom + (i * stateDifference), lastMove->stateTo + (i * stateDifference), NULL, true);
		else {
			tSymbolTableRecord with = (au->symbolTable.at(lastMove->symbolTableIndex));
			mkMove(au, lastMove->stateFrom + (i * stateDifference), lastMove->stateTo + (i * stateDifference), &with, false);
		}

		// create states
		for(int j = 1; j <= stateDifference; j++) {
			addState(au, false, currentState);
		}
	}
	// check and do {m} || {m,n} || {m,}
	if((optionalTimes + 1) == 0) { // {m,}
		// last *
		tGroupingRecord record; 
		record.endState = *currentState;
		record.startState = *currentState - stateDifference;
		record.endStateCreated = true;
		quantityMore(au, true, record, *currentState);
	} else if(compulsoryTimes > optionalTimes) {
		errx(1, "repeating - wrong values - {%d,%d}", compulsoryTimes + 1, optionalTimes + 1);
	} else { // {m,n}
		// create epsilon moves only where optional
		for(int i = 1; i <= optionalTimes - compulsoryTimes; i++) {
			mkMove(au, *currentState - (i*stateDifference), *currentState, NULL, true);
		}
	}
	DEBUG("{} repeating end" << endl);
	return true;
}

/* -------------------------------------------------------------------------- */
/** \brief repeatLast Repeat last state
 *
 * Example a{4}
 * 
 * \param au Automaton
 * \param currentState Current state
 * \param compulsoryTimes Compulsory times of repetition
 * \param optionalTimes Optional times of repetition
 */
/* -------------------------------------------------------------------------- */
void repeatLast(tAutomata *au, int *currentState, int compulsoryTimes, int optionalTimes) {
	tMove lastMove = au->moves.back();
	tSymbolTableRecord with = au->symbolTable.at(lastMove.symbolTableIndex);

	DEBUG("{} repeating " << compulsoryTimes << "-" << optionalTimes << "times" << endl);
	int times = (compulsoryTimes > optionalTimes)? compulsoryTimes : optionalTimes;
	for(int i = 1; i <= times; i++) {
		mkMove(au, *currentState, au->stateCount, &with, false);
		addState(au, false, currentState);
	}

	// check and do {m} || {m,n} || {m,}
	if((optionalTimes + 1) == 0) { // {m,}
		// last *
		tGroupingRecord record; // dummy
		quantityMore(au, false, record, *currentState);
	} else if(compulsoryTimes > optionalTimes) {
		errx(1, "repeating - wrong values - {%d,%d}", compulsoryTimes + 1, optionalTimes + 1);
	} else { // {m,n}
		// create epsilon moves only where optional
		for(int i = 1; i <= optionalTimes - compulsoryTimes; i++) {
			mkMove(au, *currentState - i, *currentState, NULL, true);
		}
	}
}

/* -------------------------------------------------------------------------- */
/** \brief insertSymbolRecord Insert symbol record into table and return its index
 * from table
 * 
 * \param table Table to insert
 * \param record Record to insert
 * 
 * \return Index of record in table
 */
/* -------------------------------------------------------------------------- */
int insertSymbolRecord(tSymbolTable &table, tSymbolTableRecord record) {
	vector<tSymbolTableRecord>::iterator iter = table.begin();
	int i = 0;
	for(; iter != table.end(); iter++) {
		if(compareSymbolRecords(record, table.at(i))) {
			DEBUG("Record found in table at position " << i << endl);
			return i;
		}
		i++;
	}
	// not int table, insert new
	table.push_back(record);
	DEBUG("Record not found in table, inserting at position " << table.size() - 1<< endl);
	return table.size() - 1;
}


/* -------------------------------------------------------------------------- */
/** \brief compareSymbolRecords Compare two symbol table records
 * 
 * \param first First record
 * \param second Second record
 * 
 * \return 
 * 	- true if records equal
 * 	- flase if not
 */
/* -------------------------------------------------------------------------- */
bool compareSymbolRecords(tSymbolTableRecord first, tSymbolTableRecord second) {

	// compare enumerated
	for(vector<char>::iterator firstIter = first.enumerated.begin(); firstIter != first.enumerated.end(); firstIter++) {
		bool found = false;
		for(vector<char>::iterator secondIter = second.enumerated.begin(); secondIter != second.enumerated.end(); secondIter++) {
			if(*firstIter == *secondIter) {
				// delete found from vector
				first.enumerated.erase(firstIter);
				second.enumerated.erase(secondIter);
				firstIter--;
				secondIter--;
				found = true;
				break;
			}
		}
		if(!found)
			return false;
	}
	// check if completly same
	if(first.enumerated.size() != 0 || second.enumerated.size() != 0)
		return false;
		
	// compare classFlags 
	for(vector<tCharClass>::iterator firstIter = first.classFlags.begin(); firstIter != first.classFlags.end(); firstIter++) {
		bool found = false;
		for(vector<tCharClass>::iterator secondIter = second.classFlags.begin(); secondIter != second.classFlags.end(); secondIter++) {
			if(*firstIter == *secondIter) {
				// delete found from vector
				first.classFlags.erase(firstIter);
				second.classFlags.erase(secondIter);
				firstIter--;
				secondIter--;
				found = true;
				break;
			}
		}
		if(!found)
			return false;
	}
	// check if completly same
	if(first.classFlags.size() != 0 && second.classFlags.size() != 0)
		return false;

	return true;
}

/* -------------------------------------------------------------------------- */
/** \brief printSymbolTableRecord Just print symbolTableRecord content int if
 * debug mode
 * 
 * \param fout Stream to print
 * \param with Record to print
 * \param epsMove Is epsilon move?
 * \param dot For dot to postscript record
 */
/* -------------------------------------------------------------------------- */
void printSymbolTableRecord(ostream &fout, tSymbolTableRecord *with, bool epsMove, bool dot) {
	string escapePrefix;
	if(dot)
		escapePrefix = "\\\\";
	else
		escapePrefix = "\\";

	// print tSymbolTableRecord content
	if(!epsMove) {
		// print classes if any
		if(with->classFlags.size() != 0) {
			if(!dot)
				fout << " with classes ";
			for(vector<tCharClass>::iterator iter = with->classFlags.begin(); iter != with->classFlags.end(); iter++) {	
				fout << CCStrings[*iter] << " ";
			}
			if(!dot)
				fout << endl;
		}
		// print enumerated characters if any
		if(with->enumerated.size() != 0) {
			if(!dot)
				fout << " with enumerated ";
			char lastCharacter;
			bool print = false;
			bool started = false;
			for(vector<char>::iterator iter = with->enumerated.begin(); iter != with->enumerated.end(); iter++) {	
				if(!started) {
					fout << " ";
					print = true;
					if((iter + 1) == with->enumerated.end() || (*(iter + 1) != ((*iter) + 1))) { // end single
						started = false;
						lastCharacter = *iter;
					} else {
						started = true;
						lastCharacter = *iter;
					}
				} else { // group already started 
					if((iter + 1) == with->enumerated.end() || (*(iter + 1) != (*iter + 1))) { // end group
							fout << "-";
							print = true;
							started = false;
							lastCharacter = *iter;
					} else { // continuing of class
						print = false;
						lastCharacter = *iter;
					}
				}
				if(print) {
					switch (lastCharacter) {
						case '\t':
							fout << escapePrefix + "t";
							break;
						case '\n':
							fout << escapePrefix + "n";
							break;
						case '\r':
							fout << escapePrefix + "r";
							break;
						case '\f':
							fout << escapePrefix + "f";
							break;
						case '\a':
							fout << escapePrefix + "a";
							break;
						case '\e':
							fout << escapePrefix + "e";
							break;
						case ' ':
							if(dot)
								fout << "\\\" \\\"";
							else
								fout << "\" \"";
							break;
						default: // if not printable character, print as hexa escape sequence
							if(isprint(*iter)) {
								fout << lastCharacter;
							} else {
								fout << escapePrefix + "x" << setw(2) << setfill('0') << hex << (int)((unsigned char)lastCharacter);
								fout << dec;
							}
							break;
					}
				}
			}
			if(!dot)
				fout << endl;
		}
	} else { // print epsilon move
		if(!dot)
			fout << " with EPSILON" << endl;
		else
			fout << " Eps.";

	}

}


/* -------------------------------------------------------------------------- */
/** \brief addEndStateRule Add to end state, rule which ends in this state
 * 
 * \param endState State where rule will be added
 * \param ruleId Rule number to be added
 * 
 * \return 
 * 	- true if success
 * 	- false else
 */
/* -------------------------------------------------------------------------- */
bool addEndStateRule(tEndState &endState, int ruleId) {
	endState.ruleId.push_back(ruleId);
	DEBUG("EndState " << endState.stateNo << " added ruleId " << ruleId << endl);
	return true;
}

/* -------------------------------------------------------------------------- */
/** \brief printSymbolTable Print Symbol Table
 * 
 * \param fout File to print
 * \param table Table to print
 */
/* -------------------------------------------------------------------------- */
void printSymbolTable(ostream &fout, tSymbolTable table) {
	vector<tSymbolTableRecord>::iterator iter = table.begin();
	int i = 0;
	for(; iter != table.end(); iter++) {
		fout << i;
		printSymbolTableRecord(fout, &(*iter), false, false);
		i++;
	}
}

/* -------------------------------------------------------------------------- */
/** \brief getModifiers Get modifiers from regex
 * 
 * \param regex Regex
 * 
 * \return Structure of set modifiers
 */
/* -------------------------------------------------------------------------- */
tModifiers getModifiers(string regex) {
	tModifiers modifiers;
	modifiers.i = false;
	modifiers.m = false;
	modifiers.s = false;
	modifiers.x = false;

	string::size_type position;
	// check front modifiers
	position = regex.find_first_of('/');
	if(position != string::npos) {
		for(string::size_type i = 0; i < position; i++)
			if(!addModifier(regex.at(i), modifiers))
				DEBUG("Modifier " << regex.at(i) << " not recognized" << endl);
	}

	// check rear modifiers
	position = regex.find_last_of('/');
	if(position != string::npos) {
		for(string::size_type i = regex.size() - 1; i > position; i--)
			if(!addModifier(regex.at(i), modifiers))
				DEBUG("Modifier " << regex.at(i) << " not recognized" << endl);
	}
	return modifiers;
}

/* -------------------------------------------------------------------------- */
/** \brief addModifier Add single modifier into structure
 * 
 * \param ch Single modifier
 * \param modifiers Structure of modifier
 * 
 * \return 
 * 	- true if correct modifier
 * 	- false if unknown modifier
 */
/* -------------------------------------------------------------------------- */
bool addModifier(char ch, tModifiers &modifiers) {
	switch(ch) {
		case 'i':
			modifiers.i = true;
			break;
		case 'm':
			modifiers.m = true;
			break;
		case 's':
			modifiers.s = true;
			break;
		case 'x':
			modifiers.x = true;
			break;
		default:
			return false;
	}
	return true;
}

/* -------------------------------------------------------------------------- */
/** \brief printModifiers Print modifiers
 * 
 * \param modifiers Modifiers structure
 */
/* -------------------------------------------------------------------------- */
void printModifiers(tModifiers modifiers) {
	cout << "Modifiers i m s x" << endl;
	cout << "          " << modifiers.i << " " << modifiers.m << " " << modifiers.s << " " << modifiers.x << endl;
}

/* -------------------------------------------------------------------------- */
/** \brief symbolRecordAddEnumerated Add enumerated symbol into symbol record,
 * reflect modifier
 * 
 * \param modifiers Modifiers 
 * \param record Record
 * \param ch Char to add
 */
/* -------------------------------------------------------------------------- */
void symbolRecordAddEnumerated(tModifiers modifiers, tSymbolTableRecord &record, char ch) {
	vector<char>::iterator iter = record.enumerated.begin();
	bool found = false;

	// check if not already in
	for(; iter != record.enumerated.end(); iter++) {
		if(*iter == ch) {
			found = true;
			break;
		}
	}
	
	
	if(modifiers.i) { // case insensitive flag set
		if(islower(ch)) {
			iter = record.enumerated.begin();
			bool upper_found = false;
			for(; iter != record.enumerated.end(); iter++) {
				if(*iter == toupper(ch)) {
					upper_found = true;
					break;
				}
			}
			if(!found) {
				record.enumerated.push_back(ch);
			}
			if(!upper_found) {
				record.enumerated.push_back(toupper(ch));
			}
		} else if(isupper(ch)) {
			iter = record.enumerated.begin();
			bool lower_found = false;
			for(; iter != record.enumerated.end(); iter++) {
				if(*iter == tolower(ch)) {
					lower_found = true;
					break;
				}
			}
			if(!found) {
				record.enumerated.push_back(ch);
			}
			if(!lower_found) {
				record.enumerated.push_back(tolower(ch));
			}
		} else {
			if(!found) {
				record.enumerated.push_back(ch);
			}
		}
	} else { // case sensitive
		if(!found)
			record.enumerated.push_back(ch);
	}
}
