/**
 * \file pcre2hw.cpp
 * \brief Functions to transform pcre into hw
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2007
 *
 * Copyright (C) 2007 CESNET
 *
 * $Id: pcre2hw.cpp,v 1.2 2007/04/29 14:10:48 xhanka00 Exp $
 *
 */

#include "pcre2hw.hpp"
__RCSID("$Id: pcre2hw.cpp,v 1.2 2007/04/29 14:10:48 xhanka00 Exp $");
#define DEBUG_HEADER "pcre2hw"

/* -------------------------------------------------------------------------- */
/** \brief generateNfaPar Generate VHDL source with NFA parameters
 * 
 * \param noOfRules Number of rules in automaton
 * \param filename Output filename
 */
/* -------------------------------------------------------------------------- */
void generateNfaPar(int noOfRules, char *filename) {
	ofstream fout(filename);
	fout << "-- nfa_par.vhd: This file is automaticaly generated by pcre_gen" << endl;
	fout << "-- Copyright (C) 2007 CESNET" << endl;
	fout << "-- Author(s): Petr Kobiersky <xkobie00@stud.fit.vutbr.cz>" << endl;
	fout << "--            Andrej Hank <xhanka00@liberouter.org>" << endl << endl;
	fout << "library IEEE;" << endl;
	fout << "use IEEE.std_logic_1164.all;" << endl << endl;
	fout << "-- ----------------------------------------------------------------------------" << endl;
	fout << "--                        Package declaration" << endl;
	fout << "-- ----------------------------------------------------------------------------" << endl << endl;
	fout << "package NFA_PAR is" << endl;
	fout << "-- output vector width from NFA" << endl;
	fout << "constant RULES_COUNT  : integer := " << noOfRules << ";" << endl;
	// only 1 char per clk, not extended FSM
	fout << "constant CHARS_PER_CLK_CYCLE : integer := 1;" << endl << endl;
	fout << "end NFA_PAR;" << endl;
	fout.close();
}

/* -------------------------------------------------------------------------- */
/** \brief generateStates Generate state VHDL components
 * 
 * \param nfa Automaton
 * \param fsignals 
 * \param farch 
 */
/* -------------------------------------------------------------------------- */
void generateStates(tAutomata *nfa, ofstream &fsignals, ofstream &farch) {
	// Architecture begin
	farch << endl << "begin" << endl;
	// Generate pseudo begin state and set it to '1'
	fsignals << "signal state_0_out : std_logic;" << endl;
	farch    << "state_0_out <= '1';" << endl;

	// Genereate states or end_states
	for (int i = 1; i < nfa->stateCount; i++) {
		// Create signals
		fsignals << "signal state_" << i << "_in : std_logic;" << endl;
		fsignals << "signal state_" << i << "_out : std_logic;" << endl;

		if (nfa->mapEndStates[i])
			farch << "state_" << i << ": END_STATE" << endl; // Generate end state
		else
			farch << "state_" << i << ": STATE" << endl; // Generate normal state

		farch << "   port map (" << endl;
		farch << "      CLK => CLK," << endl;
		farch << "      RESET => RESET," << endl;
		farch << "      STATE_IN => state_" << i << "_in," << endl;
		farch << "      STATE_OUT => state_" << i << "_out" << endl;
		farch << "      );" << endl << endl;
	}
}

/* -------------------------------------------------------------------------- */
/** \brief existRuleID Returns true if ruleID is in rules vector
 * 
 * \param vect tEndState.ruleId vector
 * \param ruleID Rule number
 * 
 * \return 
 * 	- true if found
 * 	- false if not found
 */
/* -------------------------------------------------------------------------- */
bool existRuleID(vector<int> vect, int ruleID) {
	vector<int>::iterator iter;
	iter = vect.begin();
	for (; iter != vect.end(); iter++) {
		if ( (*iter) == ruleID )
			return true;
	}
	return false;
}

/* -------------------------------------------------------------------------- */
/** \brief mapEndStates Map End states to output with clasification results
 * 
 * \param nfa Automaton
 * \param noOfRules Number of rules in automaton
 * \param farch Output file to generate VHDL code
 */
/* -------------------------------------------------------------------------- */
void mapEndStates(tAutomata *nfa, int noOfRules, ofstream &farch) {
	// For all rules
	for (int i = 0; i < noOfRules; i++) {
		// Map to output results of clasification
		farch << "OUTPUT(" << i <<") <= CLASIFICATION(" << i <<")";

		// Set iterator to begin of endState vector
		vector<tEndState>::iterator endStIter;
		endStIter = nfa->endStates.begin();

		// For all end states
		for (; endStIter != nfa->endStates.end(); endStIter++) {
			// If state belongs to rule
			if ( existRuleID( (*endStIter).ruleId, i) )
				farch << " and state_" << (*endStIter).stateNo << "_out";
		}
		farch << ";" << endl;
	}
}


/* -------------------------------------------------------------------------- */
/** \brief nfa2hw Transform pcre NFA to VHDL representation
 * 
 * \param nfa Automaton
 * \param noOfRules Number of rules
 * \param filename Architecture output file
 * \param pack_filename Package output file
 */
/* -------------------------------------------------------------------------- */
void nfa2hw(tAutomata *nfa, int noOfRules, char *filename, char *pack_filename) {

	// Generate file with nfa parameters
	generateNfaPar(noOfRules, pack_filename);

	// Open file stream for tmp files
	ofstream fsignals(TMP_SIGNALS_FILE);
	ofstream farch(TMP_ARCHITECTURE_FILE);

	generateStates(nfa, fsignals, farch);

	//-----------------------------------------------------------
	// This is beeing generated into VHDL
	//
	//  __               _                         _
	// |  |             | |----------|\___________| |
	// |D |             |_|        |-|/           |_|
	// |E |     __    state 1      | move 1     state 2
	// |C |-----\ \________________| state_1_out and move_logic_1
	// |  |-----/_/
	// |__|  move_logic DEC(XX) or DEC(XX)
	//
	//-----------------------------------------------------------
	
	// generate move_logic
	vector<tSymbolTableRecord>::iterator symbolTableIter;
	int i = 0;
	
	// generate move logic according to symbol table
	symbolTableIter = nfa->symbolTable.begin();
	// for each record in symbol table
	for (; symbolTableIter != nfa->symbolTable.end(); symbolTableIter++) {
		fsignals << "signal move_logic_" << i << " : std_logic;" << endl;
		farch << "move_logic_" << i << " <= ";

		vector<char>::iterator charIter = symbolTableIter->enumerated.begin();
		int j = 0;
		// create integrating "or" component for enumerated characteres
		// TODO add char classes
		for (; charIter != symbolTableIter->enumerated.end(); charIter++) {
				if (j != 0)
					farch << " or ";
				farch << "DEC_IN" << "(" << (unsigned int)(*charIter) <<")";
				j++;
		} // for
		farch << ";" << endl;
		i++;
	} // for


	// generate output moves
	for (int i = 0; i < nfa->stateCount; i++) {
		vector<tMove>::iterator moveiter = nfa->moves.begin();
		int j = 0;
		for (; moveiter != nfa->moves.end(); moveiter++) {
			// check all moves if i is source state
			if (moveiter->stateFrom == i) {
				if(moveiter->epsMove) {
					fsignals << "signal move_" << j << "_out : std_logic;" << endl;
					farch << "move_" << j << "_out <= ";
					farch << "state_" << i << "_out;" << endl;
				} else {
					fsignals << "signal move_" << j << "_out : std_logic;" << endl;
					farch << "move_" << j << "_out <= ";
					farch << "state_" << i << "_out and ";
					farch << "move_logic_" << moveiter->symbolTableIndex << ";" << endl;
				}
			}
			j++;
		}
	}


	// generate state inputs according to move outputs
	// for each state except first one (0 - no input)
	list<int> dstMoves;
	list<int>::iterator dstMovesIter;

	for (int i = 1; i < nfa->stateCount; i++) {
		dstMoves.clear();
		vector<tMove>::iterator moveiter = nfa->moves.begin();
		int j = 0;
		for (; moveiter != nfa->moves.end(); moveiter++) {
			if (moveiter->stateTo == i)
				dstMoves.push_back(j);
			j++;
		}
		farch << "state_" << i << "_in <= ";

		dstMovesIter = dstMoves.begin();
		int k = 0;
		for (; dstMovesIter != dstMoves.end(); dstMovesIter++) {
			if (k > 0) farch << " or ";
			farch << "move_" << (*dstMovesIter) << "_out";
			k++;
		}
		farch << ";" << endl;
	}

	// Map end states to output
	mapEndStates(nfa, noOfRules, farch);

	farch << "end architecture NFA_ARCH;";

	// Close tmp files
	fsignals.close();
	farch.close();

	// Join .tmp files into efsm_arch_full
	string command1 = "cat " ARCHITECTURE_BEGIN_FILE " " TMP_SIGNALS_FILE " " TMP_ARCHITECTURE_FILE " > ";
	command1.append(filename);
	system(command1.c_str());
	return;
}
