###############################################################################
#  nfa_split.py: Module for PATTERN MATCH
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vlastimil Kosar <ikosar@fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

import b_state
import re
from b_nfa import b_nfa
from b_dfa import b_dfa
import nfa_data
import msfm_parser
import copy
import sym_char
import sym_char_class
import clark_nfa
import os;
import sys;
import math
from collections import deque
import cPickle

class nfa_det_part(b_dfa):
    """Deterministic part of NFA."""
    def __init__(self, index = 0):
        b_dfa.__init__(self)
        #self._states      = dict()    # set of all states 
        self._inset       = set()     # set of starting states
        self._outset      = set()     # set of outgoing states
        #self._transitions = set()     # Transitions 
        #self._final       = set()     # Final states 
        #self._alphabet    = dict()    # Finite alphabet - char classes
        self._stateWidth  = 16        # Data width of state
        self._statistic   = dict()    # Statistics dictionary
        self._LUTInputs   = 6         # Number of LUT inputs in FPGA
        self._index = index           # Number of deterministci part
        self._template = "templates/vhdl/det_part.vhd" # VHDL template of whole deterministic unit
        self._decoder_template = "templates/vhdl/decoder.vhd" # VHDL template of decoder
        self._encoder_template = "templates/vhdl/encoder.vhd" # VHDL template of encoder
        self._memory_template = "templates/vhdl/memory.vhd"   # VHDL template of memory
        self._mem_size = 0
        self._solution = None


    # ---- Inset and Outset --------------
    def  starting_states(self):
        """Return set of starting (input) states."""
        return self._inset;

    def  out_states(self):
        """Return output states."""
        return self._outset;


    # ---- Optimizatons ------------------
    def  split_det_part(self, k):
        """Split deterministic part in order to reduce complexity of
           input encoder."""
        newDetPartsList = list()
        
        reachableStates = [0]*len(self._inset)
        
        mapper = dict()
        for transition in self._automaton.transitions:
            if mapper.has_key(transition[0]) == True:
                mapper[transition[0]].add((transition[1], transition[2]))
            else:
                mapper[transition[0]] = set()
                mapper[transition[0]].add((transition[1], transition[2]))
        
        i = 0
        for state in self._inset:
            states = list()
            transitionSet = list()
            alphabetSet = list()
            outsetSet = list()
            fifo = deque()
            fifo.append(state)
            used = set()
            while len(fifo) > 0:
                cstate = fifo.popleft()
                used.add(cstate)
                if mapper.has_key(cstate):
                    transitions = mapper[cstate]
                    for trans in transitions:
                        if trans[1] not in used and trans[1] not in fifo:
                            fifo.append(trans[1])
                        states.append(trans[1])
                        transitionSet.append((cstate, trans[0], trans[1]))
                        alphabetSet.append(trans[0])
                        if trans[1] in self._outset:
                            outsetSet.append(trans[1])
                        if cstate in self._outset:
                            outsetSet.append(cstate)
            
            reachableStates[i] = [len(states), state, states, transitionSet, alphabetSet, outsetSet]
            i += 1
            
               
        
        for i in range(0, k):
            newDetPartsList.append(nfa_det_part(self._index*k+i))
        
        reachableStates.sort()
        
        
        index = 0
        while index < len(reachableStates):
            for state in self._automaton.states.keys():
                if state in reachableStates[index][2]:
                    newDetPartsList[index % k]._automaton.states[state] = self._automaton.states[state]
                    
            for fstate in self._automaton.final:
                if fstate in reachableStates[index][2]:
                    newDetPartsList[index % k]._automaton.final.add(fstate)
            
            for trans in reachableStates[index][3]:
                newDetPartsList[index % k]._automaton.transitions.add(trans)
            
            for symbol in self._automaton.alphabet.keys():
                if symbol in reachableStates[index][4]:
                    newDetPartsList[index % k]._automaton.alphabet[symbol] = self._automaton.alphabet[symbol]
            
            newDetPartsList[index % k]._inset.add(reachableStates[index][1])
            
            for outState in reachableStates[index][5]:
                newDetPartsList[index % k]._outset.add(outState)
                
            index += 1
               
        return newDetPartsList

    def  move_trasitions(self, other_det_part, num_of_transitions):
        """Move transition from other_det_part. Amount of transitions
           is specified by num_of_transitions. (to be specified 
           precisely)"""


    # ---- Transition table functions ---- 
    def  ttable_size(self):
        """Return  size of transition table which corresponds to
        deterministic part."""
        return len(self._automaton.transitions)

    def  ttable_min(self):
        """Return size of memory (Bytes) which is needed for minimal
           representation of transition table. Character classes is
           used to reduce memory utilization."""
        return self.ttable_size() + self.ttable_size()*(self._stateWidth/8) + int(math.ceil(self.ttable_size() * 3 / 8)) 

    def  ttable_overlap(self, char_classes = None):
        """Return size of memory (Bytes) for overlapping of tt rows."""
        return self._mem_size + self._mem_size * (self._stateWidth/8) + int(math.ceil(self._mem_size * 3 / 8))

    def  ttable_overlap_BRAMS(self, bram_size=2048, bram_word=8):
        """Return size of memory (BRAMS) for overlapping of tt rows."""
        #brams = (self._mem_size + bram_size / 2) / bram_size * bram_size
        rem = self._mem_size % bram_size
        res = self._mem_size - rem + bram_size
        print(res)
        brams = res / bram_size
        print(brams)
        mult = (1 + self._stateWidth/8) * 8
        print(mult)
        print(int(math.ceil(mult / bram_word)) * brams)
        return int(math.ceil(mult / bram_word)) * brams

    #def  ttable_overlap_mapping(self):
    # NOTE: removed due to using getTableByHeuristic and getTabelByGA
        """Map transitions to memory. Overlapping of rows is used. The 
           return value is vector of memory words. Character classes is
           used to reduce memory utilization."""

    # ---- Shared Decoder ---------------------- 
    def  sharedec_LUTs(self, char_classes):
        """Return size of shared decoder in LUTs."""

    def  sharedec_FFs(self, char_classes):
        """Return size of shared decoder in FFs."""

    # ---- Input encoder and output decoder ---- 
    def  encoder_size(self):
        """Return size of input encoder in LUTs."""
        if self._statistic.has_key("Encoder size") == True:
            return self._statistic["Encoder size"]
        else:
            self.encoder_HDL()
            return self._statistic["Encoder size"]

    def  encoder_HDL(self):
        """Return HDL description of input encoder."""
        inList = list()
        
        f = open(self._encoder_template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        
        for mset in self._inset:
            inList.append(mset)
            
        inList.sort()
        
        binList = [0] * len(inList)
        
        for i in range(0, len(inList)):
            binVal = bin(self._solution[inList[i]])

            newBin = str()
            
            for j in range(0, len(binVal)):
                if j > 1:
                    newBin += binVal[j]
                    
            binVal = newBin    
            
            newBin = str()
            
            if len(binVal) < self._stateWidth:
                for j in range(0, self._stateWidth - len(binVal)):
                    newBin += "0"
                    
            newBin += binVal
            
            binList[i] = newBin
         
        #wires = [list()] * self._stateWidth
        wires = list()
        #for i in range(0, len(binList)):
        #    for j in range(0, self._stateWidth):
        #        if binList[i][j] == "1":
        #            wires[j].append(i)
        for i in range (0, self._stateWidth):
            wire = list()
            for j in range (0, len(binList)):
                if binList[j][self._stateWidth - 1 - i] == '1':
                    wire.append(j)
            wires.append(wire)
                    
        for i in range(0, len(wires)):
            wires[i].sort()
            
        print(wires)
        wiresHDL = [str()] * self._stateWidth
        
        luts = 0
        stages = 0
        
        for i in range(0, len(wires)):
            if len(wires[i]) < 1:
                wiresHDL[i] = "bin(" + str(i) + ") <= '0';\n"
            else:
                wiresHDL[i] = "bin(" + str(i) + ") <= INPUT(" + str(wires[i][0]) + ")"
                for j in range(1, len(wires[i])):
                    wiresHDL[i] += " or INPUT(" + str(wires[i][j]) + ")"
                wiresHDL[i] += ";\n"
                luts += self._countLUTsForInpunts(len(wires[i]))
                stg = self._countLUTStagesForInpunts(len(wires[i]))
                if stg > stages:
                    stages = stg
                    
        self._statistic["Encoder size"] = luts
        self._statistic["Encoder stages"] = stages
        
        result = str()
        
        for i in range(0, len(wiresHDL)):
            result += wiresHDL[i]
        
        result = tmp[0] + str(self._index) + tmp[1] + str(len(self._inset)) + tmp[2] + str(self._stateWidth) + tmp[3] + str(self._index) + tmp[4] + str(self._index) + tmp[5] + result + tmp[6]
        
        return result
        
    def  decoder_size(self):
        """Return size of output decoder in LUTs."""
        if self._statistic.has_key("Decoder size") == True:
            return self._statistic["Decoder size"]
        else:
            self.decoder_HDL()
            return self._statistic["Decoder size"]
    
    def  decoder_HDL(self):
        """Return HDL description of ouput decoder."""
        outlist = list()
        
        f = open(self._decoder_template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        
        for set in self._outset:
            outlist.append(set)
            
        outlist.sort()
        
        luts = 0
        
        descriptionList = list()
        
        for i in range(0, len(outlist)):
            luts += self._countLUTsForInpunts(self._stateWidth)
            
            chrDec  = "    VD_" + str(i) + ": entity work.VALUE_DECODER\n" 
            chrDec += "    generic map(\n"
            chrDec += "        DATA_WIDTH => " + str(self._stateWidth) + ",\n"
            chrDec += "        VALUE      => " + str(outlist[i]) + "\n"
            chrDec += "    )\n"
            chrDec += "    port map(\n"
            chrDec += "        INPUT  => INPUT,\n"
            chrDec += "        OUTPUT => OUTPUT(" + str(i) + ")\n"
            chrDec += "    );\n\n"
            descriptionList.append(chrDec)
            
        result = str()
        
        for i in range(0, len(descriptionList)):
            result += descriptionList[i]
        
        self._statistic["Decoder size"] = luts
        
        result = tmp[0] + str(self._index) + tmp[1] + str(self._stateWidth) + tmp[2] + str(len(self._outset)) + tmp[3] + str(self._index) + tmp[4] + str(self._index) + tmp[5] + result + tmp[6]
        
        return result

    def memory_HDL(self, wrap = False, bram_size=2048, bram_word=8):
        """ Return HDL description of deterministic unit memory. """
        if (bram_size != 2048 or bram_word != 8):
            print("ERROR: WRONG BRAM SIZE - " + str(bram_size) + "x" + str(bram_word) + " - NOT IMPLEMENTED")
            raise Exception()
        
        #TODO: corect it
        #self._last = 5
            
        #size = int(math.ceil(self._last / bram_size))
        size = int(self.ttable_overlap_BRAMS() / 3)
        
        nearPwr = 1;
        while nearPwr < size:
            nearPwr = nearPwr * 2;
        
        msize = size * bram_size
        
        memoryTarget = [0] * msize
        memorySymbol = [0] * msize
        memoryFlags = [[0,0,0]] * msize
            
        for trans in self._automaton.transitions:
            addr = self._solution[trans[0]] ^ ord(self._automaton.alphabet[trans[1]].char)
            if wrap == True:
                addr = addr % msize
            if self._solution.has_key(trans[2]):
                memoryTarget[addr] = self._solution[trans[2]]
            else:
                memoryTarget[addr] = trans[2]
                
            memorySymbol[addr] = ord(self._automaton.alphabet[trans[1]].char)
            
            memoryFlags[addr][0] = 1
            if trans[2] in self._outset:
                memoryFlags[addr][1] = 1
            if trans[2] in self._automaton.final:
                memoryFlags[addr][2] = 1
        
        signals = "signal addr: std_logic_vector("+ str(int(math.log(bram_size, 2)) - 1) + " downto 0);\n"
        signals += "signal sel: std_logic_vector("+ str(int(math.log(nearPwr, 2)) - 1) + " downto 0);\n"
        for i in range(0, nearPwr):
            signals += "signal n_" + str(i) + ": std_logic;\n"
            signals += "signal f_" + str(i) + ": std_logic;\n"
            signals += "signal v_" + str(i) + ": std_logic;\n"
            signals += "signal target_" + str(i) + ": std_logic_vector(" + str(self._stateWidth - 1) + " downto 0);\n"
            signals += "signal symbol_" + str(i) + ": std_logic_vector(7 downto 0);\n"
        
        logic = list()
        for i in range(0, size):
            logic.append(self._map2memory(memoryTarget, memorySymbol, memoryFlags, i*bram_size, (i+1)*bram_size, i))
            
        targetMux = str()
        symbolMux = str()
        nMux = str()
        fMux = str()
        vMux = str()
        
        targetMux = "with sel select\n"
        auxStr = ""
        for i in range(0, nearPwr):
            auxStr += "target_" + str(i) + " when conv_std_logic_vector(" + str(i) + ", " + str(int(math.log(nearPwr, 2))) +"),\n"
        targetMux +="TARGET <= " + auxStr + "(others => '0') when others;\n"    
        
        symbolMux = "with sel select\n"
        auxStr = ""
        for i in range(0, nearPwr):
            auxStr += "symbol_" + str(i) + " when conv_std_logic_vector(" + str(i) + ", " + str(int(math.log(nearPwr, 2))) +"),\n"
        symbolMux +="SYMBOL <= " + auxStr + "(others => '0') when others;\n" 
        
        nMux = "with sel select\n"
        auxStr = ""
        for i in range(0, nearPwr):
            auxStr += "n_" + str(i) + " when conv_std_logic_vector(" + str(i) + ", " + str(int(math.log(nearPwr, 2))) +"),\n"
        nMux +="N <= " + auxStr + "(others => '0') when others;\n" 
        
        fMux = "with sel select\n"
        auxStr = ""
        for i in range(0, nearPwr):
            auxStr += "f_" + str(i) + " when conv_std_logic_vector(" + str(i) + ", " + str(int(math.log(nearPwr, 2))) +"),\n"
        fMux +="F <= " + auxStr + "(others => '0') when others;\n" 
        
        vMux = "with sel select\n"
        auxStr = ""
        for i in range(0, nearPwr):
            auxStr += "v_" + str(i) + " when conv_std_logic_vector(" + str(i) + ", " + str(int(math.log(nearPwr, 2))) +"),\n"
        vMux +="V <= " + auxStr + "(others => '0') when others;\n" 
        
        logic.append(targetMux)
        logic.append(symbolMux)
        logic.append(nMux)
        logic.append(fMux)
        logic.append(vMux)
        logic.append("addr <= MEMORY_ADDR(" + str(int(math.log(bram_size, 2)) - 1) + " downto 0);\n")
        logic.append("sel <= MEMORY_ADDR(" + str(int(math.log(bram_size, 2)) + int(math.log(nearPwr, 2)) - 1) + " downto " + str(int(math.log(bram_size, 2))) +");\n")
        
        f = open(self._memory_template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        f.close()
        
        logicStr = ""
        for i in range(0, len(logic)):
            logicStr += logic[i]
        
        result = tmp[0] + str(self._index) + tmp[1] + str(self._stateWidth) + tmp[2] + str(8) + tmp[3] + str(max([self._stateWidth, 8])) + tmp[4] + str(self._index) + tmp[5] + str(self._index) + tmp[6] +  signals + tmp[7] + logicStr + tmp[8]
        
        return result
            
    def _map2memory(self, memoryTarget, memorySymbol, memoryFlags, start, stop, index):
        targetBRAMs = int(math.ceil(self._stateWidth / 8))
        if targetBRAMs < 2:
            targetBRAMs = 2
            
        hexWidth = targetBRAMs * 2;
        
        targetHexs = list()
        symbolHexs = list()
        for i in range(start, stop):
            # preparing target part
            pHex = hex(memoryTarget[i])
            
            sHex = pHex[2:];
            
            
            if len(sHex) < hexWidth:
                nulStr = ""
                for j in range(0, hexWidth - len(sHex)):
                    nulStr += "0"
                rHex = nulStr + sHex
            else:
                rHex = sHex;
                    
            row = list()
            for i in range(0, hexWidth, 2):
                val = rHex[i] + rHex[i+1]
                row.append(val)
                    
            row.reverse()
            targetHexs.append(row)
            
            #preparing symbol part
            pHex = hex(memorySymbol[i])
            
            sHex = pHex[2:];
            
            if len(sHex) < 2:
                nulStr = ""
                for j in range(0, 2 - len(sHex)):
                    nulStr += "0"
                rHex = nulStr + sHex
            else:
                rHex = sHex
        
            symbolHexs.append(rHex)
            
        validHexs = list()
        nfaHexs = list()
        finalHexs = list()
        for i in range(0, int((stop - start)/256)):
            cntV = long(0)
            cntN = long(0)
            cntF = long(0)
            for j in range(0, 256):
                numV = long(memoryFlags[i*256+j][0]) << j
                numN = long(memoryFlags[i*256+j][1]) << j
                numF = long(memoryFlags[i*256+j][2]) << j
                cntV = cntV | numV
                cntN = cntN | numN
                cntF = cntF | numF
            
            pHex = hex(cntV)
            
            sHex = pHex[2:];
            
            if len(sHex) < 64:
                nulStr = ""
                for j in range(0, 64 - len(sHex)):
                    nulStr += "0"
                rHex = nulStr + sHex
            else:
                rHex = sHex
            
            rrHex = ""
            for j in range(0, len(rHex) - 1):
                rrHex += rHex[i]
            
            validHexs.append(rrHex)
            
            pHex = hex(cntN)
            
            sHex = pHex[2:];
            
            if len(sHex) < 64:
                nulStr = ""
                for j in range(0, 64 - len(sHex)):
                    nulStr += "0"
                rHex = nulStr + sHex
            else:
                rHex = sHex
            
            rrHex = ""
            for j in range(0, len(rHex) - 1):
                rrHex += rHex[i]
            
            nfaHexs.append(rrHex)
            
            pHex = hex(cntF)
            
            sHex = pHex[2:];
            
            if len(sHex) < 64:
                nulStr = ""
                for j in range(0, 64 - len(sHex)):
                    nulStr += "0"
                rHex = nulStr + sHex
            else:
                rHex = sHex
            
            rrHex = ""
            for j in range(0, len(rHex) - 1):
                rrHex += rHex[i]
            
            finalHexs.append(rrHex)
            
        targetHexsFinal = list()
        for i in range(0, int((stop - start)*8/256)):
            hexStr = [str()]*len(targetHexs[0])
            for j in range(0, 32):
                for k in range(0, len(targetHexs[0])):
                    hexStr[k] = targetHexs[i*32+j][k] + hexStr[k]
            targetHexsFinal.append(hexStr)
        
        symbolHexsFinal = list()
        for i in range(0, int((stop - start)*8/256)):
            hexStr = str()
            for j in range(0, 32):
                    hexStr = symbolHexs[i*32+j] + hexStr
            symbolHexsFinal.append(hexStr)
            
        
        logicList = list()
        targetBram0 = str()
        targetBram1 = str()
        symbolBram = str()
        targetBram0 = "BRAM_TARGET_0_" + str(index) + ": RAMB16_S9_S9\n"
        targetBram0 += "  -- synthesis translate_off\n"
        targetBram0 += "  generic map (\n"
        for i in range(0, len(targetHexsFinal)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            targetBram0 += "INIT_" + sHex + " => X\"" + targetHexsFinal[i][0]
            if i != len(targetHexsFinal) - 1:
                targetBram0 += "\",\n"
            else:
                targetBram0 += "\"\n"
        for i in range(0, len(nfaHexs)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            targetBram0 += "INITP_" + sHex + " => X\"" + nfaHexs[i]
            if i != len(nfaHexs) - 1:
                targetBram0 += "\",\n"
            else:
                targetBram0 += "\"\n"
        targetBram0 += ")\n"
        targetBram0 += "-- synthesis translate_on\n"
        targetBram0 += "port map (\n"
        targetBram0 += "DIA     => X\"00\",\n"
        targetBram0 += "DIPA    => '0',\n"
        targetBram0 += "ADDRA   => addr\n"
        targetBram0 += "ENA     => MEMORY_RQ\n"
        targetBram0 += "WEA     => '0',\n"
        targetBram0 += "SSRA    => '0',\n"
        targetBram0 += "CLKA    => CLK,\n"
        targetBram0 += "DOA     => target_"+str(index)+"(7 downto 0),\n"
        targetBram0 += "DOPA    => n_" + str(index) + ",\n"
        targetBram0 += "DIB     => X\"00\",\n"
        targetBram0 += "DIPB    => '0',\n"
        targetBram0 += "ADDRB   => addr\n"
        targetBram0 += "ENB     => '0'\n"
        targetBram0 += "WEB     => '0',\n"
        targetBram0 += "SSRB    => '0',\n"
        targetBram0 += "CLKB    => CLK,\n"
        targetBram0 += "DOB     => open,\n"
        targetBram0 += "DOPB    => open,\n"
        targetBram0 += ");\n\n"
        
        targetBram1 = "BRAM_TARGET_1_" + str(index) + ": RAMB16_S9_S9\n"
        targetBram1 += "  -- synthesis translate_off\n"
        targetBram1 += "  generic map (\n"
        for i in range(0, len(targetHexsFinal)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            targetBram1 += "INIT_" + sHex + " => X\"" + targetHexsFinal[i][1]
            if i != len(targetHexsFinal) - 1:
                targetBram1 += "\",\n"
            else:
                targetBram1 += "\"\n"
        for i in range(0, len(finalHexs)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            targetBram1 += "INITP_" + sHex + " => X\"" + finalHexs[i]
            if i != len(nfaHexs) - 1:
                targetBram1 += "\",\n"
            else:
                targetBram1 += "\"\n"
        targetBram1 += ")\n"
        targetBram1 += "-- synthesis translate_on\n"
        targetBram1 += "port map (\n"
        targetBram1 += "DIA     => X\"00\",\n"
        targetBram1 += "DIPA    => '0',\n"
        targetBram1 += "ADDRA   => addr\n"
        targetBram1 += "ENA     => MEMORY_RQ\n"
        targetBram1 += "WEA     => '0',\n"
        targetBram1 += "SSRA    => '0',\n"
        targetBram1 += "CLKA    => CLK,\n"
        targetBram1 += "DOA     => target_"+str(index)+"(15 downto 8),\n"
        targetBram1 += "DOPA    => f_" + str(index) + ",\n"
        targetBram1 += "DIB     => X\"00\",\n"
        targetBram1 += "DIPB    => '0',\n"
        targetBram1 += "ADDRB   => addr\n"
        targetBram1 += "ENB     => '0'\n"
        targetBram1 += "WEB     => '0',\n"
        targetBram1 += "SSRB    => '0',\n"
        targetBram1 += "CLKB    => CLK,\n"
        targetBram1 += "DOB     => open,\n"
        targetBram1 += "DOPB    => open,\n"
        targetBram1 += ");\n\n"
        
        symbolBram = "BRAM_SYMBOL_" + str(index) + ": RAMB16_S9_S9\n"
        symbolBram += "  -- synthesis translate_off\n"
        symbolBram += "  generic map (\n"
        for i in range(0, len(symbolHexsFinal)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            symbolBram += "INIT_" + sHex + " => X\"" + symbolHexsFinal[i]
            if i != len(symbolHexsFinal) - 1:
                symbolBram += "\",\n"
            else:
                symbolBram += "\"\n"
        for i in range(0, len(validHexs)):
            hexNum = hex(int(i))
            sHex = hexNum[2:];
            if len(sHex) < 2:
                sHex = "0" + sHex
            symbolBram += "INITP_" + sHex + " => X\"" + validHexs[i]
            if i != len(nfaHexs) - 1:
                symbolBram += "\",\n"
            else:
                symbolBram += "\"\n"
        symbolBram += ")\n"
        symbolBram += "-- synthesis translate_on\n"
        symbolBram += "port map (\n"
        symbolBram += "DIA     => X\"00\",\n"
        symbolBram += "DIPA    => '0',\n"
        symbolBram += "ADDRA   => addr\n"
        symbolBram += "ENA     => MEMORY_RQ\n"
        symbolBram += "WEA     => '0',\n"
        symbolBram += "SSRA    => '0',\n"
        symbolBram += "CLKA    => CLK,\n"
        symbolBram += "DOA     => symbol_"+str(index)+",\n"
        symbolBram += "DOPA    => v_" + str(index) + ",\n"
        symbolBram += "DIB     => X\"00\",\n"
        symbolBram += "DIPB    => '0',\n"
        symbolBram += "ADDRB   => addr\n"
        symbolBram += "ENB     => '0'\n"
        symbolBram += "WEB     => '0',\n"
        symbolBram += "SSRB    => '0',\n"
        symbolBram += "CLKB    => CLK,\n"
        symbolBram += "DOB     => open,\n"
        symbolBram += "DOPB    => open,\n"
        symbolBram += ");\n\n"
        
        return targetBram0 + targetBram1 + symbolBram
            

    # --------------- Det Unit ----------------
    def  detunit_HDL(self):
        """Return HDL description of deterministic unit."""
        
        signalList = list()
        descriptionList = list()
        
        sameFinal = dict()
        for fstate in self._automaton.final:
            if sameFinal.has_key(self._automaton.states[fstate].get_regexp_number()) == True:
                sameFinal[self._automaton.states[fstate].get_regexp_number()].add(fstate)
            else:
                sameFinal[self._automaton.states[fstate].get_regexp_number()] = set()
                sameFinal[self._automaton.states[fstate].get_regexp_number()].add(fstate)
        
        self._fStateNum = len(sameFinal)
        self._luts = 0
        signalList.append("    signal bitmap_in : std_logic_vector(" + str(len(sameFinal)) + " - 1 downto 0);\n")
        for final in sameFinal.keys():
            signalList.append("    signal final_" + str(final) + " : std_logic;\n")
            if len(sameFinal[final]) == 1:
                state = sameFinal[final].pop()
                descriptionList.append("    final_" + str(final) + " <= state_out_" + str(state) + ";\n")
            else:
                first = True
                self._luts += self._countLUTsForInpunts(len(sameFinal[final]))
                for pfinal in sameFinal[final]:
                    if first == True:
                        text = "    final_" + str(final) + " <= state_out_" + str(pfinal)
                        first = False
                    else:
                        text += " or state_out_" + str(pfinal)
                text += ";\n"
                descriptionList.append(text)
                
        sfKeys = sameFinal.keys()
        for i in range(0, len(sfKeys)):
            descriptionList.append("    bitmap_in(" + str(i) + ") <= final_" + str(sfKeys[i]) + ";\n")
            
        f = open(self._template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        f.close()
        
        result = tmp[0] + str(self._index) + tmp[1] + str(8) + tmp[2] + str(len(self._inset)) + tmp[3] + str(len(self._outset)) + tmp[4] + str(self._stateWidth) + tmp[5] + str(max([self._stateWidth, 8])) + tmp[6] + str(len(self._automaton.final)) + tmp[7] + str(self._index) + tmp[8] + str(self._index) + tmp[9] + "" + tmp[10] + str(self._index) + tmp[11] + str(self._index) + tmp[12] + str(self._index) + tmp[13] + "" + tmp[14] 
        
        return result

    def  detunit_LUTs(self):
        """Return estimation of consumed LUTs for the implementation of 
           deterministic unit."""

    # --------------- OverAll statistics ----------------
    def  num_of_LUTs(self):
        """Return estimation of consumed LUTs for the whole 
           deterministic unit."""

    def  num_of_FFs(self):
        """Return estimation of consumed FFs for the whole 
           deterministic unit."""

    def  num_of_BRAMS(self):
        """Return estimation of consumed BRAMs for the whole 
           deterministic unit."""

    def  size_of_memory(self):
        """Return estimation of consumed memory (in Bytes) for the 
        whole deterministic unit"""
        
    def report_memory(self):
        return self.ttable_overlap()
        
    def report_logic(self, bram_size=2048, bram_word=8):
        lut = self.encoder_size() + self.decoder_size() + self._stateWidth
        ff = 8 + len(self._automaton.final)
        mem = self.ttable_overlap_BRAMS(bram_size, bram_word)
        return (lut, ff, mem)
    
    def  frequency_estimation(self):
        """Estimation of frequency."""
        
    def _countLUTsForInpunts(self, inputs):
        i = inputs
        res = 0;
        if i <= self._LUTInputs:
            res = 1;
        else:
            while i > self._LUTInputs:
                if i/self._LUTInputs == round(i/self._LUTInputs):
                    res = res + int(round((i / self._LUTInputs), 0))
                    i = int(round((i / self._LUTInputs), 0))
                else:
                    res = res + int(round((i / self._LUTInputs) + 0.5, 0))
                    i = int(round((i / self._LUTInputs) + 0.5, 0))
            res = res + 1;
        return res
                
    def _countLUTStagesForInpunts(self, inputs):
        i = inputs
        res = 0;
        if i <= self._LUTInputs:
            res = 1;
        else:
            while i > self._LUTInputs:
                if i/self._LUTInputs == round(i/self._LUTInputs):
                    res = res + 1
                    i = int(round((i / self._LUTInputs), 0))
                else:
                    res = res + 1
                    i = int(round((i / self._LUTInputs) + 0.5, 0))
            res = res + 1;
        return res


class nfa_nondet_part(clark_nfa.clark_nfa):
    """NonDeterministic part of NFA. Based on Clark's NFA implementation."""
    def __init__(self):
        clark_nfa.clark_nfa.__init__(self)
        self.width = 8
        self.template = "templates/vhdl/nondet_part.vhd"
        self._statistic = dict()
        self._LUTInputs = 6
        self._inset = set()
        self._outset = set()
        
    def  get_HDL(self, useBram = False):
        """Return HDL description of NFA unit implemented in Clark's approach."""
        f = open(self.template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        
        self._useBram = useBram
        self._luts = 0
        
        chdec = self._get_char_dec_HDL()
        logic = self._get_logic_HDL()        
        final = self._get_final_HDL()
        ioset = self._get_io_HDL()
        
        dataSignal = chdec[0] + logic[0] + final[0] + ioset[0]
        dataImplementation = chdec[1] + logic[1] + final[1] + ioset[1]
        
        textSignal = "-- --------------- GENERATED BY NFA_SPLIT.PY ----------------\n"
        textImplementation = "-- --------------- GENERATED BY NFA_SPLIT.PY ----------------\n"
        
        for element in dataSignal:
            textSignal += element
        for element in dataImplementation:
            textImplementation += element
        textSignal += "-- --------------- END ----------------\n"
        textImplementation += "-- --------------- END ----------------\n"
        
        result = tmp[0] + str(self.width) + tmp[1] + str(self._fStateNum) + tmp[2] + str(len(self._inset)) + tmp[3] + str(len(self._outset)) + tmp[4] + textSignal + tmp[5] + textImplementation + tmp[6]
        return result
        
    def _get_io_HDL(self):
        signalList = list()
        descriptionList = list()
        
        self._insetMap = dict()
        self._outsetMap = dict()
        
        index = 0
        for state in self._inset:
            signalList.append("    signal inset_" + str(state) + " : std_logic;\n")
            descriptionList.append("    inset_" + str(state) + " <= INPUT(" + str(index) + ");\n")
            self._insetMap[index] = state
            index += 1
            
        index = 0
        for state in self._outset:
            signalList.append("    signal outset_" + str(state) + " : std_logic;\n")
            descriptionList.append("    OUTPUT(" + str(index) + ") <= outset_" + str(state) + ";\n")
            descriptionList.append("    outset_" + str(state) + " <= state_out_" + str(state) + ";\n")
            self._outsetMap[index] = state
            index += 1
            
        return (signalList, descriptionList)
        
    def _get_logic_HDL(self):
        """ Return HDL description of states and transitions as (signals, description). """
        signalList = list()
        descriptionList = list()
        
        for state in self._automaton.states.keys():
            signalList.append("    signal state_in_" + str(state) + " : std_logic;\n")
            signalList.append("    signal state_out_" + str(state) + " : std_logic;\n")
            if state == self._automaton.start:
                text  =  "    STATE_" + str(state) + ": entity work.STATE\n"
                text +=  "    generic map(\n"
                text +=  "        DEFAULT     => '1'\n"
                text +=  "    )\n"
                text +=  "    port map(\n"
                text +=  "        CLK    => CLK,\n"
                text +=  "        RESET  => local_reset,\n"
                text +=  "        INPUT  => '0',\n"
                text +=  "        WE     => we,\n"
                text +=  "        OUTPUT => state_out_" + str(state) + "\n"
                text +=  "    );\n\n"
            else:
                text  =  "    STATE_" + str(state) + ": entity work.STATE\n"
                text +=  "    generic map(\n"
                text +=  "        DEFAULT     => '0'\n" 
                text +=  "    )\n"
                text +=  "    port map(\n"
                text +=  "        CLK    => CLK,\n"
                text +=  "        RESET  => local_reset,\n"
                text +=  "        INPUT  => state_in_" + str(state) + ",\n"
                text +=  "        WE     => we,\n"
                text +=  "        OUTPUT => state_out_" + str(state) + "\n"
                text +=  "    );\n\n"
            descriptionList.append(text)
        
        inputTransitions = dict()
        for transition in self._automaton.transitions:
            if inputTransitions.has_key(transition[2]) == True:
                inputTransitions[transition[2]].add((transition[0], transition[1]))
            else:
                inputTransitions[transition[2]] = set()
                inputTransitions[transition[2]].add((transition[0], transition[1]))
                                
        for state in inputTransitions.keys():
            if len(inputTransitions[state]) == 1 and state not in self._inset:
                data = inputTransitions[state].pop()
                descriptionList.append("    state_in_" + str(state) + " <= state_out_" + str(data[0]) + " and symbol_" + str(data[1]) + ";\n")
                self._luts += self._countLUTsForInpunts(2)
            else:
                text = str()
                first = True
                for transition in inputTransitions[state]:
                    if first == True:
                        text += "    state_in_" + str(state) + " <= (state_out_" + str(transition[0]) + " and symbol_" + str(transition[1]) + ")"
                        first = False
                    else:
                        text += " or (state_out_" + str(transition[0]) + " and symbol_" + str(transition[1]) + ")"
                if state in self._inset:
                    text += " or inset_" + str(state)
                    self._luts += self._countLUTsForInpunts(2*len(inputTransitions[state]) + 1)
                else:
                    self._luts += self._countLUTsForInpunts(2*len(inputTransitions[state]))
                text += ";\n"
                descriptionList.append(text)
        return (signalList, descriptionList)
            
# -----------------------------------------------------------------------
#
#   NFA Split architecture - reduction of logic resources 
#
# -----------------------------------------------------------------------

class nfa_split(b_nfa):
    """NFA split class - new representaiton of NFA automaton."""
    def __init__(self):
        b_nfa.__init__(self)

        self._det_parts  = list();  # deterministic parts - list of
                                    # nfa_det_part classes
        self._nfa_part   = None;    # non-deterministic part is 
                                    # defined as a reference to Clark
                                    # NFA approach
        self._Transitions = dict();   # Contains transition table
        self._EndStates = set();      # Contains list of endstates.
        self._Alphabet = dict();      # Each symbol from transition table is set
                                      # of n-tuples (n character is accepted 
                                      # per step)
        self._StartState = 0;
        
        self._allNoncollSets = None
        self.statesCollisions = dict()
        self._statistics = dict() # statistics, key - string name
        self.template = "templates/vhdl/unit.vhd"
        self._reduced = False

    def _transform_nfa_data(self):
        """ Create nfa_split internal representation of nfa_data """
        #TODO: may be removed in future
        self._Alphabet = self._automaton.alphabet
        self._EndStates = self._automaton.final
        self._StartState = self._automaton.start
        
        for transition in self._automaton.transitions:
            Symbol = transition[1]
    
            Source = transition[0]
            TargetSet = set()
            #Target set is the set of states which are targeted by Symbol and 
            #from SourceSet
    
            TargetSet.add(transition[2])
             
            #if SourceSet is not in transition dictionary (no outgoing transition 
            #existed before)
            if(Source not in self._Transitions.keys()):                 
                #Source State does not exist yet
                #create another dictionary to map symbols to target set
                StateRecord = dict()         
                StateRecord[Symbol] = TargetSet                   
    
                #add a newly created line to the transition table
                self._Transitions[Source] = StateRecord 
    
            #if line exists, but symbol does not exist 
            elif(Symbol not in self._Transitions[Source]):                  
                  #Add target state and symbol into dictionary 
                  #(into the line for source state)
                  self._Transitions[Source][Symbol] = TargetSet 
    
            else:   
                 #If both symbol and source state exists, just add another target 
                 #state into target set
                 self._Transitions[Source][Symbol].add(TransTuple[2])
    
    # ----------- Deterministic parts creation --------
    def create_det_parts(self, num=1, method=0, useCharClasses = False):
        """Create deterministic parts of NFA. Maximal number of
        parts can be specified. Table mapping type can be specified. 
        Use 0 for heuristic, 1 for GA - experimental"""
        if self._automaton == None or len(self._automaton.transitions) == 0:
            return False
        
        if self._allNoncollSets == None:        
            self.remove_epsilons()
            #self._removeCharClasses()
            self._statistics["NFA States"] = len(self._automaton.states)
            self._statistics["NFA Transitions"] = len(self._automaton.transitions)
            
            dfa = b_dfa()
            dfa.create_from_nfa_data(self._automaton)
            dfa.determinise(True)
            
            self._statistics["DFA States"] = len(dfa._automaton.states)
            self._statistics["DFA Transitions"] = len(dfa._automaton.transitions)
            
            # Get for each state its collision states
            self.statesCollisions = dict()
            for state in self._automaton.states.keys():
                self.statesCollisions[state] = set()
            
            activeStatesSizeList = [0]*len(dfa._state_representation)
            
            for detState in range(0, len(dfa._state_representation)):
                activeStatesSizeList[detState] = len(dfa._state_representation[detState])
                for state in dfa._state_representation[detState]:
                    mappedStates = copy.deepcopy(dfa._state_representation[detState])
                    mappedStates.remove(state)
                    if len(mappedStates) > 0:
                        self.statesCollisions[state].update(mappedStates)
                        
            maxActiveStates = max(activeStatesSizeList) 
            self._statistics["Max Active States"] = maxActiveStates
            
            # free memory
            dfa = None
            activeStatesSizeList = None
            
            self._allNoncollSets = self._AllNonCollisionSets(self.statesCollisions)
        else:
            #self.remove_epsilons()
            self._det_parts = list()
            
        nonColSize = list()
        colSize = 0
        
        for i in range (0, min(num, len(self._allNoncollSets))):
            self._det_parts.append(nfa_det_part())
            nonColSize.append(len(self._allNoncollSets[i]))
            for state in self._automaton.states.keys():
                if state in self._allNoncollSets[i]:
                    self._det_parts[i]._automaton.states[state] = self._automaton.states[state]
            
            for fstate in self._automaton.final:
                if fstate in self._allNoncollSets[i]:
                    self._det_parts[i]._automaton.final.add(fstate)
            
            for transition in self._automaton.transitions:
                if transition[0] in self._allNoncollSets[i] and transition[2] in self._allNoncollSets[i]:
                    self._det_parts[i]._automaton.transitions.add(transition)
                    self._det_parts[i]._automaton.alphabet[transition[1]] = self._automaton.alphabet[transition[1]]
                if transition[0] in self._allNoncollSets[i] and transition[2] not in self._allNoncollSets[i]:
                    self._det_parts[i]._outset.add(transition[2])
                if transition[0] not in self._allNoncollSets[i] and transition[2] in self._allNoncollSets[i]:
                    self._det_parts[i]._inset.add(transition[0])
                    self._det_parts[i]._automaton.transitions.add(transition)
                    self._det_parts[i]._automaton.alphabet[transition[1]] = self._automaton.alphabet[transition[1]]
                    self._det_parts[i]._automaton.states[transition[0]] = self._automaton.states[transition[0]]
                    
            self._det_parts[i]._automaton.Flags["Deterministic"] = True
            self._det_parts[i]._automaton.Flags["Epsilon Free"] = True
            
            self._det_parts[i]._removeCharClasses()
            
            if useCharClasses == True:
                self._det_parts[i].reduce_alphabet()
                self._reduced = True
            else:
                self._reduced = False
            
            if method == 0:
                res = self._det_parts[i].getTableByHeuristic()
                self._det_parts[i]._solution = res[0]
                self._det_parts[i]._mem_size = res[1]
                print(res[1])
            if method == 1:
                res = None
                size = len(self._det_parts[i]._automaton.transitions)
                while res == None:
                    res = self._det_parts[i].getTableByGA(size)
                    if res == None:
                        size = 2*size
                self._det_parts[i]._solution = res[0]
                self._det_parts[i]._mem_size = res[1] 
            
        collSet = set()
        for i in range (min(num, len(self._allNoncollSets)), len(self._allNoncollSets)):
            collSet |= self._allNoncollSets[i]
            
        colSize = len(collSet)
        
        self._statistics["NCA Sizes"] = nonColSize
        self._statistics["N Size"] = colSize
            
        detSizeTrans = list()
        detSizeStates = list()
        detSizeInsets = list()
        detSizeOutsets = list()
        for i in range(0, len(self._det_parts)):
            detSizeTrans.append(len(self._det_parts[i]._automaton.transitions))
            detSizeStates.append(len(self._det_parts[i]._automaton.states))
            detSizeInsets.append(len(self._det_parts[i]._inset))
            detSizeOutsets.append(len(self._det_parts[i]._outset))
            self._det_parts[i]._index = i
        
        self._statistics["DFA Parts States"] = detSizeStates
        self._statistics["DFA Parts Transitions"] = detSizeTrans
        self._statistics["DFA Parts Inset"] = detSizeInsets
        self._statistics["DFA Parts Outset"] = detSizeOutsets
            
            
        self._nfa_part = nfa_nondet_part()
        
        self._nfa_part._inset       = set()     # set of starting states
        self._nfa_part._outset      = set()     # set of outgoing states
        
            
        for state in self._automaton.states.keys():
            if state in collSet:
                self._nfa_part._automaton.states[state] = self._automaton.states[state]
        
        for fstate in self._automaton.final:
            if fstate in collSet:
                self._nfa_part._automaton.final.add(fstate)
               
        for transition in self._automaton.transitions:
            if transition[0] in collSet and transition[2] in collSet:
                self._nfa_part._automaton.transitions.add(transition)
                self._nfa_part._automaton.alphabet[transition[1]] = self._automaton.alphabet[transition[1]]
            if transition[0] in collSet and transition[2] not in collSet:
                self._nfa_part._outset.add(transition[2])
                self._nfa_part._automaton.transitions.add(transition)
                self._nfa_part._automaton.alphabet[transition[1]] = self._automaton.alphabet[transition[1]]
            if transition[0] not in collSet and transition[2] in collSet:
                self._nfa_part._inset.add(transition[2])

        if useCharClasses == True:
            self._nfa_part.reduce_alphabet()
            self._reduced = True
        else:
            self._reduced = False

        self._statistics["NFA Part States"] = len(self._nfa_part._automaton.states)
        self._statistics["NFA Part Transitions"] = len(self._nfa_part._automaton.transitions)
        self._statistics["NFA2DFAs Target states"] = len(self._nfa_part._outset)
         
          
    # ----------------- Internal functions -----------
    def _GetCollisionCard(self, ModstatesCollisions):
        collisionCard = dict()
        for state in ModstatesCollisions.keys():
            collisionCard[state] = len(ModstatesCollisions[state])
        return collisionCard
        
    # heuristic 2 - for given state compute how much collisions will removed if we will use this state
    def _GetRemovedCollisions(self, state, ModstatesCollisions):
        collisions = ModstatesCollisions[state]
        num = 0
        for i in collisions:
            num += len(ModstatesCollisions[i])
        return num

    # find set of states with minimal number of collisions
    def _minSet(self, collisionCard):
        minimalSet = set()
        keys = collisionCard.keys()
        if len(collisionCard) > 0:
            #minimum = collisionCard[keys[0]]
            minimum = min(collisionCard.values())
            for k in collisionCard.keys():
                if collisionCard[k] < minimum:
                    minimum = collisionCard[i]
            for k in collisionCard.keys():
                if collisionCard[k] == minimum:
                    minimalSet.add(k)
        return minimalSet

    # find one best state - acording heuristic 2
    def _findMaximumHeur(self, minimalSet, ModstatesCollisions):
        maximalSet = set()
        maximum = self._GetRemovedCollisions(copy.deepcopy(minimalSet).pop(), ModstatesCollisions)
        for i in minimalSet:
            if self._GetRemovedCollisions(i, ModstatesCollisions) > maximum:
                maximum = self._GetRemovedCollisions(i, ModstatesCollisions)
        for i in minimalSet:
            if self._GetRemovedCollisions(i, ModstatesCollisions) == maximum:
                maximalSet.add(i)
        #if len(maximalSet) > 1:
            # do it recursive sometimes
        return maximalSet.pop()
            
    # remove collision from statesCollisions
    def _removeStateFromstatesCollisions(self, state, ModstatesCollisions):
        for k in ModstatesCollisions.keys():
            if state in ModstatesCollisions[k]:
                ModstatesCollisions[k].remove(state)
        return ModstatesCollisions
            
    # Compute set of states which aren't in collision
    # NOTE: func is chaging statesCollisions
    def _NonCollisionSet(self, ModstatesCollisions):
        nonCollision = set()
        collisionCard = self._GetCollisionCard(ModstatesCollisions)
        # we don't need to compute heuristic, etc. for states where are 0 collisios
        delset = set() 
        for i in ModstatesCollisions.keys():
            if collisionCard[i] == 0:
                nonCollision.add(i)
                delset.add(i)
        for i in delset:
            del ModstatesCollisions[i]
            del collisionCard[i]
        while len(ModstatesCollisions) > 0:
            minimalSet = self._minSet(collisionCard)
            if len(minimalSet) > 1:
                state = self._findMaximumHeur(minimalSet, ModstatesCollisions)
            else:
                state = minimalSet.pop()
            nonCollision.add(state)
            collisions = copy.deepcopy(ModstatesCollisions[state])
            for i in collisions:
                del ModstatesCollisions[i]
                del collisionCard[i]
                ModstatesCollisions = self._removeStateFromstatesCollisions(i, ModstatesCollisions)
            del ModstatesCollisions[state]
            del collisionCard[state]
            ModstatesCollisions = self._removeStateFromstatesCollisions(state, ModstatesCollisions)
            collisionCard = self._GetCollisionCard(ModstatesCollisions)
        return nonCollision
            
    def _modSet(self, workStatesCollisions, ModstatesCollisions, usedStates):
        workStatesCollisions = copy.deepcopy(ModstatesCollisions)
        for i in usedStates:
            del workStatesCollisions[i]
            workStatesCollisions = self._removeStateFromstatesCollisions(i, workStatesCollisions)
        return workStatesCollisions
        
    # compute all noncollision sets - sets are in collision, but elements of set are not
    def _AllNonCollisionSets(self, ModstatesCollisions):
        allSets = list()
        usedStates = set()
        workStatesCollisions = copy.deepcopy(ModstatesCollisions)
        noncoll = self._NonCollisionSet(workStatesCollisions)
        allSets.insert(len(allSets),noncoll)
        usedStates |= noncoll
        workStatesCollisions = self._modSet(workStatesCollisions, ModstatesCollisions, usedStates)
        while len(workStatesCollisions)> 0:
            noncoll = self._NonCollisionSet(workStatesCollisions)
            allSets.insert(len(allSets),noncoll)
            usedStates |= noncoll
            workStatesCollisions = self._modSet(workStatesCollisions, ModstatesCollisions, usedStates)
        return allSets
                        
    # ----------------- NFA estimation ---------------
    def nfa_decoder_size_LUTs(self):
        """Size of NFA shared decoder in LUTs - Clark."""

    def nfa_decoder_size_FFs(self):
        """Size of NFA shared decoder in FFs - Clark."""

    def nfa_size_LUTs(self):
        """Size of NFA part in LUTs - Clark."""

    def nfa_size_FFs(self):
        """Size of NFA part in FFs - Clark."""
    
    def report_nfa_logic(self):
        self._nfa_part.get_HDL()
        return self._nfa_part.report_logic()

    # ---------------- Det parts representation ------------
    def det_size_LUTs(self):
        """Amount of resources (LUTs) consumed by all deterministic 
        parts."""

    def det_size_FFs(self):
        """Amount of resources (FFs) consumed by all deterministic 
        parts."""

    def det_size_BRAMs(self, bram_size = 2048, bram_word = 8 ):
        """Return memory resources which are needed to represent
        all deterministic parts."""
        
    def report_dfa_logic(self, bram_size = 2048, bram_word = 8 ):
        lut = 0
        ff = 0
        mem = 0
        for dfa in self._det_parts:
             res = dfa.report_logic(bram_size, bram_word)
             lut += res[0]
             ff += res[1]
             mem += res[2]
        
        return (lut, ff, mem)

    # --------------- OverAll statistics ----------------
    def  num_of_LUTs(self):
        """Return estimation of all consumed LUTs."""

    def  num_of_FFs(self):
        """Return estimation of all consumed FFs."""

    def  num_of_BRAMS(self):
        """Return estimation of all consumed BRAMs."""

    def report_memory(self):
        """Return estimation of consumed memory (in Bytes)."""
        res = 0
        for dfa in self._det_parts:
             res += dfa.ttable_overlap()
        return res
        
    def report_logic(self, bram_size = 2048, bram_word = 8 ):
        nfa = self.report_nfa_logic()
        dfa = self.report_dfa_logic(bram_size, bram_word)
        preDecoderSize = 0
        if self._reduced == True:
            preDecoderSize = int(math.ceil((len(self._det_parts)+1) / 8))
        return (nfa[0] + dfa[0], nfa[1] + dfa[1], nfa[2] + dfa[2] + preDecoderSize)

    def  frequency_estimation(self):
        """Estimation of frequency."""
        
    # --------- OverAll HDL generation -------
    def get_HDL(self):
        """ Return dict of generated VHDL units"""
        hdls = dict()
        
        hdls["nondet_part.vhd"] = self._nfa_part.get_HDL()
        for i in range(0, len(self._det_parts)):
            hdls["det_part" + str(i) + ".vhd"] = self._det_parts[i].detunit_HDL()
            hdls["encoder" + str(i) + ".vhd"] = self._det_parts[i].encoder_HDL()
            hdls["decoder" + str(i) + ".vhd"] = self._det_parts[i].decoder_HDL()
            hdls["memory" + str(i) + ".vhd"] = self._det_parts[i].memory_HDL()
        
        f = open(self.template, "rb");
        blob = f.read()
        tmp = re.split("%\$%", blob)
        f.close()
            
        result = tmp[0] + str(8) + tmp[1] + str(len(self._automaton.final)) + tmp[2] + str(len(self._nfa_part._inset)) + tmp[3] + str(len(self._nfa_part._outset)) + tmp[4]
        
        hdls["nfa_split.vhd"] = result
        
        return hdls;

    def split_det_parts(self, k):
        """Split deterministic parts in order to reduce complexity of
           input encoder. Each part is split to k-parts)"""
        
        newDetPartsList = list()
        
        for i in range(0, len(self._det_parts)):
            parts = list()
            #self._det_parts[i]._createCharClasses()
            parts = self._det_parts[i].split_det_part(k)
            newDetPartsList += parts
        
        self._det_parts = newDetPartsList
        
        #for i in range(0, len(self._det_parts)):
        #    self._det_parts[i]._removeCharClasses()
        
        detSizeTrans = list()
        detSizeStates = list()
        detSizeInsets = list()
        detSizeOutsets = list()
        for i in range(0, len(self._det_parts)):
            detSizeTrans.append(len(self._det_parts[i]._automaton.transitions))
            detSizeStates.append(len(self._det_parts[i]._automaton.states))
            detSizeInsets.append(len(self._det_parts[i]._inset))
            detSizeOutsets.append(len(self._det_parts[i]._outset))
            self._det_parts[i]._index = i
            res = self._det_parts[i].getTableByHeuristic()
            self._det_parts[i]._solution = res[0]
            self._det_parts[i]._mem_size = res[1]
        
        self._statistics["DFA Parts States"] = detSizeStates
        self._statistics["DFA Parts Transitions"] = detSizeTrans
        self._statistics["DFA Parts Inset"] = detSizeInsets
        self._statistics["DFA Parts Outset"] = detSizeOutsets
        
    def save(self, file):
        """ Save object using cPickle. """
        output = open(file, 'wb')
        cPickle.dump(self, output)
        output.close()
        
    def load(self, file):
        """ Load object using cPickle. """
        pkl_file = open(file, 'rb')
        data = cPickle.load(pkl_file)
        pkl_file.close()
        return data
        
if __name__ == '__main__':
    ns = nfa_split()
    Test0 = msfm_parser.msfm_parser()
    #Test0.set_text("/abc[d-f]/")
    Test0.load_file("rules/l7/selected/selected.pcre")
    #nfa0 = Test0.get_nfa()
    #ns.create_from_nfa_data(nfa0)
    ns.create_by_parser(Test0)
    #print(str(ns.get_automaton().transitions))
    #ns._automaton.Show("class.dot")
    #ns._removeCharClasses()
    #print(str(ns.get_automaton().transitions))
    #ns._automaton.Show("char.dot")
    #ns._createCharClasses()
    #ns._automaton.Show("newclass.dot")
    #print(str(ns.get_automaton().transitions))
    ns.create_det_parts()
    #print(ns._det_parts[0].detunit_HDL())
    print(str(ns._statistics))
    ns.split_det_parts(2)
    #print(ns._det_parts[0].detunit_HDL())
    print(str(ns._statistics))
    #print(str(ns.get_automaton().transitions))
    print(str(ns.report_logic()))
    for i in range(0, len(ns._det_parts)):
        print("DFA " + str(i) + ": " + str(ns._det_parts[i]._statistic["Encoder stages"]))