###############################################################################
#  nfa_data.py: Module for PATTERN MATCH - class for NFA representation
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vlastimil Kosar <ikosar@fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

import cPickle
import sym_char
from b_state import b_State

# ------------------------------------------------------------------
#
#    Data structure for NFA representation
#
# ------------------------------------------------------------------

class nfa_data:
    """A class to specify NFA automaton (Q,T,q0,Delta,F).
    Atributes:
    states      - Finite set of states.
    alphabet    - Symbols of alphabet.
    start       - ID of Start state.
    transitions - Transitions.
    final       - Final states.
    Flags       - Flags for specified properties.
    """
    def __init__(self):
        """ Constructor for initialisation of object atributes."""
        self.states      = dict();      # Finite set of states
        self.alphabet    = dict();      # Symbols of alphabet
        self.start       = -1;          # ID of Start state
        self.transitions = set();       # Transitions
        self.final       = set();       # Final states
        self.Flags       = dict();      # Flags for specified properties


    """NFA representation"""
    def SaveToFile(self, FileName):
        """ Save nfa_data to file (serialisation).
        Parameters:
        FileName - name of file in which nfa_data will be saved.
        Returns:
        True if success, False otherwise.
        """
        File = open(FileName,'w')
        cPickle.dump(self,File)
        File.close()
        return True

    def LoadFromFile(self, FileName):
        """ Load nfa_data from file (serialisation).
        Parameters:
        FileName - name of file from which nfa_data will be loaded.
        Returns:
        True if success, False otherwise.
        """
        File = open(FileName,'r')
        self = cPickle.load(File)
        File.close()
        return True

    def Show(self, FileName, sizeStr=" size=\"8,5\"\n"):
        """ Save graphviz dot file, representing graphical structure of nfa_data.
        Parameters:
        FileName - name of file into which nfa_data graphical representation will be saved.
        sizeStr - Size of resulting image. Set to " " to set unbounded dimensions. Format of
                  this parameter is / size="x,y"\n/ to set width to x inches and height to y inches.
        Returns:
        True if success, False otherwise.
        """
        # If object does not contain automaton, stop.
        if len(self.states) == 0:
            return False
        else:
            # Otherwise open file and save the representation.
            f = open(FileName,"w")

            #Print header of the dot file
            f.write("digraph \" Automat \" {\n    rankdir=LR;\n "+sizeStr);
            f.write("node [shape = doublecircle];\n");

            #Print end states as double circles
            for EndSt in self.final:
                f.write(self.states[EndSt].get_text())
                f.write(";\n");

            f.write("node [shape=circle];\n");

            #print all transitions. States are print as a circle
            for Source in self.transitions:
                f.write(self.states[Source[0]].get_text())
                f.write(" -> ")
                f.write(self.states[Source[2]].get_text())
                f.write(" [ label = \"")

                s = self.alphabet[Source[1]].get_text()
                i = 0
                modStr = str()

                # for better readability, like [^abc]
                if len(s) > 128:
                	s_a = []
                	for i in range(0, 256):
                		s_a.append(chr(i))
                	for c in s:
                		if c in s_a:
                			s_a.remove(c)
                	s = s_a
                i = 0
                # unprintable characters save in hexa
                while i < len(s):
   	                if (ord(s[i]) > 127 or ord(s[i]) < 30):
   	                    modStr = modStr + "\\" + hex(ord(s[i]))
   	                elif s[i] == '"':
   	                    modStr = modStr + "\\" + hex(ord(s[i]))
   	                elif s[i] == '\'':
   	                    modStr = modStr + "\\" + hex(ord(s[i]))
   	                else:
   	                    modStr = modStr + s[i]
   	                i = i + 1
                if len(self.alphabet[Source[1]].get_text()) > 128:
                	modStr = "[^" + modStr + "]"

                f.write(modStr)
                f.write("\" ];\n")
            f.write("}")
            f.close();
            return True

    def __repr__(self):
        """ Returns representation of object."""
        return "States: " + str(self.states) + "\nAlphabet: " + str(self.alphabet) + "\nStart: " + str(self.start) + "\nTransitions: " + str(self.transitions) + "\nFinal: " + str(self.final) + "\nFlags: " + str(self.Flags) + "\n"

    def __str__(self):
        """ Returns representation of object."""
        return repr(self)

    def ExportToFsm(self, FileName="automaton.fsm"):
	"""Save automaton to file in FSM format. Based on FSM man
	page: http://www2.research.att.com/~fsmtools/fsm/man4.html """

	fw = open(FileName, 'w') # file write
	t_s = {}                 # transitions sorted

	# sort transitions
	for s in self.states:
		t_s[s] = []
	for t in self.transitions:
		t_s[t[0]].append(t)

	states = list(sorted(self.states))
	states.remove(self.start)
	# save transitions for start state
	for t in t_s[self.start]:
		if isinstance(self.alphabet[t[1]], sym_char.b_Sym_char):
			fw.write(str(self.start) + ' ' + str(t[2]) \
			+ ' ' + str(ord(self.alphabet[t[1]].char)) + '\n')
		else :
			for c in self.alphabet[t[1]].charClass:
				fw.write(str(self.start) + ' ' + str(t[2]) \
				+ ' ' + str(ord(c)) + '\n')
		# check final state
		if self.start in self.final:
			fw.write(str(self.start) + '\n')
	# save transitions for other states
	for s in states:
		for t in t_s[s]:
			if isinstance(self.alphabet[t[1]], \
			sym_char.b_Sym_char):
				fw.write(str(s) + ' ' + str(t[2]) + ' ' \
				+ str(ord(self.alphabet[t[1]].char)) + '\n')
			else :
				for c in self.alphabet[t[1]].charClass:
					fw.write(str(s) + ' ' + str(t[2]) \
					+ ' ' + str(ord(c)) + '\n')
		# check final state
		if s in self.final:
			fw.write(str(s) + '\n')
	fw.close()

    def ImportFromFsm(self, FileName="automaton.fsm"):
	"""Load automaton from file in FSM format. Based on FSM man
	page: http://www2.research.att.com/~fsmtools/fsm/man4.html """

	fr = open(FileName, 'r') # file read

	# initialization
        self.states      = dict();      # Finite set of states
        self.alphabet    = dict();      # Symbols of alphabet
        self.start       = -1;          # ID of Start state
        self.transitions = set();       # Transitions
        self.final       = set();       # Final states
        self.Flags       = dict();      # Flags for specified properties

	# set alphabet
	for c in range(0, 256):
		self.alphabet[c] = chr(c)

	# first line indicating start state
	line = fr.readline()
	line = line.split()
	src = int(line[0])
	self.start = src
	self.states[src] = b_State(mid = src)
	# line is transition
	if len(line) > 1:
		des = int(line[1])
		ch = int(line[2])
		if src != des:
			self.states[des] = b_State(mid = des)
		self.transitions.add((src, ch, des))
	# first line is start state and too final state
	# (line is final state)
	else :
		self.final.add(src)
		self.states[src]._rnum = src

	# from 2 line to EndOfFile
	for line in fr.readlines():
		line = line.split()
		src = int(line[0])
		if src not in self.states:
			self.states[src] = b_State(mid = src)
		# line is transition
		if len(line) > 1:
			des = int(line[1])
			ch = int(line[2])
			if des not in self.states:
				self.states[des] = b_State(mid = des)
			self.transitions.add((src, ch, des))
		# line is final state
		else :
			self.final.add(src)
			self.states[src]._rnum = src

	self.Flags["ImportFromFsm"] = True

###############################################################################
# End of File nfa_data.py                                                     #
###############################################################################
