###############################################################################
#  b_automaton.py: Module for PATTERN MATCH - base virtual automaton class
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vlastimil Kosar <ikosar@fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

import nfa_data
import b_automaton 
import b_state
import copy
from b_ptrn_match import b_ptrn_match
import sym_char
import sym_char_class
import sym_char_class_dfa
import sym_kchar

class b_Automaton(b_ptrn_match):
    """A base class for the pattern matching approaches based on finite automaton. """
    def __init__(self):
        """ Constructor - inits object atributes:
        _automaton - Automaton data
        _mapper    - Used to map state to (symbol, target) list
        _statistic - Dictionary of statistic informations
        """
        b_ptrn_match.__init__(self)
        self._automaton = nfa_data.nfa_data() # Automaton data
        self._mapper = None                   # Used to map state to (symbol, target) list
        self._statistic = dict()              # Dictionary of statistic informations
 
    def create_from_nfa_data(self, nfa, safe = True):
        """Create automaton from nfa_data object.
           Parameters:
           nfa - nfa_data object, from which automaton is created.
           safe - if True perform deep copy, otherwise set reference. Default value is True.
        """
        if safe:
            self._automaton = copy.deepcopy(nfa)
        else:
            self._automaton = nfa

    def create_by_parser(self, nfa_parser_class):
        """This function is used to create automaton from set of regular expressions. 
        Parameters:
        nfa_pareser_class - an instation of nfa_parser class
        Returns:
        False if creation of automaton failed or True if creation was successful."""
        # Check if there are some regular expressions to parse.
        if nfa_parser_class.num_lines == 0:
            return False
        
        # get an automaton from nfa parser
        self._automaton = nfa_parser_class.get_nfa()
        
        # Check if we got the automaton
        if self._automaton == None:
            return False
        
        # Check if we have more than 1 regular expression
        if nfa_parser_class.num_lines > 1:
            # If there are, parse them
            while nfa_parser_class.next_line() == True:
                # get an automaton from nfa parser
                nfa = nfa_parser_class.get_nfa()
                # Check if we got the automaton
                if nfa == None:
                    print("WARNING: Regular expresion can't be parsed.")
                else:
                    # Join current automaton with parsed one
                    self.join(nfa, False)
            return True 
        else:
            return True

    def get_automaton(self, safe = True):
        """
        Return nfa_data object from the automaton.
        Parameters:
        safe - if True return deep copy, otherwise return reference. Default value is True.
        """
        if safe:
            return copy.deepcopy(self._automaton)
        else:
            return self._automaton
    
    def join(self, nfa, modify_reg_exp_num = False):
        """Join two automata. Joins current automaton with second. Current automaton is modified.
        Parameters:
        nfa - nfa_data object which contains second automaton.
        modify_reg_exp_num - if set to True, regular expression numbers in final states are modified to not collide with current automaton numbers. The formula is NEW_NUMBER = MAX_CURRENT_NUMBER + OLD_NUMBER + 1. If set to False, regular expression numbers in final states are uneffected.
        Flags:
        Sets flags Deterministic and Epsilon Free to False.
        """
        # Find max state number
        max_state_num = self._automaton.start
        for i in self._automaton.states.keys():
            if self._automaton.states[i].get_id() > max_state_num:
                max_state_num = self._automaton.states[i].get_id()
        max_state_num += 1        
        
        # Creates mapping between alphabet char and its number. Also find max alphabet number.
        alphabetBack = dict()
        max_alphabet_num = -1
        for i in self._automaton.alphabet.keys():
            alphabetBack[self._automaton.alphabet[i]] = self._automaton.alphabet[i].get_id()
            if self._automaton.alphabet[i].get_id() > max_alphabet_num:
                max_alphabet_num = self._automaton.alphabet[i].get_id()
        max_alphabet_num += 1
        
        # Creates new states for states of second automaton.
        for i in nfa.states.keys():
            self._automaton.states[max_state_num + nfa.states[i].get_id()] = b_state.b_State(max_state_num + nfa.states[i].get_id(), nfa.states[i].get_regexp_number())
        
        # dictionary for transforming the alphabet.
        transform_alphabet = dict()
        
        # Join two alphabets, modify alphabet numbers if needed.
        for i in nfa.alphabet.keys():
            # If char is in current alphabet, just add to transformation dictionary.
            if alphabetBack.has_key(nfa.alphabet[i]) == True:
                transform_alphabet[i] = alphabetBack[nfa.alphabet[i]]
            else:
                # If not, add to current alphabet and add to transformation dictionary.
                self._automaton.alphabet[max_alphabet_num] = copy.deepcopy(nfa.alphabet[i])
                self._automaton.alphabet[max_alphabet_num]._id = max_alphabet_num
                alphabetBack[self._automaton.alphabet[max_alphabet_num]] = self._automaton.alphabet[max_alphabet_num].get_id()
                transform_alphabet[i] = alphabetBack[nfa.alphabet[i]]
                max_alphabet_num += 1
                
        # Add transitions from second automaton to current one. Modify state and alphabet numbers.
        for transition in nfa.transitions:
            self._automaton.transitions.add((max_state_num + transition[0], transform_alphabet[transition[1]], max_state_num + transition[2]))
        
        # Add epsilon transition from start state of current automaton to old start state of second automaton.    
        self._automaton.transitions.add((self._automaton.start, -1, max_state_num + nfa.start))
        
        # If reg. exp. numbers should be modified, find max reg. exp. number.
        if modify_reg_exp_num == True:
            max_rnum = -1
            for state in self._automaton.final:
                if self._automaton.states[state]._rnum > max_rnum:
                    max_rnum = self._automaton.states[state]._rnum
            max_rnum += 1
        
        # Add final states from second automaton, modify reg. exp. numbers if needed.
        for state in nfa.final:
            self._automaton.final.add(max_state_num + state)
            if modify_reg_exp_num == True:
                self._automaton.states[max_state_num + state]._rnum = self._automaton.states[max_state_num + state]._rnum + max_rnum
        
        # Clear flags.
        self._automaton.Flags["Deterministic"] = False
        self._automaton.Flags["Epsilon Free"] = False
            
    def _epsilon_closure(self, state, StateOutSymbols):
        """ Compute epsilon closure for selected state.
        Parameters:
        state           - state from which epsilon closure is computed.
        StateOutSymbols - mapping between states and their transitions.
        Returns:
        Set containing epsilon closure for state."""
        Stack = list()

        #Specified state is first state to be processed
        Stack.append(state) 

        Closure = set()
        while len(Stack) != 0 :         #Until there are states to be processed
            ActState = Stack.pop()      #Pick one of them
            if ActState in Closure:
                continue
            Closure.add(ActState)       #and add it into the eps closure set
            
            #If there are any outgoing transition from this state
            if ActState in StateOutSymbols.keys():

                #and if there is epsilon outgoing transition
                #Add all targets of eps transitions into stack for the further 
                #processing
                for trans in StateOutSymbols[ActState]:
                    if trans[0] == -1:
                        Stack.append(trans[1])
                        
        return Closure
                        
    def remove_epsilons(self):
        """ Remove all epsilon transitions from automaton. Also removes all isolated, unreachable and blind states.
        Flags:
        Sets flag Epsilon Free to True."""
        # Dictionary mapping between states and their transitions.
        StateOutSymbols = dict()
        
        # Compute the mapping between states and their transitions.
        for transition in self._automaton.transitions:
            if StateOutSymbols.has_key(transition[0]) == True:
                StateOutSymbols[transition[0]].add((transition[1], transition[2]))
            else:
                StateOutSymbols[transition[0]] = set()
                StateOutSymbols[transition[0]].add((transition[1], transition[2]))
                
        # For all states remove epsilon transitions.  
        for state in StateOutSymbols.keys():
            Clos = self._epsilon_closure(state, StateOutSymbols) # compute eps closure for the state
            Clos.remove(state)                  # Outgoing trans from specific state 
                                                # do not have to be transformed (they
                                                # are already in right state)
                                                
            #If there is an intersection of closure and endStates, add current state to end states and update reg. exp. number.
            interEnd = Clos.intersection(self._automaton.final)
            if len(interEnd) != 0:
                for fstate in interEnd:
                    self._automaton.states[state]._rnum = self._automaton.states[fstate]._rnum
                self._automaton.final.add(state)
                
            if len(Clos) != 0:                # If there are any other states
                for St in Clos:
                    #Tests if St has any outgoing transition
                    if St in StateOutSymbols.keys():
                        #Test all of their outgouing transitions, add new transitions, update StateOutSymbols
                        for transition in StateOutSymbols[St]:
                            self._automaton.transitions.add((state, transition[0], transition[1]))
                            if StateOutSymbols.has_key(state) == True:
                                StateOutSymbols[state].add((transition[0], transition[1]))
                            else:
                                StateOutSymbols[state] = set()
                                StateOutSymbols[state].add((transition[0], transition[1]))
                                
        # select all eps transitions
        epsTransitions = set()
        for transition in self._automaton.transitions:
            if transition[1] == -1:
                epsTransitions.add(transition)
                
        # remove all eps transitions        
        for transition in epsTransitions:
            self._automaton.transitions.remove(transition)
        
        # remove eps char from alphabet
        if self._automaton.alphabet.has_key(-1):
            del self._automaton.alphabet[-1]
        
        # Also removes all isolated, unreachable and blind states.
        self.remove_unreachable()
        
        # Set flags
        self._automaton.Flags["Epsilon Free"] = True       
                
    def remove_unreachable(self):
        """ Removes isolated, unreachable and blind states. """
        # Set of states which have outgres transitions. 
        hasOutgres = set()
        # Set of states which have ingres transitions. 
        hasIngres = set()
        # Set of all states. 
        states = set(self._automaton.states.keys())
        
        # Compute ingress and outgress transitions for all states.
        for transition in self._automaton.transitions:
            hasOutgres.add(transition[0])
            hasIngres.add(transition[2])
        
        # Compute set of states having only ingres or only outgres transitions.
        onlyOne = hasOutgres.symmetric_difference(hasIngres)
        
        # Remove start state from this set.
        if self._automaton.start in onlyOne:
            onlyOne.remove(self._automaton.start)
        
        # Remove final states from this set if they have ingres transitions.
        for fstate in self._automaton.final:
            if (fstate in onlyOne) and (fstate in hasIngres):
                onlyOne.remove(fstate)
                
        # Create set of states, which have ingres or outgres transitions.
        some = hasOutgres.union(hasIngres)
        
        # Create set of states, which do not have transition. 
        none = states.difference(some)
        
        # Create set of states, which do not have transition or have only ingres or outgres transitions.
        remove = none.union(onlyOne)
        
        # Set of removeable transitions.
        removeableTrans = set()
        
        # Remove removeable states, remove them from final states if needed.
        for state in remove:
            del self._automaton.states[state]
            if state in self._automaton.final:
                self._automaton.final.remove(state)
        
        # Add removeable transitions to theirs set.
        for trans in self._automaton.transitions:
            if trans[0] in remove:
                removeableTrans.add(trans)
            if trans[2] in remove:
                removeableTrans.add(trans)
        
        # Remove removeable transitions.
        for trans in removeableTrans:
            self._automaton.transitions.remove(trans)
            
    def search(self, input_string):
        """This function will find patterns in the given string by the specified approach. This default version uses nfa_data for it. Approaches should reimplement this function.
        Parameters:
        input_string - input string.
        Returns:
        Bitmap of matched regular expressions."""
        # Stack of states. State is list (tupple) consisting of state and unprocessed part of input string.
        Stack = list()
        # Set of actual states.
        ActState = set()
        
        # Create start state.
        ActState.add((self._automaton.start, input_string))
        # Add start state to stack.0
        Stack.append(ActState)
        
        # Create mapping between reg. exp. number and coresponding final states.
        sameFinal = dict()
        for fstate in self._automaton.final:
            if sameFinal.has_key(self._automaton.states[fstate].get_regexp_number()) == True:
                sameFinal[self._automaton.states[fstate].get_regexp_number()].add(fstate)
            else:
                sameFinal[self._automaton.states[fstate].get_regexp_number()] = set()
                sameFinal[self._automaton.states[fstate].get_regexp_number()].add(fstate)
        
        # Compute number of reg. exp.
        rules = len(sameFinal)
        # dictionary is no longer needed.
        sameFinal = None
        
        # Init bitmap to zeros.
        bitmap = [0] * rules
        
        # If needed create mapping between states and outgoing transitions.
        if self._mapper == None:
            self._mapper = dict()
            for transition in self._automaton.transitions:
                if self._mapper.has_key(transition[0]) == True:
                    self._mapper[transition[0]].add((transition[1], transition[2]))
                else:
                    self._mapper[transition[0]] = set()
                    self._mapper[transition[0]].add((transition[1], transition[2]))
        
        # Until stack is empty, search
        while len(Stack) != 0:
            # Pop state from stack.
            ActState = Stack.pop()
            newActState = set()
            # Create new state. Accept char if possible and add state to new state.
            for state in ActState:
                if self._mapper.has_key(state[0]):
                    for transition in self._mapper[state[0]]:
                        try:
                            res = self._automaton.alphabet[transition[0]].accept(state[1])
                        except:
                            pass
                        else:
                            newActState.add((transition[1],res))
                # If in final state, set coresponding bitmap field to 1.
                if self._automaton.states[state[0]].is_final() == True:
                    bitmap[self._automaton.states[state[0]].get_regexp_number()] = 1
            # If possible add new state to stack.
            if len(newActState) > 0:
                Stack.append(newActState)
        
        # Return bitmap.
        return bitmap
            
    def _removeCharClasses(self):
        """ Remove char classes from automaton. Removed char classes are substituted with equivalent chars and coresponding transitions are added. """
        # Dict for mapping from chars to their numbers.
        charDict = dict()
        
        # Find max index of alphabet chars. Create mapping from chars to their numbers.
        maxIndex = -1
        for symbol in self._automaton.alphabet.keys():
            if isinstance(self._automaton.alphabet[symbol], sym_char.b_Sym_char):
                charDict[self._automaton.alphabet[symbol]] = symbol
            #print(str(type(symbol)) + " - " + str(symbol))
            if symbol > maxIndex:
                maxIndex = symbol
        maxIndex += 1
                
        #print(str(charDict.keys()))
             
        # Set for removeable transitions.
        removeTransSet = set()
        # Set for added transitions.
        addTransSet = set()
        
        # For all transitions remove transitions with char class symbol and add coresponding transitions with char symbols.
        for transition in self._automaton.transitions:
            # if symbol is char class.
            if isinstance(self._automaton.alphabet[transition[1]], sym_char_class.b_Sym_char_class):
                # For all chars in char class.
                for char in self._automaton.alphabet[transition[1]].charClass:
                    # Create new char.
                    sym = sym_char.b_Sym_char(char, char, maxIndex)
                    # If new char is not in alphabet, add new char to alphabet and get index.
                    if charDict.has_key(sym) == False:
                        self._automaton.alphabet[maxIndex] = sym
                        charDict[sym] = maxIndex
                        maxIndex += 1
                        index = maxIndex - 1
                    else:
                        # Otherwise get index.
                        index = charDict[sym]
                    # Add transition with this char.
                    addTransSet.add((transition[0], index, transition[2]))
                    #print("ADD: " + str((transition[0], index, transition[2])))
                # Remove transition with char class. This transition was replaced with coresponding transitions with char symbols.
                removeTransSet.add(transition)
                #print("REMOVE: " + str(transition))
        
        #print(str(self._automaton.alphabet))
        # Remove transitions.
        for transitions in removeTransSet:
            self._automaton.transitions.remove(transitions)
        
        # Add transitions.
        for transitions in addTransSet:
            self._automaton.transitions.add(transitions)
        
        # Add char class symbols to removeable symbols set.
        removeSymbolSet = set()
        for symbol in self._automaton.alphabet.keys():
            if isinstance(self._automaton.alphabet[symbol], sym_char_class.b_Sym_char_class):
                removeSymbolSet.add(symbol)
        
        # Remove char class symbols from alphabet.
        for symbol in removeSymbolSet:
            del self._automaton.alphabet[symbol]

    def _createCharClasses(self):
        """ Creates char classes, if they can be created. Replaced transitions are removed. Unused symbols are removed after creation of char classes."""
        # Mapping between start and end state of transition and coresponding transitions.
        stateTupple = dict()        
        for transition in self._automaton.transitions:
            if stateTupple.has_key((transition[0], transition[2])) == True:
                stateTupple[(transition[0], transition[2])].add(transition)
            else:
                stateTupple[(transition[0], transition[2])] = set()
                stateTupple[(transition[0], transition[2])].add(transition)
                       
        # Find max alphabet index.
        maxIndex = -1
        for symbol in self._automaton.alphabet.keys():
            if symbol > maxIndex:
                maxIndex = symbol
        maxIndex += 1
               
        symbolDict = dict()
        
        # For all state tupple pairs perform creation of char classes.
        for key in stateTupple.keys():
            # If transition between states exist, continue.
            if len(stateTupple[key]) > 0:
                # Set of symbols.
                symbolSet = set()
                # Find all symbols on transitions between the states.
                for transition in stateTupple[key]:
                    # If symbol is char class, add all chars.
                    if isinstance(self._automaton.alphabet[transition[1]], sym_char_class.b_Sym_char_class):
                        symbolSet |= self._automaton.alphabet[transition[1]].charClass
                    # If symbol is char, add char.
                    if isinstance(self._automaton.alphabet[transition[1]], sym_char.b_Sym_char):
                        symbolSet.add(self._automaton.alphabet[transition[1]].char)
                    # Remove transition.
                    self._automaton.transitions.remove(transition)
                # If more than one char symbol exist, add char class.
                if len(symbolSet) > 1:
                    # Create char class name.
                    strSymSetMod = str()
                    for sym in symbolSet:
                        strSymSetMod += sym
                    strSymSetMod = "[" + strSymSetMod + "]"
                    # Create char class.
                    tmp = sym_char_class.b_Sym_char_class(strSymSetMod, symbolSet, maxIndex)
                    # If char class is not in alphabet add it.
                    if symbolDict.has_key(tmp) == False:                         
                        self._automaton.alphabet[maxIndex] = tmp
                        symbolDict[tmp] = maxIndex
                        index = maxIndex
                    else:
                        # Otherwise just look up char class index in alphabet.
                        index = symbolDict[tmp]                     
                else:
                    # Otherwise add char.
                    char = symbolSet.pop()
                    # Create char.
                    tmp = sym_char.b_Sym_char(char, char, maxIndex)
                    # If char is not in alphabet add it.
                    if symbolDict.has_key(tmp) == False:                         
                        self._automaton.alphabet[maxIndex] = tmp
                        symbolDict[tmp] = maxIndex
                        index = maxIndex
                    else:
                        # Otherwise just look up char index in alphabet.
                        index = symbolDict[tmp]
                # Add transition to automaton.
                self._automaton.transitions.add(((key[0], index, key[1])))
                maxIndex += 1
                
        # Create symbol occurancy count table.
        removeSymbolsDict = dict()
        for symbol in self._automaton.alphabet.keys():
            removeSymbolsDict[symbol] = 0
            
        # Update symbol occurancy count table, if symbol exists in transitions.
        for transition in self._automaton.transitions:
            removeSymbolsDict[transition[1]] += 1
        
        # Remove unused symbols from alphabet.
        for symbol in removeSymbolsDict.keys():
            if removeSymbolsDict[symbol] == 0:
                del self._automaton.alphabet[symbol]

    def  reduce_alphabet(self):
        """Reduce alphabet by character classes. This create another type of char class compatibil with DFA. Alphabet shouldn't contain PCRE/NFA char classes. """
        # Number of rows in transition table.
        rows = max(self._automaton.states.keys()) + 1
        # Number of columns in transition table.
        columns = max(self._automaton.alphabet.keys()) + 1
         
        # Create transition table.
        matrix = list()
        for i in range(0, columns):
            column = [-1]*rows
            matrix.append(column)
        
        # Mark used columns.
        used = [False] * columns
        
        # Fill transition table.
        for trans in self._automaton.transitions:
            matrix[trans[1]][trans[0]] = trans[2]
         
        # Mark unfilled columns as used.
        for i in range(0, columns):
            if not i in self._automaton.alphabet.keys():
                used[i] = True
            else:
                if max(matrix[i]) == -1:
                    used[i] = True
        
        # Find which columns are same and can be grouped in DFA char class.
        classes = dict()
        for i in range(0, columns):
            if used[i] == False:
                same = set()
                same.add(i)
                for j in range(i+1, columns):
                    if used[j] == False:
                        if matrix[i] == matrix[j]:
                            same.add(j)
                            used[j] = True
                used[i] = True
                if len(same) > 1:
                    classes[self._automaton.alphabet[i].char] = same
        
        # Store transformation between alphbet index and new alphabet index.
        # Used for removal of transitions which have symbol which have been
        # included in char class and for creating transitions which have new
        # char class symbol
        transformDict = dict()
        
        # Count of alphabet classes.
        cnt = 0
        # Starting index of new alphabet symbols.
        start = max(self._automaton.alphabet.keys()) + 1
        # Create DFA alphabet classes.
        for symbol in classes.keys(): 
            # Create DFA char class name.
            strSymSetMod = str()
            charClass = set()
            for index in classes[symbol]:
                charClass.add(self._automaton.alphabet[index].char)
                transformDict[index] = start
                strSymSetMod += self._automaton.alphabet[index].char
            strSymSetMod = "[" + strSymSetMod + "]"
            # Create DFA char class object.
            symObject = sym_char_class_dfa.b_Sym_char_class_dfa(strSymSetMod, symbol, charClass, start)
            # Update count.
            cnt += 1
            # Update starting index.
            start += 1
            # Add symbol to the alphabet.
            self._automaton.alphabet[symObject.get_id()] = symObject
         
        # List of removeable transitions.
        transitionRemoveList = list()
        # Set of added transitions.
        transitionAddSet = set()
        # Find trasitions for removal and find transitions for add.
        for trans in self._automaton.transitions:
            if transformDict.has_key(trans[1]):
                transitionRemoveList.append(trans)
                transitionAddSet.add(((trans[0], transformDict[trans[1]] , trans[2])))
        
        # Add transitions.
        for trans in transitionAddSet:
            self._automaton.transitions.add(trans)
        
        # Set count of DFA char classes into statistic.
        self._statistic["DFA Char Classes"] = cnt
        
        # Remove transitions.
        for i in range(0, len(transitionRemoveList)):
            self._automaton.transitions.remove(transitionRemoveList[i])
        
        # Create symbol occurancy count table.
        removeSymbolsDict = dict()
        for symbol in self._automaton.alphabet.keys():
            removeSymbolsDict[symbol] = 0
        
        # Update symbol occurancy count table, if symbol exists in transitions.    
        for transition in self._automaton.transitions:
            removeSymbolsDict[transition[1]] += 1
        
        # Remove unused symbols from alphabet.
        for symbol in removeSymbolsDict.keys():
            if removeSymbolsDict[symbol] == 0:
                del self._automaton.alphabet[symbol]
                
    def stride_2(self, all_chars = None):
        """ Transform automaton to 2-stride automaton. This new automaton accept 2 chars per transition. If automaton is already strided, then 2*stride automaton will be created. In this case the automaton accept 2*stride chars per cycle. Removes Deterministic Flag. Input automaton must be eps free. Sets Strided Flag to True and set Stride Flag to current stride.
        
        Parameters:
        all_chars - set of all chars in alphabet. If Not set, it defaults to ASCII 0 - 255.
        """
        
        # TODO: check if 2-strided automaton is deterministic if source automaton was deterministic
        self._automaton.Flags["Deterministic"] = False
        
        mapper = dict()
        
        # create mapper implemented as dict that maps state x symbol -> state
        for transition in self._automaton.transitions:
            if mapper.has_key(transition[0]) == True:
                if mapper[transition[0]].has_key(transition[1]) == True:
                    mapper[transition[0]][transition[1]].add(transition[2])
                else:
                    mapper[transition[0]][transition[1]] = set()
                    mapper[transition[0]][transition[1]].add(transition[2])
            else:
                mapper[transition[0]] = dict()
                mapper[transition[0]][transition[1]] = set()
                mapper[transition[0]][transition[1]].add(transition[2])
        
        # new transitions set
        new_transitions = set()
        # new alphabet dict (id->symbol)
        new_alphabet = dict()
        # new reverse alphabet dict (symbol->id)
        reverse_alphabet = dict()
        # reverse state dict (state->id)
        reverse_state = dict()
        # new states dict (id->states)
        new_states = dict()
        
        # next free state index
        state_index = max(self._automaton.states.keys()) + 1
        
        # next free alphabet index
        alphabet_index = 0
        
        # default automaton is 1 - strided
        add = 1
        
        # if automaton is strided, get the existing stride
        if self._automaton.Flags.has_key("Stride"):
            add = self._automaton.Flags["Stride"]
        
        # if complete alphabet is not defined, use default one - ASCII chars 0 - 255
        if all_chars == None:
            all_chars = set()
            for i in range(0, 256):
                all_chars.add(chr(i))
        
        # for all states compute strided transitions
        for state in self._automaton.states:
            # check if state has outgoing transitions
            if mapper.has_key(state):
                # for all symbols from state 
                for symbol in mapper[state].keys():
                    # for all target states for given source state and symbol = intermediate states
                    for inner_state in mapper[state][symbol]:
                        # if intermediate state is final handle it
                        if inner_state in self._automaton.final:
                            # add new end state and add all chars char class to second position
                            # if state has outgoing transition continue follow them, this is done
                            #to ensure deterministic behavior of strided DFA
                            
                            # create local copy of complete alphabet. This copy is created for all uncovered strides
                            local_chars = list()
                            for i in range(0, add):
                                local_chars.append(copy.deepcopy(all_chars))
                            
                            # if final state has outgoing transitions follow them
                            if mapper.has_key(inner_state):
                                # follow all symbol for intermediate state
                                for inner_symbol in mapper[inner_state].keys():
                                    # for all target states for given source state and symbol = target states
                                    for target_state in mapper[inner_state][inner_symbol]:
                                                                                
                                        target_symbol = 0
                            
                                        first = 0
                                        second = 0
                                        complete = 0
                                        
                                        # Uses symbol from first transition (State -> Intermediate State)
                                        # create first part of 2-strided symbol from char object
                                        if isinstance(self._automaton.alphabet[symbol], sym_char.b_Sym_char):
                                            first_s = set()
                                            first_s.add(self._automaton.alphabet[symbol].char)
                                            first_s = frozenset(first_s)
                                            first = list()
                                            first.append(first_s)
                                        # create first part of 2-strided symbol from charClass object
                                        if isinstance(self._automaton.alphabet[symbol], sym_char_class.b_Sym_char_class):
                                            first_s = frozenset(self._automaton.alphabet[symbol].charClass)
                                            first = list()
                                            first.append(first_s)
                                        # create first part of 2-strided symbol from kchar object (already strided char)
                                        if isinstance(self._automaton.alphabet[symbol], sym_kchar.b_Sym_kchar):
                                            first = list(self._automaton.alphabet[symbol].kchar)
                                        
                                        # Uses symbol from second transition (Intermediate State -> Target State)
                                        # create second part of 2-strided symbol from char object    
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_char.b_Sym_char):
                                            second_s = set()
                                            second_s.add(self._automaton.alphabet[inner_symbol].char)
                                            second_s = frozenset(second_s)
                                            second = list()
                                            second.append(second_s)
                                            
                                            # remove used symbol from local copy of complete alphabet.
                                            local_chars[0] = local_chars[0] - second_s
                                        # create second part of 2-strided symbol from charClass object
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_char_class.b_Sym_char_class):
                                            second_s = frozenset(self._automaton.alphabet[inner_symbol].charClass)
                                            second = list()
                                            second.append(second_s)
                                            
                                             # remove used symbol from local copy of complete alphabet.
                                            local_chars[0] = local_chars[0] - self._automaton.alphabet[inner_symbol].charClass
                                        # create second part of 2-strided symbol from kchar object (already strided char)
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_kchar.b_Sym_kchar):
                                            second = list(self._automaton.alphabet[inner_symbol].kchar)
                                            
                                             # remove used symbol from local copy of complete alphabet.
                                            for i in range(len(self._automaton.alphabet[inner_symbol].kchar)):
                                                local_chars[i] = local_chars[i] - self._automaton.alphabet[inner_symbol].kchar[i]
                                        
                                        # crate complete strided symbol
                                        complete = list()
                                        complete += first
                                        complete += second
                                        complete = tuple(complete)
                                        
                                        # create complete strided symbol description (will shown in graphicasl representation)
                                        complete_str = "{"
                                        
                                        for i in range(0, len(complete)):
                                            if len(complete[i]) == 1:
                                                for char in complete[i]:
                                                    complete_str += str(char)
                                                    #print str(complete)
                                            else:
                                                complete_str += "["
                                                for char in complete[i]:
                                                    complete_str += char + ","
                                                complete_str =  complete_str[0:len(complete_str)-1] + "]"
                                        
                                        complete_str += "}"
                                        
                                        # Create strided symbol object
                                        new_symbol = sym_kchar.b_Sym_kchar(complete_str, complete, alphabet_index)
                                        # Check if symbol is already not added
                                        if new_symbol not in reverse_alphabet.keys():
                                            # add symbol to alphabet
                                            new_alphabet[alphabet_index] = new_symbol
                                            reverse_alphabet[symbol] = alphabet_index
                                            target_symbol = alphabet_index
                                            alphabet_index += 1
                                        else:
                                            # get symbol id
                                            target_symbol = reverse_alphabet[new_symbol]
                                        
                                        # add transition
                                        new_transitions.add((state, target_symbol, target_state))
                            
                            target_symbol = 0
                            target_state = 0
                            
                            first = 0
                            second = 0
                            complete = 0
                            
                            is_zero_size = False
                            
                            # check if some part of local alphabet is empty -> all posible transitions from final state are coverd
                            for i in range (0, add):
                                if len(local_chars[i]) == 0:
                                    is_zero_size = True
                            
                            # if there are somu unused symbols, then create transition to new final state, because otherwise the strided
                            # automaton wouldn't be equivalent to its non strided version.
                            if is_zero_size == False:
                                # Uses symbol from first transition (State -> Intermediate State)
                                # create first part of 2-strided symbol from char object, and create second part from unused symbols left in local_chars
                                # crate complete strided symbol
                                if isinstance(self._automaton.alphabet[symbol], sym_char.b_Sym_char):
                                    first = set()
                                    first.add(self._automaton.alphabet[symbol].char)
                                    first = frozenset(first)
                                    second = frozenset(local_chars[0])
                                    complete = (first, second)
                                # create first part of 2-strided symbol from charClass object, and create second part from unused symbols left in local_chars
                                # crate complete strided symbol
                                if isinstance(self._automaton.alphabet[symbol], sym_char_class.b_Sym_char_class):
                                    first = frozenset(self._automaton.alphabet[symbol].charClass)
                                    second = frozenset(local_chars[0])
                                    complete = (first, second)
                                # create first part of 2-strided symbol from kchar object (already strided char), and create second part from unused symbols left in local_chars
                                # crate complete strided symbol
                                if isinstance(self._automaton.alphabet[symbol], sym_kchar.b_Sym_kchar):
                                    first = list(self._automaton.alphabet[symbol].kchar)
                                    second = list()
                                    for i in range (0, add):
                                        second.append(frozenset(local_chars[i]))
                                    
                                    complete = list()
                                    complete += first
                                    complete += second
                                    complete = tuple(complete)
                                
                                # create complete strided symbol description (will shown in graphicasl representation)
                                complete_str = "{"
                                            
                                for i in range(0, len(complete)):
                                    if len(complete[i]) == 1:
                                        for char in complete[i]:
                                            complete_str += str(char)
                                    else:
                                        complete_str += "["
                                        for char in complete[i]:
                                            complete_str += char + ","
                                        complete_str =  complete_str[0:len(complete_str)-1] + "]"
                                
                                complete_str += "}"
                                
                                # Create strided symbol object  
                                new_symbol = sym_kchar.b_Sym_kchar(complete_str, complete, alphabet_index)
                                # Check if symbol is already not added
                                if new_symbol not in reverse_alphabet.keys():
                                    # add symbol to alphabet
                                    new_alphabet[alphabet_index] = new_symbol
                                    reverse_alphabet[symbol] = alphabet_index
                                    target_symbol = alphabet_index
                                    alphabet_index += 1
                                else:
                                    # get symbol id
                                    target_symbol = reverse_alphabet[new_symbol]
                                
                                # add new final state
                                # if new final state for final state reg exp number does not exist add the new final state, otherwise get its id
                                if self._automaton.states[inner_state].get_regexp_number() not in reverse_state.keys():
                                    new_state = b_state.b_State(state_index, self._automaton.states[inner_state].get_regexp_number())
                                    #self._automaton.states[state_index] = new_state
                                    new_states[state_index] = new_state
                                    reverse_state[self._automaton.states[inner_state].get_regexp_number()] = state_index
                                    target_state = state_index
                                    self._automaton.final.add(state_index)
                                    state_index += 1
                                else:
                                    target_state = reverse_state[self._automaton.states[inner_state].get_regexp_number()]
                                
                                # add new transition
                                new_transitions.add((state, target_symbol, target_state))
                        else:
                            # if final state has outgoing transitions follow them
                            if mapper.has_key(inner_state):
                                # follow all symbol for intermediate state
                                for inner_symbol in mapper[inner_state].keys():
                                    # for all target states for given source state and symbol = target states
                                    for target_state in mapper[inner_state][inner_symbol]:
                                        
                                        target_symbol = 0
                            
                                        first = 0
                                        second = 0
                                        complete = 0
                                        
                                        # Uses symbol from first transition (State -> Intermediate State)
                                        # create first part of 2-strided symbol from char object
                                        if isinstance(self._automaton.alphabet[symbol], sym_char.b_Sym_char):
                                            first_s = set()
                                            first_s.add(self._automaton.alphabet[symbol].char)
                                            first_s = frozenset(first_s)
                                            first = list()
                                            first.append(first_s)
                                        # create first part of 2-strided symbol from charClass object
                                        if isinstance(self._automaton.alphabet[symbol], sym_char_class.b_Sym_char_class):
                                            first_s = frozenset(self._automaton.alphabet[symbol].charClass)
                                            first = list()
                                            first.append(first_s)
                                        # create first part of 2-strided symbol from kchar object (already strided char)
                                        if isinstance(self._automaton.alphabet[symbol], sym_kchar.b_Sym_kchar):
                                            first = list(self._automaton.alphabet[symbol].kchar)
                                        
                                        # Uses symbol from second transition (Intermediate State -> Target State)
                                        # create second part of 2-strided symbol from char object   
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_char.b_Sym_char):
                                            second_s = set()
                                            second_s.add(self._automaton.alphabet[inner_symbol].char)
                                            second_s = frozenset(second_s)
                                            second = list()
                                            second.append(second_s)
                                        # create second part of 2-strided symbol from charClass object
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_char_class.b_Sym_char_class):
                                            second_s = frozenset(self._automaton.alphabet[inner_symbol].charClass)
                                            second = list()
                                            second.append(second_s)
                                        # create second part of 2-strided symbol from kchar object (already strided char)
                                        if isinstance(self._automaton.alphabet[inner_symbol], sym_kchar.b_Sym_kchar):
                                            second = list(self._automaton.alphabet[inner_symbol].kchar)
                                        
                                        # crate complete strided symbol    
                                        complete = list()
                                        complete += first
                                        complete += second
                                        complete = tuple(complete)
                                            
                                        complete_str = "{"
                                        
                                        # create complete strided symbol description (will shown in graphicasl representation)
                                        for i in range(0, len(complete)):
                                            if len(complete[i]) == 1:
                                                for char in complete[i]:
                                                    complete_str += str(char)
                                                    #print str(complete)
                                            else:
                                                complete_str += "["
                                                for char in complete[i]:
                                                    complete_str += char + ","
                                                complete_str =  complete_str[0:len(complete_str)-1] + "]"
                                                #print str(complete)
                                        complete_str += "}"
                                        
                                        # Create strided symbol object    
                                        new_symbol = sym_kchar.b_Sym_kchar(complete_str, complete, alphabet_index)
                                        # Check if symbol is already not added
                                        if new_symbol not in reverse_alphabet.keys():
                                            # add symbol to alphabet
                                            new_alphabet[alphabet_index] = new_symbol
                                            reverse_alphabet[symbol] = alphabet_index
                                            target_symbol = alphabet_index
                                            alphabet_index += 1
                                        else:
                                            # get symbol id
                                            target_symbol = reverse_alphabet[new_symbol]
                                        
                                        # add transition
                                        new_transitions.add((state, target_symbol, target_state))
        
        # set transitions in automaton to new transition
        self._automaton.transitions = new_transitions
        # set alphabet in automaton to new alphabet
        self._automaton.alphabet = new_alphabet
        
        # add new states
        for state in new_states.keys():
            self._automaton.states[state] = new_states[state]
        
        # set automaton flafs
        self._automaton.Flags["Stride"] = 2*add
        self._automaton.Flags["Strided"] = True
        
        # remove ureachable states
        self.remove_unreachable()
        
    def reduce_by_simulation(self):
        simulation = dict()
           
        for state in self._automaton.states.keys():
            simulation[state] = list()
        
        print "Initialization of simulation done."
        
        # maps relation between state and its outgoing transitions
        mapper = dict()
        for state in self._automaton.states.keys():
            mapper[state] = set()
        for transition in self._automaton.transitions:
            if mapper.has_key(transition[0]) == True:
                mapper[transition[0]].add(transition)
            else:
                mapper[transition[0]] = set()
                mapper[transition[0]].add(transition)
        
        # maps relation between state and its ingoing transitions
        reverse_mapper = dict()
        for state in self._automaton.states.keys():
            reverse_mapper[state] = set()
        for transition in self._automaton.transitions:
            if reverse_mapper.has_key(transition[2]) == True:
                reverse_mapper[transition[2]].add(transition)
            else:
                reverse_mapper[transition[2]] = set()
                reverse_mapper[transition[2]].add(transition)
        
        
        print "Mappers created."
        
        # compute initial simulation
        for src_state in self._automaton.states.keys():
            for dst_state in self._automaton.states.keys():
                src_languages = self._automaton.states[src_state].get_regexp_number()
                dst_languages = self._automaton.states[dst_state].get_regexp_number()
                if not isinstance(src_languages, set):
                    src_tmp = set()
                    if src_languages > -1:
                        src_tmp.add(src_languages)
                    src_languages = src_tmp
                    
                if not isinstance(dst_languages, set):
                    dst_tmp = set()
                    if dst_languages > -1:
                        dst_tmp.add(dst_languages)
                    dst_languages = dst_tmp
                
                if src_languages == set() or ((src_languages <= dst_languages) and (dst_languages <= src_languages)):
                    simulation[src_state].append((src_state, dst_state))
        
        print "Initial simulation computed."
        
        # refine simulation
        refine = True
        new_simulation = dict()
        for state in self._automaton.states.keys():
            new_simulation[state] = list()

        while refine == True:
            refine = False
            for src_state in self._automaton.states.keys():
                pair = False
                for simulation_pair in simulation[src_state]:
                    lhs = True
                    for lhs_transition in mapper[simulation_pair[0]]:
                        rhs = False
                        for rhs_transition in mapper[simulation_pair[1]]:
                            rhs = rhs or (lhs_transition[1] == rhs_transition[1] and ((lhs_transition[2], rhs_transition[2]) in simulation[lhs_transition[2]]))
                        lhs = lhs and rhs
                    pair = pair or lhs
                    if lhs == True:
                        new_simulation[simulation_pair[0]].append(simulation_pair)
                refine = refine or pair
            if simulation == new_simulation:
                refine = False
            simulation = new_simulation
            new_simulation = dict()
            for state in self._automaton.states.keys():
                new_simulation[state] = list()
            
            print "Iteration done."
            print simulation
            
        #restricting binary relation (simulation) to symetric antireflexive relation
        for state in simulation.keys():
            for simulation_pair in simulation[state]:
                if (simulation_pair[1], simulation_pair[0]) in simulation[simulation_pair[1]]:
                    if (simulation_pair[1] != simulation_pair[0]):
                        new_simulation[state].append(simulation_pair)
        
        print "Relation restriction done."
        
        print new_simulation
        
        equal_states = set()
        
        # create set of equal states from dict of simulation pairs
        for state in new_simulation.keys():
            equal_state = set()
            for simulation_pair in new_simulation[state]:
                equal_state.add(simulation_pair[0])
                equal_state.add(simulation_pair[1])
            if equal_state != set():
                equal_states.add(frozenset(equal_state))
        
        # get new state index
        state_index = max(self._automaton.states.keys()) + 1
        
        to_remove = set()
        
        for equal_state in equal_states:
            to_remove |= equal_state
                  
        # compute new and removed transitions and states using simulation relation
        for equal_state in equal_states:
            new_states = list()
            new_transitions = set()
            removeable_states = list()
            removeable_transitions = set()
            # move states to remove list, add new transitions to new list
            req_exp_num = -1
            for state in equal_state:
                removeable_states.append(state)
                removeable_transitions |= mapper[state]
                removeable_transitions |= reverse_mapper[state]
                for transition in  mapper[state]:
                    if transition[2] not in equal_state:
                        new_transitions.add((state_index, transition[1], transition[2]))
                    else:
                        new_transitions.add((state_index, transition[1], state_index))
                for transition in reverse_mapper[state]:
                    if transition[0] not in equal_state:
                        new_transitions.add((transition[0], transition[1], state_index))
                    else:
                        new_transitions.add((state_index, transition[1], state_index))
                rnum = self._automaton.states[state].get_regexp_number()
                if  rnum != -1:
                    if req_exp_num == -1:
                        req_exp_num = rnum
                    else:
                        if rnum != req_exp_num:
                            print "ERROR: Imposible happend, simulation reduction of multilaguage NFA conected end states from different languges."
                            return
            
            new_state = b_state.b_State(state_index, req_exp_num)
            state_tupple = (state_index, new_state)
            new_states.append(state_tupple)
            state_index += 1
       
            # change automaton    
            # add new states
            for state in new_states:
                self._automaton.states[state[0]] = state[1]
                mapper[state] = set()
                reverse_mapper[state] = set()
                if state[1].is_final():
                    self._automaton.final.add(state[0])
            
            # remove old states
            for state in removeable_states:
                del self._automaton.states[state]
                del mapper[state]
                mapper[state] = set()
                del reverse_mapper[state]
                reverse_mapper[state] = set()
                if state in self._automaton.final:
                    self._automaton.final.remove(state)
                    
            # add new transitions
            self._automaton.transitions |= new_transitions
            
            # update mapper and reverse mapper with new transitions
            for transition in new_transitions:
                if mapper.has_key(transition[0]) == True:
                    mapper[transition[0]].add(transition)
                else:
                    mapper[transition[0]] = set()
                    mapper[transition[0]].add(transition)
            
            for transition in new_transitions:
                if reverse_mapper.has_key(transition[2]) == True:
                    reverse_mapper[transition[2]].add(transition)
                else:
                    reverse_mapper[transition[2]] = set()
                    reverse_mapper[transition[2]].add(transition)
            
            # remove old transitions
            self._automaton.transitions -= removeable_transitions
            
            # update mapper and reverse mapper with removed transitions
            for transition in removeable_transitions:
                if transition[0] in mapper.keys():
                    if transition in mapper[transition[0]]:
                        mapper[transition[0]].remove(transition)
                
            for transition in removeable_transitions:
                if transition[2] in reverse_mapper.keys():
                    if transition in reverse_mapper[transition[2]]:
                        reverse_mapper[transition[2]].remove(transition)
                
        # final clean
        removeable_transitions = set()
        for transition in self._automaton.transitions:
            if transition[0] in to_remove or transition[2] in to_remove:
                removeable_transitions.add(transition)
                
        self._automaton.transitions -= removeable_transitions
        
        #print "REMOVE: " + str(removeable_transitions)
        #print "NEW:    " + str(new_transitions)
        
    def reduce_by_flip_simulations(self):
        old_states_num = len(self._automaton.states)
        old_transitions_num = len(self._automaton.transitions)
        
        self.reduce_by_simulation()
        
        new_transitions = set()
        for transition in self._automaton.transitions:
            new_transitions.add((transition[2], transition[1], transition[0]))
        self._automaton.transitions = new_transitions
        
        self.reduce_by_simulation()
        
        new_transitions = set()
        for transition in self._automaton.transitions:
            new_transitions.add((transition[2], transition[1], transition[0]))
        self._automaton.transitions = new_transitions
        
        new_states_num = len(self._automaton.states)
        new_transitions_num = len(self._automaton.transitions)
        
        while (new_states_num != old_states_num or new_transitions_num != old_transitions_num):
            old_states_num = new_states_num
            old_transitions_num = new_transitions_num
            
            self.reduce_by_simulation()
            
            new_transitions = set()
            for transition in self._automaton.transitions:
                new_transitions.add((transition[2], transition[1], transition[0]))
            self._automaton.transitions = new_transitions
            
            self.reduce_by_simulation()
            
            new_transitions = set()
            for transition in self._automaton.transitions:
                new_transitions.add((transition[2], transition[1], transition[0]))
            self._automaton.transitions = new_transitions
            
            new_states_num = len(self._automaton.states)
            new_transitions_num = len(self._automaton.transitions)
        
        
###############################################################################
# End of File b_automaton.py                                                  #
###############################################################################