/***************************************************************************
############################################################################
#  hybrid_fa.c: Algorithm for Hybrid FA (transit through automat)
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol
############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or
#  implied warranties, including, but not limited to, the implied warranties
#  of merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$
***************************************************************************/

/*
 *  hybrid_fa.c   ver. 1.0
 *
 *  Algorithm for Hybrid FA (transit through automat)
 *  =================================================
 *
 *  Jaroslav Suchodol,  August 2010
 *
 */

#include <stdio.h>
#include <stdlib.h>

# define false 0
# define true 1

typedef unsigned int u_i;

/* definition of global variables for DFA part */
u_i count_states, start_state, alp_count_trans, alphabet_length;
u_i count_final_states;

/* array of transitions chars (alphabet),
 * index to this array is in 'alp_in_begin/end' */
u_i *alphabet;

/* array of indexes where begin/end char(s) for transition,
 * index to this array is structure T_STATE.trans_char */
u_i *alp_in_begin;
u_i *alp_in_end;

/* array of final states */
u_i *final_states;

// structure for representing one state in DFA part
typedef struct t_state {
  u_i trans_count;  /* count of transitions for current state */
  u_i *trans_char;  /* index transition char(s) */
  u_i *trans_dest;  /* destination state */
} T_STATE;
T_STATE *state;

// NFA transition
typedef struct t_n_t {
  u_i *c;   /* transition char(s) */
  u_i c_c;  // count char for transition
  u_i d;    // destination state of transition
} T_N_T;

// NFA state
typedef struct t_n_s {
  T_N_T *t; /* transitions for state */
  u_i t_c;  // t_c=trans_count
  u_i r;    // real number state
  int a;    // mark of the active state
  int a_a;  // auxiliary for activate state
  int f;    // mark of the final state
} T_N_S;

// one NFA part (automat)
typedef struct t_n {
  T_N_S *s;   /* states of NFA */
  u_i s_c;    // states count
  u_i s_s;    // start state
  u_i act;    // activation DFA state for given NFA
  int c_c;    // indication counting constraint
} T_N;    // T_NFA
T_N *n;   // NFA parts
u_i n_c;  // count NFA parts

/* Free dynamically allocated memory. */
void free_memory()
{
  u_i i, j, k;
  for (i = 0; i < count_states; i++) {
    free(state[i].trans_char); free(state[i].trans_dest);
  }
  for (i = 0; i < n_c; i++) {
    for (j = 0; j < n[i].s_c; j++) {
      for (k = 0; k < n[i].s[j].t_c; k++) {
        free(n[i].s[j].t[k].c);
      }
      free(n[i].s[j].t);
    }
    free(n[i].s);
  }
  free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
  free(final_states); free(n);
}

/* Parse input file (first argument of program) and
 * make structure from it for later use. */
void parse_file(char *FileName) {
  FILE *fr;
  u_i i, j, k, l;
  u_i f_i, f_j, f_k;

  // open parse file
  if ((fr = fopen(FileName, "r")) == NULL) {
    fprintf(stderr, "Error: file \"%s\" could not be open for reading!\n",
      FileName);
    exit(1);
  }
/* First parse DFA part. */
  /* parse COUNT OF STATES */
  fscanf(fr, "%u\n", &count_states);
  state = (T_STATE *) malloc(count_states * sizeof(T_STATE));
  /* parse ALPHABET */
  // first parse indexes
  fscanf(fr, "%u\n", &alp_count_trans);
  alp_in_begin = (u_i *) malloc(alp_count_trans * sizeof(u_i));
  alp_in_end = (u_i *) malloc(alp_count_trans * sizeof(u_i));
  if (alp_in_begin == NULL || alp_in_end == NULL || state == NULL) {
    fprintf(stderr, "Error: not enough memory!\n");
    free(state); free(alp_in_begin); free(alp_in_end);
    exit(1);
  }
  for (i = 0; i < alp_count_trans; i++) {
    fscanf(fr, "%u->%u|", &alp_in_begin[i], &alp_in_end[i]);
  }
  // second parse chars
  fscanf(fr, "%u->", &alphabet_length);
  alphabet = (u_i *) malloc(alphabet_length * sizeof(u_i));
  if (alphabet == NULL) {
    fprintf(stderr, "Error: not enough memory!\n");
    free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
    exit(1);
  }
  for (i = 0; i < alphabet_length; i++) {
    fscanf(fr, "%u|", &alphabet[i]);
  }
  /* parse STARTING STATE */
  fscanf(fr, "%u", &start_state);
  /* parse TRANSITIONS */
  // first parse count transitions for each state
  for (i = 0; i < count_states; i++) {
    fscanf(fr, "%u|", &state[i].trans_count);
  }
  // second parse transition char with destination state
  for (i = 0; i < count_states; i++) {
    state[i].trans_char = (u_i *) malloc(state[i].trans_count * sizeof(u_i));
    state[i].trans_dest = (u_i *) malloc(state[i].trans_count * sizeof(u_i));
    if (state[i].trans_char == NULL || state[i].trans_dest == NULL) {
      fprintf(stderr, "Error: not enough memory!\n");
      for (f_i = 0; f_i <= i; f_i++) {
        free(state[f_i].trans_char); free(state[f_i].trans_dest);
      }
      free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
      exit(1);
    }
    for (j = 0; j < state[i].trans_count; j++) {
      fscanf(fr, "%u->%u|", &state[i].trans_char[j], &state[i].trans_dest[j]);
    }
  }
  /* parse FINAL STATES */
  fscanf(fr, "%u", &count_final_states);
  final_states = (u_i *) malloc(count_final_states * sizeof(u_i));
  if (final_states == NULL) {
    fprintf(stderr, "Error: not enough memory!\n");
    for (i = 0; i < count_states; i++) {
      free(state[i].trans_char); free(state[i].trans_dest);
    }
    free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
    free(final_states);
    exit(1);
  }
  for (i = 0; i < count_final_states; i++) {
    fscanf(fr, "%u|", &final_states[i]);
  }
/* Second parse NFA parts. */
  /* parse COUNT OF NFA PARTS */
  fscanf(fr, "\n");
  fscanf(fr, "count NFA: %u\n", &n_c);
  n = (T_N *) malloc(n_c * sizeof(T_N));
  if (n == NULL) {
    fprintf(stderr, "Error: not enough memory!\n");
    for (i = 0; i < count_states; i++) {
      free(state[i].trans_char); free(state[i].trans_dest);
    }
    free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
    free(final_states); free(n);
    exit(1);
  }
  // fill data structures for each NFA
  for (k = 0; k < n_c; k++) {
    /* parse ACTIVATION STATE and indication counting constraint*/
    fscanf(fr, "%u->nfa->c_c: %d\n", &n[k].act, &n[k].c_c);
    /* parse COUNT OF STATES */
    fscanf(fr, "%u\n", &n[k].s_c);
    n[k].s = (T_N_S *) malloc(n[k].s_c * sizeof(T_N_S));
    if (n[k].s == NULL) {
      fprintf(stderr, "Error: not enough memory!\n");
      for (i = 0; i < count_states; i++) {
        free(state[i].trans_char); free(state[i].trans_dest);
      }
      for (f_k = 0; f_k <= k; f_k++) {
        free(n[f_k].s);
      }
      free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
      free(final_states); free(n);
      exit(1);
    }
    /* parse REAL STATE NUMBERS and assign default values to 'f' and 'a'*/
    for (i = 0; i < n[k].s_c; i++) {
      fscanf(fr, "%u|", &n[k].s[i].r);
      n[k].s[i].a = false; n[k].s[i].a_a = false; n[k].s[i].f = false;
    }
    /* parse STARTING STATE */
    fscanf(fr, "%u\n", &n[k].s_s);
    /* parse TRANSITIONS between states */
    for (j = 0; j < n[k].s_c; j++) {
      // count transitions for given state
      fscanf(fr, "t_c: %u\n", &n[k].s[j].t_c);
      n[k].s[j].t = (T_N_T *) malloc(n[k].s[j].t_c * sizeof(T_N_T));
      if (n[k].s[j].t == NULL) {
        fprintf(stderr, "Error: not enough memory!\n");
        for (i = 0; i < count_states; i++) {
          free(state[i].trans_char); free(state[i].trans_dest);
        }
        for (f_k = 0; f_k <= k; f_k++) {
          for (f_j = 0; f_j <= j; f_j++) {
            free(n[f_k].s[f_j].t);
          }
          free(n[f_k].s);
        }
        free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
        free(final_states); free(n);
        exit(1);
      }
      // fill chars, count chars, destination state
      for (i = 0; i < n[k].s[j].t_c; i++) {
        fscanf(fr, "c_c: %u, d: %u\n", &n[k].s[j].t[i].c_c, &n[k].s[j].t[i].d);
        n[k].s[j].t[i].c = (u_i *) malloc(n[k].s[j].t[i].c_c * sizeof(u_i));
        if (n[k].s[j].t[i].c == NULL) {
          fprintf(stderr, "Error: not enough memory!\n");
          for (l = 0; l < count_states; l++) {
            free(state[l].trans_char); free(state[l].trans_dest);
          }
          for (f_k = 0; f_k <= k; f_k++) {
            for (f_j = 0; f_j <= j; f_j++) {
              for (f_i = 0; f_i <= i; f_i++) {
                free(n[f_k].s[f_j].t[f_i].c);
              }
              free(n[f_k].s[f_j].t);
            }
            free(n[f_k].s);
          }
          free(state); free(alp_in_begin); free(alp_in_end); free(alphabet);
          free(final_states); free(n);
          exit(1);
        }
        for (l = 0; l < n[k].s[j].t[i].c_c; l++) {
          fscanf(fr, "%u|", &n[k].s[j].t[i].c[l]);
        }
        fscanf(fr, "\n");
      }
    }
    /* parse FINAL STATES */
    u_i s_f_c;
    fscanf(fr, "%u\n", &s_f_c);
    for (i = 0; i < s_f_c; i++) {
      fscanf(fr, "%u|", &j);
      n[k].s[j].f = 1;
    }
  }
  // close parse file
  if (fclose(fr) == EOF) {
    fprintf(stderr, "Error: file \"%s\" could not be closed!\n", FileName);
  }
}

/* Passing through automat with chars from file
 * which is second argument of program. */
void passing_automat(char *FileName) {
  FILE *fr;
  int c;                        /* one input char from file */
  u_i cur_state = start_state;  /* current state */
  u_i i, j, k, l, m, flag;      /* helpful variables */
  u_i num_char = 0;             /* number of actual working char */
  u_i num_tran_nfa = 0;         /* number of performed transitions in NFA */
  u_i num_tran_dfa = 0;         /* number of performed transitions in DFA */
  u_i match = 0;            /* indication of any match */

  // open data file
  if ((fr = fopen(FileName, "rb")) == NULL) {
    fprintf(stderr, "Error: file \"%s\" could not be open for reading!\n",
    FileName);
    free_memory();
    exit(1);
  }
  /* Follow algorithm for way through automat. */
  while ((c = getc(fr)) != EOF) {
    ++num_char;
  /* NFA part. */
    // look for active NFA
    for (i = 0; i < n_c; i++) {
      for (j = 0; j < n[i].s_c; j++) {
        // active state in NFA
        if (n[i].s[j].a == true) {
          // in current state try all transitions
          for (k = 0; k < n[i].s[j].t_c; k++) {
            for (l = 0; l < n[i].s[j].t[k].c_c; l++) {
              // found transition for working char
              if (n[i].s[j].t[k].c[l] == c) {
                ++num_tran_nfa;
                // deactivate current state
                if (n[i].s_s != n[i].s[j].r || n[i].c_c == true) {
                  n[i].s[j].a = false;
                }
                // activate destination state after all states
                // 'for' because of real state number
                for (m = 0; m < n[i].s_c; m++) {
                  if (n[i].s[m].r == n[i].s[j].t[k].d) {
                    n[i].s[m].a_a = true;
                    break;
                  }
                }
                // check destination state about final state
                if (n[i].s[m].f == true) {
                  printf("*** match on char %u ***\n", num_char);
                  match = 1;
                }
              }
            }
          }
          // did NOT found any transition for working char
          // deactivate current state
          if (n[i].s_s != n[i].s[j].r || n[i].c_c == true) {
            n[i].s[j].a = false;
          }
        }
      }
      // activate destination states
      for (m = 0; m < n[i].s_c; m++) {
        if (n[i].s[m].a_a == true) {
          n[i].s[m].a_a = false;
          n[i].s[m].a = true;
        }
      }
    }
  /* DFA part. */
    flag = 0;
    // in current state try all transitions
    for (j = 0; j < state[cur_state].trans_count; j++) {
      for (k = alp_in_begin[state[cur_state].trans_char[j]];
         k <= alp_in_end[state[cur_state].trans_char[j]]; k++) {
        // found transition for actual working char ->
        if (c == alphabet[k]) {
          ++num_tran_dfa;
          // -> move to next state
          cur_state = state[cur_state].trans_dest[j];
          // check about current state activating any NFA
          for (i = 0; i < n_c; i++) {
            if (cur_state == n[i].act) {
              // activate starting state
              n[i].s[n[i].s_s].a = true;
            }
          }
          flag = 1;
          break;
        }
      }
      // found transition for actual working char ->
      if (flag) {
        // -> skip working on other transitions
        break;
      }
    }
    // was NOT found any transition for actual working char
    if (!flag) {
      // unknown char for current state -> go to start state
      ++num_tran_dfa;
      cur_state = start_state;
      // skip looking for finite state and
      // immediately move to next char
      continue;
    }
    // algorithm is in any finite state of automat
    for (i = 0; i < count_final_states; i++) {
      if (cur_state == final_states[i]) {
        printf("*** match on char %u ***\n", num_char);
        match = 1;
      }
    }
  }
  // algorithm was NOT anytime in finite state
  if (!match) 
    printf("*** Result: NOT match ***\n");
  free_memory();
  printf("Number overall executed transitions: %u\n",
      num_tran_dfa + num_tran_nfa);
  printf("Number executed transitions in DFA: %u\n", num_tran_dfa);
  printf("Number executed transitions in NFA: %u\n", num_tran_nfa);
  if (fclose(fr) == EOF) {
    fprintf(stderr, "Error: file \"%s\" could not be closed!\n", FileName);
    exit(1);
  }
}

/* Main function. */
int main(int argc, char *argv[]) {
  if (argc == 3) {
    parse_file(argv[1]);
    passing_automat(argv[2]);
  } else {
    fprintf(stderr, "Error: bad arguments!\n");
    fprintf(stderr, "Example: ./hybrid_fa parse_file data_file\n");
    exit(1);
  }
  return 0;
}

