############################################################################
#  ddfa.py: Module for PATTERN MATCH algorithm Delayed Input DFA
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol
############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or
#  implied warranties, including, but not limited to, the implied warranties
#  of merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""Module for pattern match: algorithm for Delayed Input DFA."""

# Based on "Algorithms to accelerate multiple regular expressions matching
# for deep packet inspection", ISBN: 1-59593-308-5
# URL: http://portal.acm.org/citation.cfm?id=1159952

from netbench.pattern_match import sym_char, sym_char_class
from netbench.pattern_match import msfm_parser
from netbench.pattern_match.b_dfa import b_dfa
import os

class DELAY_DFA(b_dfa):
  """Class for Delayed Input DFA automat."""
  
  def __init__(self, FileName, d_b = 0):
    """Construct basic items and make automat."""

    b_dfa.__init__(self)
    self.make_ddfa(FileName, d_b)

  def make_ddfa(self, FileName, d_b):
    """Function for make Delay DFA from RE in FileName.
    Second argument is 'diameter bound', if is not set then construction
    of default transitions will be without bound."""

    # Parse input file
    parser = msfm_parser.msfm_parser()
    parser.load_file(FileName)
    # Make automat from RE which was in input file
    self.dfa = b_dfa()
    os.chdir("../..")
    self.dfa.create_by_parser(parser)
    os.chdir("algorithms/delay_dfa/")
    # Make Deterministic FA
    self.dfa.determinise(False)
    self.dfa_tran = len(self.dfa._automaton.transitions)
# Make Delay DFA from Deterministic FA (just make default transitions).
    weight_set = {}
    # sort transitions
    sort_trans = {}
    for i in range(0, len(self.dfa._automaton.states.keys()), 1):
      sort_trans[i] = []
    for trans in self.dfa._automaton.transitions:
      sort_trans[trans[0]].append(trans[1:])
    # for each combination of 2 states compute their
    # weight of reduction transitions
    for i in range(0, len(self.dfa._automaton.states.keys()) - 1, 1):
      for j in range(i + 1, len(self.dfa._automaton.states.keys()), 1):
        space_reduction = 0
        # compute number of same transitions
        for k in range(0, len(sort_trans[i]), 1):
          if sort_trans[i][k] in sort_trans[j]:
            space_reduction += 1
        # countdown by 1 because of default transition
        space_reduction = space_reduction - 1
        # append combination of states i and j to weight_set,
        # if they got reduction
        if space_reduction > 0:
          if space_reduction in weight_set.keys():
            weight_set[space_reduction].append((i,j))
          else :
            weight_set[space_reduction] = []
            weight_set[space_reduction].append((i,j))
    # follow code for refined_max_span_tree
    a_s = []  # states already added to spanning tree(s)
    s_t = []  # spanning trees
    for i in range(255, 0, -1):
      if weight_set.has_key(i):
        # select edge from weight_set[i] which leads to the smallest
        # growth in the diameter of the default tree
        while len(weight_set[i]) != 0:
          for_delete = []
          small_edge = 0    # smallest growth diameter edge
          small_edge_d = 0  # diameter of smallest edge
          small_edge_t = 0  # tree in which is the smallest edge
          for edge in weight_set[i]:
            new_tree = True
            for tree in s_t:
              # do not add edge which states are already in tree
              if edge[0] in tree.s and edge[1] in tree.s:
                # do not add this edge to any tree, just skip it
                for_delete.append(edge)
                new_tree = False
                break
              # in tree exist one state from edge
              elif edge[0] in tree.s or edge[1] in tree.s:
                new_tree = False
                # edge which joining two trees
                if edge[0] in a_s and edge[1] in a_s:
                  # find second tree
                  if edge[0] in tree.s:
                    tree0 = tree
                    for t in s_t:
                      if edge[1] in t.s:
                        tree1 = t
                  else :
                    tree1 = tree
                    for t in s_t:
                      if edge[0] in t.s:
                        tree0 = t
                  # compute diameter
                  diameter = compute_diameter(tree0, edge, d_b)
                  diameter += compute_diameter(tree1, edge, d_b)
                # edge which joining two states in same tree
                else :
                  diameter = compute_diameter(tree, edge, d_b)

                # edge do not maintain diameter bound
                if diameter == 0:
                  for_delete.append(edge)
                  break
                # lowest diameter, immediately join edge to tree
                elif diameter == 1:
                  if edge[0] in tree.s:
                    tree.s.append(edge[1])
                    tree.n_s[edge[0]].append(edge[1])
                    tree.n_s[edge[1]] = [edge[0]]
                    a_s.append(edge[1])
                  else :
                    tree.s.append(edge[0])
                    tree.n_s[edge[1]].append(edge[0])
                    tree.n_s[edge[0]] = [edge[1]]
                    a_s.append(edge[0])
                  for_delete.append(edge)
                  break
                # check the smallest edge
                else :
                  if small_edge_d == 0:
                    small_edge_d = diameter
                    small_edge = edge
                    small_edge_t = s_t.index(tree)
                  elif diameter < small_edge_d:
                    small_edge_d = diameter
                    small_edge = edge
                    small_edge_t = s_t.index(tree)
            # there is no tree to which we can add current edge
            if new_tree:
              # create new tree and add current edge to it
              s_t.append(SPANNING_TREE())
              s_t[-1].s = [edge[0], edge[1]]
              s_t[-1].n_s[edge[0]] = [edge[1]]
              s_t[-1].n_s[edge[1]] = [edge[0]]
              a_s.append(edge[0])
              a_s.append(edge[1])
              for_delete.append(edge)
          # adding the smallest (in growth diameter) edge in the tree
          if small_edge_d != 0:
            # edge connects two trees
            if small_edge[0] in a_s and small_edge[1] in a_s:
              # find second tree
              if small_edge[0] in s_t[small_edge_t].s:
                tree0 = s_t[small_edge_t]
                for t in s_t:
                  if small_edge[1] in t.s:
                    tree1 = t
              else :
                tree1 = s_t[small_edge_t]
                for t in s_t:
                  if small_edge[0] in t.s:
                    tree0 = t
              # merger trees (into first tree - 0)
              tree0.s += tree1.s
              tree0.n_s.update(tree1.n_s)
              s_t.remove(tree1)
              tree0.n_s[small_edge[0]].append(small_edge[1])
              tree0.n_s[small_edge[1]].append(small_edge[0])
            # 'normally' connect edge to the tree
            else :
              if small_edge[0] in s_t[small_edge_t].s:
                s_t[small_edge_t].s.append(small_edge[1])
                s_t[small_edge_t].n_s[small_edge[0]].append(small_edge[1])
                s_t[small_edge_t].n_s[small_edge[1]] = [small_edge[0]]
                a_s.append(small_edge[1])
              else :
                s_t[small_edge_t].s.append(small_edge[0])
                s_t[small_edge_t].n_s[small_edge[1]].append(small_edge[0])
                s_t[small_edge_t].n_s[small_edge[0]] = [small_edge[1]]
                a_s.append(small_edge[0])
            for_delete.append(small_edge)
          # delete the marked edges
          for m_edge in for_delete:
            weight_set[i].remove(m_edge)
    # selection of root nodes
    # root -> node which have shortest way to all others states
    roots = [-1] * len(s_t) # root nodes
    for tree in s_t:
      best_state_d = 0  # best state diameter
      for s in tree.s:
        distance = 1
        diameter = 0
        computed = [s]
        not_computed = tree.n_s[s]
        while len(not_computed) != 0:
          diameter += len(not_computed) * distance
          distance += 1
          computed += not_computed
          tmp = []
          for state in not_computed:
            for x in tree.n_s[state]:
              if x not in computed:
                tmp.append(x)
          not_computed = tmp
        if best_state_d == 0:
          roots[s_t.index(tree)] = s
          best_state_d = diameter
        elif diameter < best_state_d:
          roots[s_t.index(tree)] = s
          best_state_d = diameter
    # directing the other states in the tree to the root
    self.def_trans = {} # default transitions
    for tree in s_t:
      for n_s in tree.n_s[roots[s_t.index(tree)]]:
        self.def_trans[n_s] = roots[s_t.index(tree)]
      computed = [roots[s_t.index(tree)]]
      not_computed = tree.n_s[roots[s_t.index(tree)]]
      while len(not_computed) != 0:
        computed += not_computed
        tmp = []
        for s in not_computed:
          for n_s in tree.n_s[s]:
            if n_s not in computed:
              self.def_trans[n_s] = s
              tmp.append(n_s)
        not_computed = tmp
    # replacement of the same transitions as the default transitions
    a_l = len(self.dfa._automaton.alphabet.keys())  # alphabet length
    self.dfa._automaton.alphabet[a_l] = \
    sym_char_class.b_Sym_char_class("def", "def", a_l)
    for s in self.def_trans.keys():
        # delete same transitions
        for k in range(0, len(sort_trans[s]), 1):
          if sort_trans[s][k] in sort_trans[self.def_trans[s]]:
            self.dfa._automaton.transitions.remove  \
            ((s, sort_trans[s][k][0], sort_trans[s][k][1]))
        # replacement of the deleted transitions to default transition
        self.dfa._automaton.transitions.add((s, a_l, self.def_trans[s]))
    self.dfa._automaton.Flags["Delay DFA"] = True

  def show(self, FileName):
    """Print states, alphabet, start, transitions, final, Flags of current
    automat. And save graphviz dot file, representing graphical structure
    of nfa_data."""

    print '*' * 80
    print "STATES:", self.dfa._automaton.states, '\n'
    print "ALPHABET:", self.dfa._automaton.alphabet, '\n'
    print "START STATE:", self.dfa._automaton.start, '\n'
    print "TRANSITIONS:", self.dfa._automaton.transitions, '\n'
    print "FINAL STATES:", self.dfa._automaton.final, '\n'
    print "FLAGS OF AUTOMAT:", self.dfa._automaton.Flags
    print '*' * 80
    self.dfa._automaton.Show(FileName)

  def report_memory(self):
    """Print some useful numbers about memory requirement."""

    a = self.dfa._automaton

    print "\nReport memory:\nNumber of transitions: DFA =", self.dfa_tran, \
        "--- Delay DFA =", len(a.transitions), '\n'

  def SaveToFile(self, FileName):
    """Make file which represent the Delay DFA automat.
    This file will be input into algorithm written in C language."""

    a = self.dfa._automaton   # shortcut
    fw = open(FileName, 'w')

    # write the number of states
    fw.write(str(len(a.states)) + '\n')
    # write ALPHABET
    alphabet = ""
    length = ""
    pom_length = 0
    for index in range(0, len(a.alphabet.keys()), 1):
      if isinstance(a.alphabet[index], sym_char.b_Sym_char):
        alphabet += str(ord(str(a.alphabet[index]))) + '|'
        length += str(pom_length) + "->" + str(pom_length) + '|'
        pom_length += 1
      else:
        for char in a.alphabet[index].charClass:
          alphabet += str(ord(char)) + '|'
        length += str(pom_length) + "->" + str(pom_length + \
           len(a.alphabet[index].charClass) - 1) + '|'
        pom_length += len(a.alphabet[index].charClass)
    fw.write(str(len(a.alphabet.keys())) + '\n' + length + '\n')
    length_alphabet = len(alphabet.split('|')) - 1
    fw.write(str(length_alphabet) + "->" + alphabet + '\n')
    # write START STATE
    fw.write(str(a.start) + '\n')
    # write TRANSITIONS
    sort = {} # sorted transitions
    for s in range(0, len(a.states)):
      sort[s] = []
    t_str = ""  # transitions in string
    for t in a.transitions:
      if t[1] != 'def':
        sort[t[0]].append(t[1:])
    c_t = ""  # count transitions for state
    for s in range(0, len(a.states)):
      c_t += str(len(sort[s])) + '|'
      for i in range(0, len(sort[s])):
        t_str += str(sort[s][i][0]) + '->' + str(sort[s][i][1]) + '|'
    fw.write(c_t + '\n' + t_str + '\n')
    # write FINAL STATES
    final = ""
    for x in list(a.final):
      final += str(x) + '|'
    fw.write(str(len(a.final)) + '\n' + final + '\n')
    # write DEFAULT TRANSITIONS
    start = str(a.start)
    d_t = ""  # default transitions in string
    for state in range(0, len(a.states), 1):
      if self.def_trans.has_key(state):
        d_t += str(self.def_trans[state]) + '|'
      else :
        d_t += str(-1) + '|'
    fw.write(d_t + '\n')

class SPANNING_TREE():
  """Class for representation one spanning tree, which are needed for
  finding default transitions."""

  def __init__(self):
    """Construct basic items."""

    self.s = []   # states in tree
    self.n_s = {} # neighbours states

# Follow helpful functions
def compute_diameter(tree, edge, d_b):
  """Function to count the diameter of connecting edge to the tree."""

  distance = 1
  diameter = 0

  if edge[0] in tree.s:
    s = edge[0]
  else :
    s = edge[1]
  computed = [s]
  not_computed = tree.n_s[s]
  while len(not_computed) != 0:
    diameter += len(not_computed) * distance
    # edge do not maintain the diameter bound
    if d_b != 0 and distance >= d_b:
      return 0
    distance += 1
    computed += not_computed
    tmp = []
    for state in not_computed:
      for x in tree.n_s[state]:
        if x not in computed:
          tmp.append(x)
    not_computed = tmp
  return diameter

