###############################################################################
#  ipv6parser.py: Module for parsing IPv6 prefixes
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol 
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""Module providing basic parser for IPv6 prefixes."""

import string
import bprefixparser
from netbench.classification import prefix
from netbench.classification import prefixset

class Ipv6Parser(bprefixparser.BPrefixParser):
	"""Simple prefix parser for IPv6."""

	def __init__(self):
		"""Constructor."""
		bprefixparser.BPrefixParser.__init__(self)
		self.lines = 0
		self.parsed_prefixes = 0
		self.nonvalid_prefixes = 0

	def load_file(self, filename):
		"""Load prefixes from a file, return True if parsing was successful.
		
		filename: Name of file in appropriate format."""

		# try to open a file
		try:
			self._file = open(filename, 'r')
		except IOError:
			return 
		self.lines = 0
		self.parsed_prefixes = 0
		self.nonvalid_prefixes = 0
		# parse each line in a file
		for line in self._file:
			self.lines += 1
			line = line.strip()
			# prefix is '*'
			if line == '*':
				# create specific prefix that match any IP
				new_prefix = prefix.Prefix(domain_size=128, value=0, length=0, display_format="prefix6")
				self.prefixset.add_prefix(new_prefix)
				self.parsed_prefixes += 1
				continue
			# split line in an array, each item of array is two byte from given prefix
			array = line.split(':')
			length = len(array)
			# prefix is not valid
			if length <= 2 or length >= 9:
				self.nonvalid_prefixes += 1
				continue
			# split last item (prefix length) of an array
			tmp = array[-1].split('/')
			array[-1] = tmp[0]
			array.append(tmp[1])
			flag = 0
			# check fields for correct values and then convert them to decimal
			for i in range(0, len(array) - 1):
				for j in array[i]:
					if j not in string.hexdigits:
						self.nonvalid_prefixes += 1
						flag = 1
						break
				if flag:
					break
				elif array[i] != '':
					if array[i+1] == '':
						array[i] = array[i] + (4 - len(array[i])) * '0'
					array[i] = int(array[i], 16)
			if flag:
				continue
			# convert the last item (prefix length) to int and check value
			if not array[-1].isdigit() or int(array[-1]) < 0 or int(array[-1]) > 128:
				self.nonvalid_prefixes += 1
				continue
			array[-1] = int(array[-1])
			# expand :: to zeros
			while array.count('') != 0:
				array = array[:array.index('')] + [0] * (8 + 1 - len(array[:-1])) + array[array.index('') + 1:]
			# set prefix on 8 fields
			array = array[:-1] + (8 - len(array[:-1])) * [0] + [array[-1]]
			value = 0
			for i in range(0, 8):
				value += array[i]
				value = value << 16
			value = value >> 16
			# prefix seems to be OK, create new object Prefix and add it to Prefixset
			new_prefix = prefix.Prefix(domain_size=128, value=value, length=array[-1], display_format="prefix6")
			self.prefixset.add_prefix(new_prefix)
			self.parsed_prefixes += 1

		self._file.close()
		return True

	def get_parsed_prefixes(self):
		"""Returns count of successful parsed prefixes from last loaded file."""
		return self.parsed_prefixes

	def get_nonvalid_prefixes(self):
		"""Returns count of nonvalid prefixes in last loaded file."""
		return self.nonvalid_prefixes

	def get_count_of_lines(self):
		"""Returns count of lines in last loaded file."""
		return self.lines

	def get_lines_of_file(self, filename):
		"""Returns count of lines in given file."""
		try:
			self._file = open(filename, 'r')
		except IOError:
			return -1

		count_lines = len(self._file.readlines())
		self._file.close()
		return count_lines
