###############################################################################
#  ipv4parser.py: Module for parsing IPv4 prefixes
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Martin Skacan <xskaca00@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""
Module for parsing IPv4 prefixes
"""

import sys
import bprefixparser
from netbench.classification import prefix
from netbench.classification import prefixset

class Ipv4Parser(bprefixparser.BPrefixParser):
    """
    Simple prefix parser for IPv4
    """
    
    def __init__(self):
        """Constructor"""
        bprefixparser.BPrefixParser.__init__(self)
        self.lines = 0
        self.parsed_prefixes = 0
        self.invalid_prefixes = 0
    
    def load_file(self, filename):
        """
        Load prefixes from a file, return True if parsing was successfull.

        filename: Name of file in appropriate format.
        """
        
        # try to open a file
        try:
            self._file = open(filename, 'r')
        except IOError:
            return False
        
        self.lines = 0
        self.parsed_prefixes = 0
        self.invalid_prefixes = 0
        # parse each line in a file
        for line in self._file:
            self.lines += 1
            tmp = line.split(' ')
            line = tmp[0]
            line = line.strip('\n ')
            # if prefix is '*'
            if (line == '*' or line == "any"):
                # create specific prefix that match any IP
                new_prefix = prefix.Prefix(domain_size=32, value=0, length=0,display_format="ipv4_prefix")
                self.prefixset.add_prefix(new_prefix)
                self.parsed_prefixes += 1
                continue
            # split line in an array. Each item of array is one byte from given prefix
            array = line.split('.')
            length = len(array)
            # prefix is not valid
            if (length > 4 or length < 2 or array[0] == ''):
                self.invalid_prefixes += 1
                continue
            # if prefix ends with '*'
            if array[-1] == '*' or array[-1] == '0':
                # replace this form by mask-notation
                if (length == 2):
                    array[1]='0'
                    array.append('0')
                    array.append('0/8')
                elif (length == 3):
                    array[2]='0'
                    array.append('0/16')
                else:
                    array[-1] = '0/24'
            
            # split last item of an array
            # array[0] - array[3] will be bytes of prefix
            # last item - array[4] will be mask of prefix (number of valid bits)                      
            tmp = array[-1].split('/')
            if (not tmp[0].isdigit() or len(tmp) != 2):
                self.invalid_prefixes += 1
                continue
            array[-1] = tmp[0]
            array.append(tmp[1])
            flag = 0
            value = 0
            # check array items and compute value of the prefix
            for i in range(0,4):
                if (not array[i].isdigit() or int(array[i]) < 0 or int(array[i]) > 255):
                    flag = 1
                    break
                value += int(array[i])
                value = value << 8
            if (flag):
                self.invalid_prefixes += 1
                continue
            value = value >> 8
            # check the length of the prefix
            if (not array[4].isdigit() or int(array[4]) < 0 or int(array[4]) > 32):
                self.invalid_prefixes += 1
                continue
            length = int(array[4])
            
            # prefix seems to be OK, create new object Prefix and add it to Prefixset
            new_prefix = prefix.Prefix(domain_size=32, value=value, length=length, display_format="ipv4_prefix")
            self.prefixset.add_prefix(new_prefix)
            self.parsed_prefixes += 1
            
        self._file.close()
        return True
    
    def get_parsed_prefixes(self):
        """ returns count of successful parsed prefixes from last loaded file """
        return self.parsed_prefixes
        
    def get_invalid_prefixes(self):
        """ returns count of invalid prefixes in last loaded file """
        return self.invalid_prefixes
        
    def get_count_of_lines(self):
        """ returns count of lines in last loaded file """
        return self.lines
        
    def get_lines_of_file(self, filename):
        """ returns count of lines in given file """
        try:
            self._file = open(filename, 'r')
        except IOError:
            return -1
            
        count_lines = len(self._file.readlines())
        self._file.close()
        return count_lines
            
        
        
        
