###############################################################################
#  trie.py: Module for unibit Trie LPM algorithm
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Martin Skacan <xskaca00@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""
Module for unibit Trie LPM algorithm
"""

import sys
import blpm
from netbench.classification import prefixset
from netbench.classification import maskedint

class Trie(blpm.BLPM):
    """
    Trie LPM algorithm for lookup prefixes
    """

    def __init__(self):
        """Constructor"""
        blpm.BLPM.__init__(self)
        self.prefixes = 0
        self.nodes = 0
        self.max_depth = 0

    def report_memory(self):
        """
        Return detailed info about algorithm memory requirements.
        """
        report = {'prefixes': self.prefixes, 'nodes': self.nodes,
                  'pointers': self.nodes - 1, 'depth': self.max_depth}
        return report
        
    def load_prefixset(self, prefixset):
        """
        Load prefixes and generate all necessary data structures. 
        """
        
        self.prefixset = prefixset
        self.tree = Tree()
        # add each prefix from prefixset into tree
        for p in self.prefixset.get_prefixes():
            key = ""
            # if prefix length > 0
            if p.get_length():
                # convert prefix into binary notation and use it as a key into the tree
                key = bin(p.get_value() >> (p.get_domain_size() - p.get_length()))[2:]
                key = (p.get_length() - len(key))*"0" + key
            self.tree.add_prefix(key, p)
            self.prefixes += 1
        # get count of nodes and height of the tree
        self.nodes = self.tree.nodes
        self.max_depth = self.tree.max_depth
        

    def lookup(self, ip):
        """
        Lookup prefixes that match ip.
        Return the list of matched prefixes. 
        If ip matches no prefix, the list will be empty.
         
        ip: value of the ip address according to maskedint.py
        """
        
        # list of prefixes that match ip     
        list = []
        # ip in binary notation
        key = bin(ip)[2:]
        key = (2**maskedint.log2(len(key)) - len(key))*"0" + key
        # begin in the root of the tree
        current_node = self.tree.root
        # searching in the tree according to key
        for i in key:
            # we are at the end of the tree (no more nodes in required direction)
            if current_node == None:
                break   
            # if current node has a value, it means that it contains a valid prefix
            if (current_node.value):
                # add prefix into the list
                list.append(current_node.value)
            # continue to the left child
            if i == "0":
                current_node = current_node.lchild
            # continue to the right child
            elif i == "1":
                current_node = current_node.rchild
            else:
                return []
                
        # check the last node
        if (current_node and current_node.value):
            list.append(current_node.value)
        
        # reverse list to be sorted in descentant order
        list.reverse()
        return list


    def display(self):
        """Display the structure of the tree"""
        
        # create shortcuts for prefixes in the PrefixSet
        shortcut = {}
        i = 0
        for prefix in self.prefixset.get_prefixes():
            i += 1
            shortcut[prefix] = "P" + str(i)
            
        stack = []
        current_node = self.tree.root
        # if root has a valid prefix, print it's shortcut
        if current_node.value:
            p = "(" + shortcut[current_node.value] + ")"
        else:
            p = ""
        print "Root" + p
        
        # traverse the tree in pre-order and print the stucture of the tree
        # and shortcuts of valid prefixes in the nodes
        while current_node:
            stack.append(current_node)
            if current_node.value:
                p = "(" + shortcut[current_node.value] + ")"
            else:
                p = ""
            if current_node.key:
                print current_node.depth * "  " + "-" + current_node.key[-1] + p
            current_node = current_node.lchild
        while stack:
            current_node = stack.pop().rchild
            while current_node:
                stack.append(current_node)
                if current_node.value:
                    p = "(" + shortcut[current_node.value] + ")"
                else:
                    p = ""
                if current_node.key:
                    print current_node.depth * "  " + "-" + current_node.key[-1] + p
                current_node = current_node.lchild
                
class Node(object):
    """
    Common class necessary for the class Trie
    Key is the path from root to the current node in the binary notation
    Value is reference to the Prefix if any
    Each Node can have two child - lchild and rchild
    """
    def __init__(self, key = "", value = None):
        self.key = key
        self.value = value
        self.depth = None
        self.parent = None
        self.lchild = None
        self.rchild = None
        self.subtree_nodes = 0
    

class Tree(object):
    """
    Common class necessary for the class Trie
    It consists of Nodes and represents the whole tree in which will be
    looking up the prefixes
    """
    def __init__(self):
        self.root = Node()
        self.root.depth = 0
        self.nodes = 1
        self.max_depth = 0
  
    def add_prefix(self, key, value):
        """
        Add prefix into tree. New Nodes will be created if needed 
        Key is prefix in the binary notation
        Value is reference to the Prefix
        """
    
        current_node = self. root
        for i in key:
            if i == "0":
                # if there is no left-child, create it and move to it
                if current_node.lchild == None:
                    current_node.lchild = Node(current_node.key + "0")
                    current_node.lchild.depth = current_node.depth + 1
                    if current_node.lchild.depth > self.max_depth:
                        self.max_depth = current_node.lchild.depth
                    current_node.lchild.parent = current_node
                    self.nodes += 1
                    current_node = current_node.lchild
                # move to the left-child
                else:
                    current_node = current_node.lchild
            elif i == "1":
                # if there is no right-child, create it and move to it
                if current_node.rchild == None:
                    current_node.rchild = Node(current_node.key + "1")
                    current_node.rchild.depth = current_node.depth + 1
                    if current_node.rchild.depth > self.max_depth:
                        self.max_depth = current_node.rchild.depth
                    current_node.rchild.parent = current_node
                    self.nodes += 1
                    current_node = current_node.rchild
                # move to the right-child
                else:
                    current_node = current_node.rchild
            else:
                return False
                
        # we are in the desired Node, store reference to the Prefix
        current_node.value = value
        return True
        