###############################################################################
#  treebitmap.py: Module for Tree-bitmap LPM algorithm
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Martin Skacan <xskaca00@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""
Module for Tree-bitmap LPM algorithm
"""

import sys
import blpm
from netbench.classification import prefixset
from netbench.classification import maskedint

##############################################################
# Niektore verzie Pythonu nepoznaju vstavanu funkciu bin() => staci odkomentovat

# def bin(n):
#    """
#    DEC to BIN conversion
#    """
#    bStr = ''
#    if n < 0: raise ValueError, "must be a positive integer"
#    if n == 0: return "0b0"
#    while n > 0:
#        bStr = str(n % 2) + bStr
#        n = n >> 1
#    return "0b" + bStr

###############################################################

class TreeBitmap(blpm.BLPM):
    """
    Tree Bitmap algorithm for lookup prefixes
    """

    def __init__(self, stride = 3):
        """Constructor"""
        blpm.BLPM.__init__(self)
        self.stride = stride
        self.prefixes = 0
        self.nodes = 0
        self.child_pointers = 0
        self.prefix_pointers = 0
        self.max_depth = 0

    def report_memory(self):
        """
        Return detailed info about algorithm memory requirements.
        """

        report = {'prefixes': self.prefixes, 'nodes': self.nodes,
                  'child_pointers': self.child_pointers,
                  'prefix_pointers': self.prefix_pointers,
                  'depth':self.max_depth}
        return report
        
    def load_prefixset(self, prefixset):
        """
        Load prefixes and generate all necessary data structures. 
        """
        
        self.prefixset = prefixset
        self.tree = Tree(self.stride)
        # add each prefix from prefixset into tree
        for p in self.prefixset.get_prefixes():
            key = ""
            if p.get_length():
                # convert prefix into binary notation and use it as a key into the tree
                key = bin(p.get_value() >> (p.get_domain_size() - p.get_length()))[2:]
                key = (p.get_length() - len(key))*"0" + key
            self.tree.add_prefix(key, p)
            self.prefixes += 1
        # get stats about the created tree
        self.nodes = self.tree.nodes
        self.child_pointers = self.tree.child_pointers
        self.prefix_pointers = self.tree.prefix_pointers
        self.max_depth = self.tree.max_depth
        

    def lookup(self, ip):
        """
        Lookup prefixes that match ip.
        Return the list of matched prefixes. 
        If ip matches no prefix, the list will be empty.
         
        ip: value of the ip address according to maskedint.py
        """
        
        # list of prefixes that match ip     
        list = []
        # ip in binary notation
        key = bin(ip)[2:]
        key = (2**maskedint.log2(len(key)) - len(key))*"0" + key
        current_node = self.tree.root
        seq = ""
        # the key will be manipulated in parts. Size of the part depends on the stride
        for i in range(0, len(key), self.stride):
            # one part of the key with the size of the stride
            seq = key[i:i+self.stride]
            # it the part has the full length, it will be continued with the next part
            if len(seq) == self.stride:
                # number to the children-bitmap
                number = int(seq,2)
                # there is a node in the desired way
                if current_node.child_bitmap[number] == 1:
                    # store all valid prefixes to the list
                    for j in range(0, len(seq)):
                        tmpseq = seq[0:j]
                        if len(tmpseq) == 0:
                            position = 0
                        else:
                            position = 2**j - 1 + int(tmpseq,2)
                        if current_node.prefix_bitmap[position] == 1:
                            count = 0
                            for k in range(0, position):
                                if current_node.prefix_bitmap[k] == 1:
                                    count += 1
                            list.append(current_node.prefixes[count][1])
                # there is no more node in the desired way                 
                else:
                    break
                # compute the position of the next Node
                count = 0
                for j in range(0,number):
                    if current_node.child_bitmap[j] == 1:
                        count += 1
                seq = ""
                # move on the next Node
                current_node = current_node.children[count][1]
        
        # we handled as much of the key as possible
        # in the current_node is now the last node of the tree in desired way
        
        # check the last node and store all valid prefixes to the list
        if len(seq) == 0:
            if current_node.prefix_bitmap[0] == 1:
                list.append(current_node.prefixes[0][1])
        else:
            for i in range(0, len(seq)):
                tmpseq = seq[0:i]
                if len(tmpseq) == 0:
                    position = 0
                else:
                    position = 2**i - 1 + int(tmpseq,2)
                if current_node.prefix_bitmap[position] == 1:
                    count = 0
                    for j in range(0,position):
                        if current_node.prefix_bitmap[j] == 1:
                            count += 1
                    list.append(current_node.prefixes[count][1])
        
        # reverse list to be sorted in descentant order    
        list.reverse()
        return list


    def display(self, node):
        """Display the structure of the tree"""
        
        # create shortcuts for prefixes in the PrefixSet
        shortcut = {}
        i = 0
        for prefix in self.prefixset.get_prefixes():
            i += 1
            shortcut[prefix] = "P" + str(i)
        
        # traverse the tree in pre-order and print the stucture of the tree
        if node:
            print node.depth *"  ", "---- NODE --------------------"
            print node.depth *"  ", "- prefix bitmap ", node.prefix_bitmap
            print node.depth *"  ", "- children bitmap ", node.child_bitmap
            print node.depth *"  ", "- prefixes in node:",
            for prefix in node.prefixes:
                print shortcut[prefix[1]],
            print "\n",node.depth *"  ", "------------------------------"
            for i in node.children:
                self.display(i[1])
        

class Node(object):
    """
    Common class necessary for the class TreeBitmap
    The size of the Node (count of prefixes and children, size if bitmaps)
      depends on stride
    parent is reference on the parent-node
    child_bitmap is a bitmap of the children-nodes
    prefix_bitmap is a bitmap of the valid prefixes in the node
    children contains reference on the children nodes
    prefixes contains reference on the valid prefixes in the node
    children and prefixes contains values in format : (priority, reference)
      because they allways have to be sorted and with no gaps between the items
    """
    def __init__(self, stride = 3):
        self.depth = None
        self.parent = None
        self.child_bitmap = []
        self.prefix_bitmap = []
        self.children = []
        self.prefixes = []
        for i in range(0,2**stride):
            self.child_bitmap.append(0)
            self.prefix_bitmap.append(0)
        del self.prefix_bitmap[-1]


class Tree(object):
    """
    Common class necessary for the class TreeBitmap
    It consists of Nodes and it represents the whole tree in which will be
    looking up the prefixes
    """
    
    def __init__(self, stride = 3):
        self.stride = stride
        self.root = Node(self.stride)
        self.root.depth = 0
        self.nodes = 1
        self.child_pointers = 0
        self.prefix_pointers = 0
        self.max_depth = 0
  
    def add_prefix(self, key, value):
        """
        Add prefix into tree. New Nodes will be created if needed
        Key is prefix in the binary notation
        Value is reference to the Prefix
        """
    
        current_node = self.root
        seq = ""
        # the key will be manipulated in parts. Size of each part is the stride
        for i in range(0, len(key), self.stride):
            # one part of the key with the size of the stride
            seq = key[i:i+self.stride]
            # the part has the full length, it will be continued with the next part
            if len(seq) == self.stride:
                # number to the children-bitmap
                number = int(seq,2)
                # if the is no Node, create it
                if current_node.child_bitmap[number] == 0:
                    current_node.child_bitmap[number] = 1
                    new_node = Node(self.stride)
                    self.nodes += 1
                    new_node.parent = current_node
                    new_node.depth = new_node.parent.depth + 1
                    if new_node.depth > self.max_depth:
                        self.max_depth = new_node.depth
                    # current node has no children, yet == > increment children pointers
                    if not current_node.children:
                        self.child_pointers += 1
                    current_node.children.append((number, new_node))
                    current_node.children.sort()
                # compute the position of the next Node
                count = 0
                for j in range(0,number):
                    if current_node.child_bitmap[j] == 1:
                        count += 1
                seq = ""
                # move on the next Node
                current_node = current_node.children[count][1]

        # the key was whole handled, in the current_node is the desired node
        # store the prefix in this node
        
        if len(seq) == 0:
            number = 0
        else:
            number = 2**len(seq) - 1 + int(seq,2)
            
        # in currnet_node is no prefix, yet ==> increment prefix pointers
        if not current_node.prefixes:
            self.prefix_pointers += 1
        
        # store prefix and update bitmap
        current_node.prefix_bitmap[number] = 1
        current_node.prefixes.append((number, value))
        current_node.prefixes.sort()
