###############################################################################
#  cpe.py: Module for LPM algorithm Controlled Prefix Expansion
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol 
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""Module for LPM algorithm Controlled Prefix Expansion."""

import blpm
from netbench.classification import prefixset

class CPE(blpm.BLPM):
	"""LPM algorithm Controlled Prefix Expansion for lookup prefixes."""

	def __init__(self, n = 3):
		blpm.BLPM.__init__(self)
		self.n = n
		self.prefixes = 0
		self.nodes = 0
		self.max_depth = 0
		self.child_pointers = 0
		self.prefix_pointers = 0

	def report_memory(self):
		"""Print detailed info about algorithm memory requirements."""
		
		return {'prefixes': self.prefixes, 'nodes': self.nodes, 'pointers': self.nodes - 1, 'depth': self.max_depth,
				'child_pointers': self.child_pointers, 'prefix_pointers': self.prefix_pointers}

	def load_prefixset(self, prefixset):
		"""Load prefixes and generate all necessary data structures."""

		self.prefixset = prefixset
		self.tree = Tree(self.n)
		# add each prefix from prefixset into tree
		for p in self.prefixset.get_prefixes():
			p_in_bin = ""
			# transfer prefix into binary notation
			if p.get_length():
				p_in_bin = bin( p.get_value() >> (p.get_domain_size() - p.get_length()) )[2:]
				p_in_bin = '0' * (p.get_length() - len(p_in_bin)) + p_in_bin
			self.tree.add_prefix(p_in_bin, p)
			self.prefixes += 1
		self.nodes = self.tree.nodes
		self.child_pointers = self.tree.child_pointers
		self.prefix_pointers = self.tree.prefix_pointers
		self.max_depth = self.tree.max_depth

	def lookup(self, ip):
		"""Lookup prefix that match ip.
		Return matched prefix.
		If ip match no prefix then None will be returned
		
		ip: value to compare with"""

		# ip in binary notation
		key = bin(ip)[2:]
		key = '0' * (2**blpm.log2(len(key)) - len(key)) + key
		# begin in the root of the tree
		current_node = self.tree.root
		# searching in the tree according to key
		for i in range(0, len(key), self.n):
			bits = key[i:(i+self.n)]
			# is somethink saved on bits pozition on current node?
			if bits in current_node.pom.keys():
				# in way is next node
				if current_node.pom[bits] == "ptr":
					current_node = current_node.value[bits]
				# end of tree
				else :
					# return what we found
					return current_node.value[bits]
			# bits are not in node, so not prefix in this place
			else :
				return None

	def display(self):
		"""Display the structure of the tree."""

		way = [""]
		pointer_list = [self.tree.root, '|']
		while len(pointer_list) >= 2:
			if pointer_list[0] == '|':
				pointer_list.pop(0)
				pointer_list.append('|')
			current_node = pointer_list[0]
			print "+++ depth of node is", current_node.depth, "and way to node is", way[0], "+++"
			print current_node.value
			for i in current_node.pom.keys():
				if current_node.pom[i] == "ptr":
					pointer_list.append(current_node.value[i])
					way.append(way[0] + i)
			print '\n',
			pointer_list.pop(0)
			way.pop(0)

class	Node(object):
	"""Common class necessary for the class Tree and CPE.
	Key is list of numbers from 0 to (2**n - 1) in binary scheme.
	Value is reference to the Prefix or reference to next node.
	pom represent list where is stored if value is prefix or reference on next node."""

	def __init__(self, n = 3):
		if n <= 0:
			n = 1
			print "Error: 'n' must be greater then 0!"
		self.value = {}
		self.pom = {}
		self.depth = None

class Tree(object):
	"""Common class necessary for the class CPE.
	It consists of Nodes and represents the whole tree in 
	which will be looking up the prefixes."""

	def __init__(self, n = 3):
		self.n = n
		self.root = Node(self.n)
		self.root.depth = 0
		self.nodes = 1
		self.max_depth = 0
		self.child_pointers = 0
		self.prefix_pointers = 0

	def add_prefix(self, key, value):
		"""Add prefix into tree. New Nodes will be created if needed.
		key is prefix in the binary notation.
		value is reference to the Prefix."""

		current_node = self.root
		for i in range(0, len(key), self.n):
			bits = key[i:(i+self.n)]
			# prefix is long, need next node
			if len(key) - i >= self.n:
				# if isnt next node create it
				if bits not in current_node.pom.keys():
					current_node.value[bits] = Node(self.n)
					current_node.pom[bits] = "ptr"
					self.nodes += 1
					self.child_pointers += 1
					current_node.value[bits].depth = current_node.depth + 1
					current_node = current_node.value[bits]
				# next node exist, so move to it
				else :
					current_node = current_node.value[bits]
			# prefix will be stored in this node
			if len(key) - i <= self.n :
				# check max_depth of tree
				if current_node.depth > self.max_depth:
					self.max_depth = current_node.depth
				# list of indexes where are references to next nodes in current node
				update = [[]]
				if len(key) -i == self.n:
					bits = ""
				# expansion and storage prefix
				for expand in range(0, pow(2, self.n - len(bits))):
					pom = bin(expand)[2:]
					_index = bits + '0' * (self.n - len(pom) - len(bits)) + pom
					# is on current place somethink?
					if _index not in current_node.pom.keys():
						# not, safe prefix
						current_node.value[_index] = value
						current_node.pom[_index] = "value"
						self.prefix_pointers += 1
					# is on current place next node?
					elif current_node.pom[_index] == "ptr":
						# yes, insert index where is pointer to update list for later update
						update[0].append(_index)
					# recursively insert in the rest of tree prefix if you can
					nodes = [current_node]
					while len(update) > 0  and update[-1] != []:
						pom_node = nodes[-1]
						# is next node already full?
						if len(pom_node.value[update[-1][-1]].pom) == pow(2, self.n):
							update[-1].pop(-1)
						else :
							# next node is not full so we move to him
							pom_node = pom_node.value[update[-1][-1]]
							update[-1].pop(-1)
							update.append([])
							nodes.append(pom_node)
							# explore node and store prefix if you can
							for i in range(0, pow(2, self.n)):
								_index = '0' * (self.n - len(bin(i)[2:])) + bin(i)[2:]
								# is on current place prefix or pointer to next node?
								if _index not in pom_node.pom.keys():
									# not, safe prefix
									pom_node.value[_index] = value
									pom_node.pom[_index] = "value"
									self.prefix_pointers += 1
								# exist in this place pointer to next nodes?
								elif pom_node.pom[_index] == "ptr":
									# yes, insert pointer to update list for later explore
									update[-1].append(_index)
						while update != [[]] and update[-1] == []:
							update.pop(-1)
							nodes.pop(-1)
		# we are done
		return True
