###############################################################################
#  bsp.py: Module for LPM algorithm Binary Search on Prefixes
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""Module for 'binary search on prefixes' LPM algorithm."""

import blpm
from netbench.classification import prefixset

class BSP(blpm.BLPM):
	"""Binary Search on Prefixes LPM algorithm for lookup prefixes."""

	def __init__(self):
		"""Constructor."""

		blpm.BLPM.__init__(self)
		self.prefixes = 0
		self.nodes = 0
		self.max_depth = 0
		self.count_lpm_mark = 0

	def report_memory(self):
		"""Print detailed info about algorithm memory requirements."""

		return {'prefixes': self.prefixes, 'nodes': self.nodes, 'pointers': self.nodes - 1, 'depth': self.max_depth,
				'count_lpm_mark': self.count_lpm_mark}
        
	def load_prefixset(self, prefixset):
		"""Load prefixes and generate all necessary data structures."""

		self.prefixset = prefixset
		all_prefixes = []
		nodes = []
		nodes_length = []
		# skip prefix with length 0
		begin = 0
		if self.prefixset.get_prefixes()[-1].get_length() == 0:
			begin = 1
		# add each prefix from prefixset into tree
		for _index in range(len(self.prefixset.get_prefixes()) -1 - begin, -1, -1):
			p = self.prefixset.get_prefixes()[_index]
			p_in_bin = ""
			# transfer prefix into binary notation
			if p.get_length():
				p_in_bin = bin( p.get_value() >> (p.get_domain_size() - p.get_length()) )[2:]
				p_in_bin = '0' * (p.get_length() - len(p_in_bin)) + p_in_bin
			all_prefixes.insert(0, p_in_bin)
			# insert prefix into node
			if p.get_length() not in nodes_length:
				nodes_length.append(p.get_length())
				nodes.append(Node())
				nodes[-1].length = p.get_length()
			nodes[-1].prefixes.append(p_in_bin)
			nodes[-1].prefixes_index.append(_index)
			nodes[-1].prefixes_mark.append(False)
		# make tree
		self.root = nodes[len(nodes) / 2]
		self.root.lchild = nodes[0:len(nodes) / 2]
		self.root.rchild = nodes[(len(nodes) / 2) + 1:]
		self.root.depth = 0
		update = [self.root]
		while len(update) != 0:
			current_node = update[-1]
			update.pop(-1)
			if current_node.lchild != []:
				tmp = current_node.lchild
				current_node.lchild = tmp[len(tmp) / 2]
				current_node.lchild.depth = current_node.depth + 1
				current_node.lchild.lchild = tmp[0:len(tmp) / 2]
				current_node.lchild.rchild = tmp[(len(tmp) / 2) + 1:]
				update.append(current_node.lchild)
			else :
				current_node.lchild = None
			if current_node.rchild != []:
				tmp = current_node.rchild
				current_node.rchild = tmp[len(tmp) / 2]
				current_node.rchild.depth = current_node.depth + 1
				current_node.rchild.lchild = tmp[0:len(tmp) / 2]
				current_node.rchild.rchild = tmp[(len(tmp) / 2) + 1:]
				update.append(current_node.rchild)
			else :
				current_node.rchild = None
			# check max_depth of tree
			if current_node.depth > self.max_depth:
				self.max_depth = current_node.depth
		# insert marks and marks_lpm into nodes
		for node in nodes:
			_min = node.prefixes_index[-1]
			_max = node.prefixes_index[0]
			# find marks
			for mark in all_prefixes[0:_min]:
				mark = mark[0:node.length]
				# longer prefix exist, make mark for it
				if mark in node.prefixes:
					if node.prefixes_mark[node.prefixes.index(mark)] == False:
						node.prefixes_mark[node.prefixes.index(mark)] = True
					continue
				if mark not in node.marks:
					node.marks.append(mark)
					self.count_lpm_mark += 1
					# find lpm for mark
					for mark_lpm in all_prefixes[_max + 1:]:
						if mark[0:len(mark_lpm)] == mark_lpm:
							node.marks_lpm.append(all_prefixes.index(mark_lpm))
							break
					if len(node.marks) != len(node.marks_lpm):
						node.marks_lpm.append(None)
				# found all possible marks
				if (len(node.marks) + len(node.prefixes)) == pow(2, node.length):
					break
		self.prefixes = len(self.prefixset.get_prefixes())
		self.nodes = len(nodes)

	def lookup(self, ip):
		"""Lookup prefix that match ip.
		Return the matched prefix or None if we did not find anyone.

		ip: value to compare with"""
		
		# ip in binary notation
		key = bin(ip)[2:]
		key = '0' * (2**blpm.log2(len(key)) - len(key)) + key
		ip_bin = key
		# begin in the root of the tree
		current_node = self.root
		lpm = None
		# search in the tree according to key
		while current_node != None:
			key = ip_bin[:current_node.length]
			# search in prefixes
			if key in current_node.prefixes:
				lpm = current_node.prefixes_index[current_node.prefixes.index(key)]
				# longer prefix exist
				if current_node.prefixes_mark[current_node.prefixes.index(key)] == True:
					current_node = current_node.rchild
				# found LPM
				else :
					break
			# search in marks
			elif key in current_node.marks:
				lpm = current_node.marks_lpm[current_node.marks.index(key)]
				current_node = current_node.rchild
			# did not find prefix in current_node, so move to left child
			else :
				current_node = current_node.lchild
		# found nothing but prefix with length 0 exist (matching all)
		if lpm == None and self.prefixset.get_prefixes()[-1].get_length() == 0:
			lpm = len(self.prefixset.get_prefixes()) - 1
		if lpm == None:
			return None
		else :
			return self.prefixset.get_prefixes()[lpm]

	def display(self):
		"""Display the structure of the tree."""

		# NEHEZKE, NUTNO PREDELAT
		update = [self.root]
		way = [str(self.root.length)]
		while update != []:
			node = update.pop(0)
			tmp = way.pop(0)
			if node.lchild != None:
				update.append(node.lchild)
				way.append(tmp + " -> " + str(node.lchild.length))
			if node.rchild != None:
				update.append(node.rchild)
				way.append(tmp + " -> " + str(node.rchild.length))
			print "Node:", str(node.length) + '. Depth of node is', str(node.depth) + '. Way to node is', tmp
			print "Prefixes:",
			for i in node.prefixes_index:
				print self.prefixset.get_prefixes()[i], " | ",
			print "\nPrefixes mark:", node.prefixes_mark
			print "Marks:", node.marks
			print "Marks LPM: ",
			for i in node.marks_lpm:
				if i != None:
					print self.prefixset.get_prefixes()[i], " | ",
				else :
					print "None |",
			print "\n"

class Node(object):
	"""Common class necessary for the class BSP(bsp tree).
	'length' is length of prefix stored in this node.
	'prefixes' is list where are the prefixes(in binary notation) with length 'length'.
	'prefixes_index' is list of indexes to true location(of prefix) in prefixset.
	'prefixes_mark' is list of True|False, caring on longer prefix exist.
	'marks' is list of longer prefixes in binary notation.
	'marks_lpm' is list of LPM(indexes to prefixset) for 'marks'.
	Each Node can have two child - lchild and rchild."""

	def __init__(self):
		self.length = None
		self.prefixes = []
		self.prefixes_index = []
		self.prefixes_mark = []
		self.marks = []
		self.marks_lpm = []
		self.depth = None
		self.lchild = None
		self.rchild = None
