###############################################################################
#  bsi.py: Module for LPM algorithm binary search on intervals
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Jaroslav Suchodol
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""Module for LPM algorithm binary search on intervals."""

import blpm
from netbench.classification import prefixset

class BSI(blpm.BLPM):
	"""Class for LPM algorithm binary search on intervals."""

	def __init__(self):
		"""Constructor."""
		
		blpm.BLPM.__init__(self)
		self.prefixes = 0

	def report_memory(self):
		"""Print detailed info about algorithm memory requirements."""

		return {'prefixes': self.prefixes, 'size(lines) of table': len(self.table_p)}

	def load_prefixset(self, prefixset):
		"""Load prefixes and generate all necessary data structures."""

		self.prefixset = prefixset
		# lists needed for algorithm
		self.table_p = []
		self.table_p_bin = []
		self.pomlist = []
		self.begin = []
		self.end = []
		# add each prefix from prefixset into table
		for p in self.prefixset.get_prefixes():
			p_in_bin = ""
			# transfer prefix into binary notation
			if p.get_length():
				p_in_bin = bin( p.get_value() >> (p.get_domain_size() - p.get_length()) )[2:]
				p_in_bin = '0' * (p.get_length() - len(p_in_bin)) + p_in_bin
			# expand prefix to specific length with 0 and 1
			p_in_bin0 = p_in_bin + '0' * (p.get_domain_size() - len(p_in_bin))
			p_in_bin1 = p_in_bin + '1' * (p.get_domain_size() - len(p_in_bin))
			# insert prefix into table
			flag = 0
			shift = len(self.table_p) / 2
			i = shift
			# this while code is through faster search index
			# where save p_in_bin0
			while i > 0 and i < len(self.table_p):
				if shift < 2:
					shift = 2
				if int(p_in_bin0, 2) < int(self.table_p_bin[i], 2):
					if int(p_in_bin0, 2) >= int(self.table_p_bin[i - 1], 2):
						flag = 1
						break
					i = i - (shift / 2)
				else :
					i = i + (shift / 2)
				shift = shift / 2
			# we found interval where prefix can be placed
			if flag or (i == 0 and len(self.table_p) > 0):
				# save prefix to found interval, but do not overwrite same longer prefix
				if int(self.table_p_bin[i-1], 2) != int(p_in_bin0, 2):
					self.table_p.insert(i, p)
					self.pomlist.insert(i, "value")
					self.table_p_bin.insert(i, p_in_bin0)
					self.begin.insert(i, int(p_in_bin0, 2))
					self.end.insert(i, int(p_in_bin1, 2))
				flag = 0
				# where save p_in_bin1
				for j in range(i, len(self.table_p)):
					# interval where save p_in_bin1
					if int(p_in_bin1, 2) < int(self.table_p_bin[j], 2):
						flag = 1
						break
					# in path searching is place with 'None' prefix, so save there current prefix
					if self.pomlist[j] == None:
						# save prefix only if we do not end here
						if int(p_in_bin1, 2) != int(self.table_p_bin[j], 2):
							self.table_p[j] = p
							self.pomlist[j] = "value"
							self.begin[j] = int(p_in_bin0, 2)
							self.end[j] = int(p_in_bin1, 2)
				# save p_in_bin1 into found interval
				if flag:
					self.table_p.insert(j, None)
					self.pomlist.insert(j, None)
					self.table_p_bin.insert(j, p_in_bin1)
					self.begin.insert(j, None)
					self.end.insert(j, None)
				# did not found interval where save p_in_bin1, so append to end of list
				else :
					# do not save same p_in_bin1
					if int(p_in_bin1, 2) != int(self.table_p_bin[j], 2):
						self.table_p.append(None)
						self.pomlist.append(None)
						self.table_p_bin.append(p_in_bin1)
						self.begin.append(None)
						self.end.append(None)
			# did not found interval where save prefix, so append prefix to end of list
			else :
				self.table_p.append(p)
				self.pomlist.append("value")
				self.table_p_bin.append(p_in_bin0)
				self.begin.append(int(p_in_bin0, 2))
				self.end.append(int(p_in_bin1, 2))
				# append p_in_bin1 too
				self.table_p.append(None)
				self.pomlist.append(None)
				self.table_p_bin.append(p_in_bin1)
				self.begin.append(None)
				self.end.append(None)
			self.prefixes += 1

	def lookup(self, ip):
		"""Lookup prefix that match ip.
		Return the matched prefix. 
		If ip match no prefix, then None will be returned.
		
		ip: value to compare with"""

		shift = len(self.table_p) / 2
		i = shift
		# where in table can be placed ip
		while i > 0 and i < len(self.table_p):
			if shift < 2:
				shift = 2
			# move to lower index
			if ip < int(self.table_p_bin[i], 2):
				# ip is between i and i-1
				if ip >= int(self.table_p_bin[i - 1], 2):
					if self.begin[i - 1] != None and self.end[i - 1] != None and \
						ip >= self.begin[i - 1] and ip <= self.end[i - 1]:
						return self.table_p[i - 1]
					return None
				i = i - (shift / 2)
			# move to higher index
			else :
				i = i + (shift / 2)
			shift = shift / 2
		# LPM not found
		return None

	def display(self):
		"""Display the table of address|prefix."""
		
		# make some terrible table :)
		if len(self.table_p) > 0:
			len_decimal = len(str(int(self.table_p_bin[-1],2)))
			str_decimal = "address in decimal"
			len_binary = len(self.table_p_bin[0])
			str_binary = "address in binary"
			str_prefix = "prefix"
			if len_decimal < len(str_decimal):
				len_decimal = len(str_decimal)
			len_prefix = 0
			for i in range(0, len(self.table_p)):
				if len(str(self.table_p[i])) > len_prefix:
					len_prefix = len(str(self.table_p[i]))
			if len_prefix < len(str_prefix):
				len_prefix = len(str_prefix)
			if len_binary == 128:
				print '+' + '-' * (len_decimal + len_prefix + 5) + '+'
			else :
				print '+' + '-' * (len_decimal + len_binary + len_prefix + 8) + '+'
			print '|'+' '+('-' * (len_decimal - len(str_decimal))) + str_decimal,'|',
			if len_binary != 128:
				print ('-' * (len_binary - len(str_binary))) + str_binary,'|',
			print '-' * (len_prefix - len(str_prefix)) + str_prefix, '|'
			for i in range(0, len(self.table_p_bin)):
				print '|' + ' ' * (len_decimal - len(str(int(self.table_p_bin[i], 2)))), int(self.table_p_bin[i], 2),
				if len_binary != 128:
					print '|', self.table_p_bin[i],
				print '|' + ' ' * (len_prefix - len(str(self.table_p[i]))), self.table_p[i], '|'
			if len_binary == 128:
				print '+' + '-' * (len_decimal + len_prefix + 5) + '+'
			else :
				print '+' + '-' * (len_decimal + len_binary + len_prefix + 8) + '+'
		else :
			print "Error: have nothing to display."
