#!/usr/bin/env python

import string
import time
import struct

import packetheader
import protoheader
import netbench.common.headerfields as hf # There must be full path to avoid problems when comparing types

class Packet(object):
    """Class for Packet storage and packet header parsing.
    
    Parsed packet can contain this fields:
      ProtoHeader     fields
      -----------  ------------
       ethernet    typelength
       ipv4        version
                   header_len
                   tos
                   total_len
                   id
                   flags
                   fragment_offset
                   ttl
                   protocol
                   checksum
                   src_addr
                   dst_addr
                   options
       tcp         src_port
                   dst_port
                   data_offset
                   tcp_flags
       udp         src_port
                   dst_port
                   udp_len
                   udp_checksum
    """

    def __init__(self):
        self.parsed = False
        self.pckthdr = packetheader.PacketHeader()
        self.raw_data = []
        self.payload_offset = []
        tmphdr = protoheader.ProtoHeader("info")
        self.pckthdr.push_header(tmphdr)
    ""
    def dumphex(self):
        """Print out raw data of a packet in hexadecimal format."""
        s = self.raw_data
        bytes = map(lambda x: '%.2x' % x, map(ord, s))
        for i in xrange(0,len(bytes)/16):
            print '        %s' % string.join(bytes[i*16:(i+1)*16],' ')
        print '        %s' % string.join(bytes[(i+1)*16:],' ')

    def decode_l2(self, s, l2header):
        """Dissector for Ethernet header"""
        linktype = 0x0
        l2header.set_field('typelength', hf.Eth_TypeLength(0x0))
        if len(s) > 14:
            linktype = struct.unpack('!H', s[12:14])[0]
        if linktype == 0x800:
            l2header.set_field('typelength', hf.Eth_TypeLength(linktype))
            return 14
        else:
        #    l2header.set_field('typelength', 0x800)
            return 0

    def decode_ipv4(self, s, ipheader):
        """Dissector for IPv4 header"""
        if len(s) > 20:
            ipheader.set_field('version', hf.IP4_Version((ord(s[0]) & 0xf0) >> 4))
            ipheader.set_field('header_len', hf.IP4_HeaderLen(ord(s[0]) & 0x0f))
            ipheader.set_field('tos', hf.IP4_ToS(ord(s[1])))
            ipheader.set_field('total_len', hf.IP4_TotalLen(struct.unpack('!H',s[2:4])[0]))
            ipheader.set_field('id', hf.IP4_ID(struct.unpack('!H',s[4:6])[0]))
            ipheader.set_field('flags', hf.IP4_Flags((ord(s[6]) & 0xe0) >> 5))
            ipheader.set_field('fragment_offset', hf.IP4_FragmentOffset(struct.unpack('!H',s[6:8])[0]) & 0x1fff)
            ipheader.set_field('ttl', hf.IP4_TTL(ord(s[8])))
            ipheader.set_field('protocol', hf.IP4_Protocol(ord(s[9])))
            ipheader.set_field('checksum', hf.IP4_Checksum(struct.unpack('!H',s[10:12])[0]))
            ipheader.set_field('src_addr', hf.IP4_SrcAddress(struct.unpack('!I',s[12:16])[0]))
            ipheader.set_field('dst_addr', hf.IP4_DstAddress(struct.unpack('!I',s[16:20])[0]))
            if ipheader.get_field('header_len')>5:
                length = ipheader.get_field('header_len')-5 # length of otpions (in 4-byte words)
                opt_data = struct.unpack('!'+'I'*length,s[20:20+4*length]) # get data as tuple of 4-byte integers
                opt_data = reduce(lambda x,y: (x<<32)+y, opt_data) # convert to one long integer
                ipheader.set_field('options', hf.IP4_Options(opt_data))
            else:
                ipheader.set_field('options', None)
            return 4*ipheader.get_field('header_len')
        else:
            return 0

    def decode_udp(self, s, udpheader):
        """Dissector for UDP header"""
        if len(s) > 7:
            udpheader.set_field('src_port', hf.UDP_SrcPort(struct.unpack('!H',s[0:2])[0]))
            udpheader.set_field('dst_port', hf.UDP_DstPort(struct.unpack('!H',s[2:4])[0]))
            udpheader.set_field('udp_len',hf.UDP_Length(struct.unpack('!H',s[4:6])[0]))
            udpheader.set_field('udp_checksum',hf.UDP_Checksum(struct.unpack('!H',s[6:8])[0]))
       
        if len(s) > 4:
            udpheader.set_field('src_port', hf.UDP_SrcPort(struct.unpack('!H',s[0:2])[0]))
            udpheader.set_field('dst_port', hf.UDP_DstPort(struct.unpack('!H',s[2:4])[0]))
 
            return 8
        else:
            return 0 

    def decode_tcp(self, s, tcpheader):
        """Dissector for TCP header"""
        tcpheader.set_field('src_port', hf.TCP_SrcPort(struct.unpack('!H',s[0:2])[0]))
        tcpheader.set_field('dst_port', hf.TCP_DstPort(struct.unpack('!H',s[2:4])[0]))
        if (len(s) >= 13):
            tcpheader.set_field('data_offset', hf.TCP_DataOffset((ord(s[12]) & 0xf0) >> 4))
            tcpheader.set_field('tcp_flags', hf.TCP_Flags(ord(s[13]) & 0x3f))
            #print ord(s[14]) 
        else:
            tcpheader.set_field('data_offset', hf.TCP_DataOffset(0))
            tcpheader.set_field('tcp_flags', hf.TCP_Flags(0))
        return 4*tcpheader.get_field('data_offset')


    def parse(self, level=10):
        """Parse packet header up to a specified level.

           level
               Specifies the maximum level a packet header should be parsed to.
        """
        self.parsed = True
        endofparsing = 0
        self.payload_offset.append(endofparsing)
        tmpl2header = protoheader.ProtoHeader("ethernet")
        endofparsing = self.decode_l2(self.raw_data, tmpl2header)
        self.payload_offset.append(endofparsing)
        self.pckthdr.push_header(tmpl2header)
        if tmpl2header.get_field("typelength") == 0x800:
            tmpipheader = protoheader.ProtoHeader("ipv4")
            tmpend = self.decode_ipv4(self.raw_data[endofparsing:], tmpipheader)
            endofparsing = endofparsing + tmpend
            self.payload_offset.append(endofparsing)
            self.pckthdr.push_header(tmpipheader)
            if tmpipheader.get_field("protocol") == 6 and tmpend > 0:
                tmptcpheader = protoheader.ProtoHeader("tcp")
                tmpend = self.decode_tcp(self.raw_data[endofparsing:], tmptcpheader)
                endofparsing = endofparsing + tmpend
                self.payload_offset.append(endofparsing)
                if tmpend > 0:
                    self.pckthdr.push_header(tmptcpheader)
            elif tmpipheader.get_field("protocol") == 17 and tmpend > 0:
                tmpudpheader = protoheader.ProtoHeader("udp")
                tmpend = self.decode_udp(self.raw_data[endofparsing:], tmpudpheader)
                endofparsing = endofparsing + tmpend
                    
                self.payload_offset.append(endofparsing)
                if tmpend > 0:
                    self.pckthdr.push_header(tmpudpheader)
 
        return self.pckthdr


    def get_payload(self,type):
        """Returns payload of a packet belonging to a given protocol layer."""
        if not self.parsed:
            self.parse()
        level = self.pckthdr.get_headers_type().index(type)
        return self.raw_data[self.payload_offset[level]:]

    def get_packet_header(self):
        """Returns packet header"""
        if not self.parsed:
            self.parse()
        return self.pckthdr



