"""
Module providing several classes for hash functions.
"""

class JenkinsHash(object):
    """
    Bob Jenkins hash, as described at
    http://burtleburtle.net/bob/hash/doobs.html
    """

    def __init__(self, seed=0):
        """
        Constructor, optinally with seed parameter.
        """
        self._seed = seed
        self._last = seed # Last computed value

    def set_seed(self, seed):
        """
        Set seed of the hash function.

        seed: integer seed of the hash function.
        """
        self._seed = seed

    def get_seed(self):
        """
        Return seed of the hash function.
        """
        return self._seed

    def get_last(self):
        """
        Return last hash result.
        """
        return self._last

    def _mix(self, a, b, c):
        """
        Mix three unsigned integers a, b, c.
        """
        a = (a-b) & 0xFFFFFFFF # Keep at 32 bits
        a = (a-c) & 0xFFFFFFFF
        a = (a^(c>>13)) & 0xFFFFFFFF
        b = (b-c) & 0xFFFFFFFF
        b = (b-a) & 0xFFFFFFFF
        b = (b^(a<<8)) & 0xFFFFFFFF
        c = (c-a) & 0xFFFFFFFF
        c = (c-b) & 0xFFFFFFFF
        c = (c^(b>>13)) & 0xFFFFFFFF
        a = (a-b) & 0xFFFFFFFF
        a = (a-c) & 0xFFFFFFFF
        a = (a^(c>>12)) & 0xFFFFFFFF
        b = (b-c) & 0xFFFFFFFF
        b = (b-a) & 0xFFFFFFFF
        b = (b^(a<<16)) & 0xFFFFFFFF
        c = (c-a) & 0xFFFFFFFF
        c = (c-b) & 0xFFFFFFFF
        c = (c^(b>>5)) & 0xFFFFFFFF
        a = (a-b) & 0xFFFFFFFF
        a = (a-c) & 0xFFFFFFFF
        a = (a^(c>>3)) & 0xFFFFFFFF
        b = (b-c) & 0xFFFFFFFF
        b = (b-a) & 0xFFFFFFFF
        b = (b^(a<<10)) & 0xFFFFFFFF
        c = (c-a) & 0xFFFFFFFF
        c = (c-b) & 0xFFFFFFFF
        c = (c^(b>>15)) & 0xFFFFFFFF

        return a, b, c

    def hash(self, key):
        """
        Compute Jenkins Hash of the key.

        key: list of integers, one for each byte.
        """
        a = b = 0x9e3779b9
        c = self._seed

        remain = len(key)
        i = 0
        
        while (remain >= 12):
            a += key[i+0] + key[i+1]<<8 + key[i+2]<<16 + key[i+3]<<24
            a = a & 0xFFFFFFFF # Keep at 32 bits...

            b += key[i+4] + key[i+5]<<8 + key[i+6]<<16 + key[i+7]<<24
            b = b & 0xFFFFFFFF

            c += key[i+8] + key[i+9]<<8 + key[i+10]<<16 + key[i+11]<<24
            c = c & 0xFFFFFFFF

            a, b, c = self._mix(a, b, c)

            i += 12
            remain -= 12


        c = (c + remain) & 0xFFFFFFFF
        # the first byte of c is reserved for the length

        if (remain >= 11):
            c += (c+(key[i+10] << 24)) & 0xFFFFFFFF
        if (remain >= 10):
            c += (c+(key[i+9] << 16)) & 0xFFFFFFFF
        if (remain >= 9):
            c += (c+(key[i+8] << 8)) & 0xFFFFFFFF
        if (remain >= 8):
            b += (b+(key[i+7] << 24)) & 0xFFFFFFFF
        if (remain >= 7):
            b += (b+(key[i+6] << 16)) & 0xFFFFFFFF
        if (remain >= 6):
            b += (b+(key[i+5] << 8)) & 0xFFFFFFFF
        if (remain >= 5):
            b += (b+key[i+4]) & 0xFFFFFFFF
        if (remain >= 4):
            a += (a+(key[i+3] << 24)) & 0xFFFFFFFF
        if (remain >= 3):
            a += (a+(key[i+2] << 16)) & 0xFFFFFFFF
        if (remain >= 2):
            a += (a+(key[i+1] << 8)) & 0xFFFFFFFF
        if (remain >= 1):
            a += (a+key[i+0]) & 0xFFFFFFFF

        a, b, c = self._mix(a,b,c)

        self._last = c
        return c


class SimpleHash(object):
    """
    Very fast hash of poor quality
    """

    def __init__(self, seed):
        """
        Constructor, optinally with seed parameter.
        """
        self.seed = seed

    def hash(self, key):
        """
        Compute Python hash of the key.

        key: list of one byte integers
        """
        sum = 0
        for i in range(0, len(key)):
            sum = sum + i
        return (sum ^ self.seed) & 0xFFFFFFFF

class Hash(object):
    """
    Very slow Python hash of high quality
    """

    def __init__(self, seed):
        """
        Constructor, optinally with seed parameter.
        """
        self.seed = seed

    def hash(self, key):
        """
        Compute Python hash of the key.

        key: string
        """
        return (hash(key) ^ self.seed) & 0xFFFFFFFF

