###############################################################################
#  prefixset.py: Module for operations with sets of prefixes
#  Copyright (C) 2009 Brno University of Technology, ANT @ FIT
#  Author(s): Viktor Pus <ipus@fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""
Module providing operations with sets of prefixes.
"""

from prefix import *

class PrefixSet(object):
    """
    Class for operations with sets of prefixes
    """

    def __init__(self):
        """
        Constructor
        """
        self._prefixes = []
        # This list of prefixes must be kept sorted - longest first!
        # Identical prefix is not inserted again.

    
    def set_display_format(self, format):
        """
        Set format in which prefixes are displayed.
        Possible values are:
        'prefix', 'prefix6', 'range', 'value'
        """
        for prefix in self._prefixes:
            prefix.set_display_format(format)
    
    def display(self, format=""):
        """
        Print all prefixes in human-readable format.

        format: Format of printing. Possible values: 
        'ipv4_prefix', 'ipv6_prefix', 'range', 'value', ''
        ('' - use previously set format, if none is set, guess from domain_size)
        """
        for prefix in self._prefixes:
            prefix.display(format)


    def match(self, ip):
        """
        Return list of all matching prefixes.
        The list is sorted by length from the longest to shortest.
        Return empty list if ip matches no prefix.

        ip: value to compare with
        """
        list = []
        for prefix in self._prefixes:
            if prefix.match(ip):
                list.append(prefix)

        return list


    def match_longest(self, ip):
        """
        Return the longest matching prefix, if ip matches some prefix.
        Return None if ip matches no prefix.

        ip: value to compare with
        """
        for prefix in self._prefixes:
            if prefix.match(ip):
                return prefix

        return None


    def match_bool(self, ip):
        """
        Return True if ip matches any prefix, False if it matches no prefix.
        (Fastest if you don't care about the value of the prefix.)

        ip: value to compare with
        """
        for prefix in self._prefixes:
            if prefix.match(ip):
                return True

        return False


    def clear(self):
        """
        Clears the PrefixSet to contain no prefix.
        """
        self._prefixes = []


    def add_prefix(self, prefix):
        """
        Adds one prefix into the PrefixSet. Prefixes are kept sorted by length.
        If you add something other than Prefix, it's appended to the end.

        prefix: Prefix or MaskedInt to be added into the PrefixSet.
        """
        # List of prefixes is kept sorted - longest first!
        for index, p in enumerate(self._prefixes):
            if p == prefix: # Do not add duplicates
                return
            if (type(prefix) == type(p) == Prefix):
                if p.get_length() < prefix.get_length():
                    self._prefixes.insert(index, prefix)
                    return

        self._prefixes.append(prefix)
    
    
    def add_prefixes(self, prefixes):
        """
        Adds a list of prefixes into the PrefixSet.

        prefixes: List of prefixes to be added into the PrefixSet.
        """
        for p in prefixes:
            self.add_prefix(p)
    

    def set_prefixes(self, prefixes):
        """
        Set the PrefixSet to contain given prefixes. 
        Delete previous prefixes.

        prefixes: list of prefixes. Will be automatically sorted and
        checked for duplicates.
        """
        self.clear()
        for p in prefixes:
            self.add_prefix(p)


    def get_prefixes(self):
        """
        Return the list of all prefixes.
        """
        return self._prefixes

    
    def is_universal(self):
        """
        Return true if all prefixes are universal.
        """
        for p in self._prefixes:
            if not p.is_universal():
                return False
        return True
    
    
    def set_range(self, r_from, r_to, domain_size):
        """
        Set the PrefixSet to cover the given continous range by prefixes.
        All necessary prefixes will be generated and added.

        r_from: Lower bound of the range (including this value).
        r_to: Higher bound of the range (including this value).
        domain_size: Number of bits of the domain.
        """
        self.clear()

        res = []

        res.extend(self._recur_range2prefixes(r_from, r_to, 0, 
            2**domain_size - 1, domain_size))

        for p in res:
            self.add_prefix(p)


    def add_range(self, r_from, r_to, domain_size):
        """
        Add range to the PrefixSet. Range will be converted to the form
        of prefixes and these will be added to the existing ones in the set.

        r_from: Lower bound of the range (including this value).
        r_to: Higher bound of the range (including this value).
        domain_size: Number of bits of the domain.
        """
        res = []

        res.extend(self._recur_range2prefixes(r_from, r_to, 0, 
            2**domain_size - 1, domain_size))

        for p in res:
            self.add_prefix(p)
        

    def _recur_range2prefixes(self, fromval, toval, x1, x2, bits):
        """
        Recursively divide state space and append prefixes covered in
        the interval (fromval, toval). x1, x2 are current positions in
        the state space.
        """

        res = []

        if fromval <= x1 and toval >= x2:
            p = Prefix(domain_size = bits,
                       value = x1,
                       length = bits - log2(x2-x1)
                      )
            res.append(p)
            return res

        if x1 != x2 + 1 and ((fromval >= x1 and fromval <= x2) or \
                             (toval >= x1 and toval <= x2)):
            x3 = x1 + (x2 - x1) / 2
            x4 = x3 + 1
            res.extend(self._recur_range2prefixes(fromval, toval, x1, x3, bits))
            res.extend(self._recur_range2prefixes(fromval, toval, x4, x2, bits))

        return res


    def get_mean_length(self):
        """ Return mean length of prefixes """
        sum = 0
        for prefix in self._prefixes:
            sum += prefix.get_length()
        if len(self._prefixes):
            mean_length = float(sum) / len(self._prefixes)
        else:
            mean_length = 0
        return mean_length
        
    def get_histogram(self):
        """
        Return histogram of prefixes
        Histogram is an array in range <0, (max_domain_size + 1)>
        histogram[i] contains count of prefixes with length i
        """
        
        maxdomain = 0
        for prefix in self._prefixes:
            if prefix.get_domain_size() > maxdomain:
                maxdomain = prefix.get_domain_size()
        histogram = [0 for i in range(maxdomain + 1)]
        for prefix in self._prefixes:
            if ('get_length' in dir(prefix)):
                histogram[prefix.get_length()] += 1
        return histogram

    def get_nesting(self, p):
        """
        Compute number of prefixes covering p (not including p)
        """
        nest = 0
        for pr in self._prefixes:
            if (not(pr == p)):
                if (pr.covers(p)):
                    nest = nest + 1

        return nest
        
    def __eq__(self, other):
        """
        Prefixsets are equal if all prefixes are equal.
        """
        return self._prefixes == other._prefixes

    def __ne__(self, other):
        """
        Prefixsets are inequal if they are not equal.
        """
        return self._prefixes != other._prefixes

    def __len__(self):
        """
        Return number of prefixes in PrefixSet.
        """
        return len(self._prefixes)
    
    def __repr__(self):
        """
        Returns string "<PrefixSet>".
        """
        return "<PrefixSet>"
    
    def __str__(self):
        """
        Calls __str__() method to all Prefixes in PrefixSet and returns outputs
        as a list string.
        """
        return (map(Prefix.__str__,self._prefixes)).__str__()
    
    def __getitem__(self, i):
        """
        Returns prefix with index i. Useful to access the only prefix in
        PrefixSet without calling get_prefixes().
        """
        return self._prefixes[i]
    
    def covers(self, other):
        """
        Return True if prefixset fully covers another prefixset.

        other: Prefixset that we want to cover.
        """
        for his in other._prefixes:
            covered = False
            for my in self._prefixes:
                if my.covers(his):
                    covered = True
                    break
            if not covered:
                return False

        return True
