###############################################################################
#  prefix.py: Module for operations with prefixes
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vaclav Bartos <xbarto11@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id$

"""
Module providing basic operations with prefixes.
"""

import sys
import string
from netbench.classification.maskedint import *

class Prefix(MaskedInt):
    """Class for basic operations with prefixes of arbitrary length\
 (IP or other). It's subclass of MaskedInt."""
    
    def __init__(self,
                 domain_size,
                 value,
                 length=None,
                 mask=None,
                 display_format=""
                ):
        """
        Constructor, you must set domain_size, value and length or mask
        (if you set both, length has priority and mask is computed from it).
        
        domain_size: Number of bits of value and mask. 32 for IPv4, 16 for port, ...
        value:       Value of prefix.
        length:      Length of prefix.
        mask:        Mask. It must be valid mask of prefix (ie. 11110000).
        display_format: Format in which MaskedInt is displayed,
                        see set_display_format() for possible values.
        """
        
        if length is not None:
            self._length = length
            mask = ((1 << length) - 1) << (domain_size - length)
        elif mask is not None:
            if mask == 0:
                self._length = 0
            elif is_prefix_mask(mask, domain_size):
                self._length = domain_size - rightmost_one(mask)
            else:
                print "ERROR: Prefix constructor: Given mask is not a valid prefix mask."
        else:
            print "ERROR: Prefix constructor: At least mask or length must be set."
        
        MaskedInt.__init__(
            self,
            value = value,
            mask = mask,
            domain_size = domain_size,
            display_format = display_format
        )
    
    
    def set_length(self, length):
        """
        Set length of the prefix in bits. 
        Zero is valid value, meaning that the prefix matches any IP.
        """
        self._length = length
        self._mask = (1 << length) - 1

    def get_length(self):
        """
        Return length of prefix in bits.
        """
        return self._length

    def set_mask(self, mask):
        """
        Set mask of Prefix. Length is recomputed to correspond to the new mask.

        mask: int or long, must be a valid mask of prefix
        """
        self._mask = mask
        if mask == 0:
            self._length = 0
        elif is_prefix_mask(mask, self._domain_size):
            self._length = self._domain_size - rightmost_one(mask)
        else:
            print "ERROR: Prefix constructor: Given mask is not a valid prefix mask."

    def __repr__(self):
        """
        Returns string "<Prefix 0xValue Length/DomainSize>"
        """
        return "<Prefix 0x%x %i/%i>"%(self._value,self._length,self._domain_size)
    
    def get_range(self):
        """
        Return list of [low, high] bounds of the range defined by the prefix,
        bounds included. Untested for 128-bit domain.
        """
        
        if (self._length == 0): # Universal prefix
            return [0, (1 << self._domain_size) - 1]
        
        low = self._value & self._mask 
        high = self._value | ~self._mask & ((1 << self._domain_size) - 1)
        
        return [low, high]
    
