###############################################################################
#  simplenificparser.py: Module for parsing input rulesets in simplified NIFIC
#  format
#  Copyright (C) 2009 Brno University of Technology, ANT @ FIT
#  Author(s): Martin Skacan <xskaca00@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id: simplenificparser.py 410 2010-08-21 13:29:22Z ipus $

import sys
import string
import bruleparser
import netbench.classification.rule as rule
import netbench.classification.prefix as prefix
import netbench.classification.maskedint as maskedint
import netbench.classification.prefixset as prefixset


class SimpleNificParser(bruleparser.BRuleParser):
    """
    Rule parser according to the simplified NIFIC rule format.
    """

    def load_file(self, filename):
        """
        Load rules from a file, return True if parsing was successfull.

        filename: Name of file in NIFIC format.
        """

        # try to open a file
        try:
            file = open(filename, 'r')
        except IOError:
            return False
            
        lines = 0
        cont = False
        for line in file:
            lines += 1
            # the rule continues from a previous line
            if cont:
                # split line into an array
                tmp = line.split()
                # the line is empty
                if tmp == []:
                    continue
                linearray.extend(tmp)
                # the rule will be continued on the next line
                if (linearray[-1] == '\\'):
                    del linearray[-1]
                    cont = True
                    continue
                else:
                    cont = False
            # on the current line starts a new rule
            else:
                # split line into an array
                linearray = line.split()
                # the line is empty
                if linearray == []:
                    continue
                # the rule will be continued on the next line
                if (linearray[-1] == '\\'):
                    del linearray[-1]
                    cont = True
                    continue
                else:
                    cont = False
            
            # transform to low characters
            for i in range(len(linearray)):
                linearray[i] = (linearray[i]).lower()
            
            error = False
            priority = -1
            conditions = {}    
            
            # the lines with comments are skipped
            if linearray[0][0] == '#':
                continue
                
            ##############################################################
            # HERE SHOULD BE SOME REGEXP TO CHECK THE SYNTAX OF THE RULE #
            ##############################################################
         
            
            # process the priority of the rule
            result = process_priority(linearray)
            if result != -1:
                priority = result
            else:
                print "Error has occured in a rule on the line", lines
                self.ruleset.clear()
                return False
                
            # process the action of the rule
            result = process_action(linearray)
            if result != -1:
                conditions.update(result)
            else:
                print "Error has occured in a rule on the line", lines
                self.ruleset.clear()
                return False
            
            # process the rest fields of the rule and find keywords in it ("proto", "from", etc.) 
            for i in range(2, len(linearray)):
            
                # process ['on' interface]
                if linearray[i] == "on":
                    # word "on" must follow one valid interface number (0-15)               
                    if not (linearray[i+1]).isdigit():
                        error = True
                        break                  
                    interface = int(linearray[i+1])
                    if interface > 15:
                        error = True
                        break
                    conditions["on_interface"] = interface
                    
                # process protocol
                elif linearray[i] == "proto":
                    result = process_proto(linearray, i)
                    if result != -1:
                        conditions.update(result)
                    else:
                        error = True
                        break
                            
                # process the from-part of the rule
                elif linearray[i] == "from":
                    result = process_from_to(linearray, i, "src")
                    if result != -1:
                        conditions.update(result)
                    else:
                        error = True
                        break
                # process the to-part of the rule
                elif linearray[i] == "to":
                    result = process_from_to(linearray, i, "dst")
                    if result != -1:
                        conditions.update(result)
                    else:
                        error = True
                        break
                        
                # process the flags of the rule    
                elif linearray[i] == "flags":
                    # F 1, S 2, R 4, P 8, A 16, U 32, E 64, C 128
                    if len(linearray) <= i+1:
                        error = True
                        break
                    if not '/' in (linearray[i+1]):
                        error = True
                        break
                    # split flags into the prefix and the mask
                    flags = (linearray[i+1]).split('/')
                    # compute the value of the flags prefix
                    result = process_flags(flags[0])
                    conditions["flags_value"] = result
                    # compute the value of the flags mask
                    result = process_flags(flags[1])
                    conditions["flags_mask"] = result                     
                
                # process the flowid of the rule
                elif linearray[i] == "flowid":
                    if len(linearray) <= i+1:
                        error = True
                        break
                    if not (linearray[i+1]).isdigit():
                        error = True
                        break                 
                    flowid = int(linearray[i+1])
                    conditions["flowid"] = flowid
            
            # there was an error in previous part
            if error:
                print "Error has occured in a rule on the line", lines
                self.ruleset.clear()
                return False
                
            # everything seems to be OK, we can create a new rule and add it into the ruleset
            # in the ruleset already is a rule with this priority (it will be updated)
            if  self.ruleset.is_rule_with_given_priority(priority):
                existing_rule_index = self.ruleset.get_index_of_rule(priority)
                new_rule = self.ruleset.get_rule(existing_rule_index)
                rule_already_exists = True
            # create a new rule
            else:
                new_rule = rule.Rule()
                new_rule.set_priority(priority)
                rule_already_exists = False
            
            # in the rule was defined src MAC address
            if conditions.has_key("src_mac"):
                val = conditions["src_mac"]
                mac = maskedint.MaskedInt(value=(val), mask = 0xffffffffffff,
                                          domain_size = 48, display_format = 'mac')              
                
                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('srcmac')
                    prefix_set.add_prefix(mac)
                    new_rule.set_condition('srcmac', prefix_set)
                else:
                    # add src IP rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.add_prefix(mac)
                    new_rule.set_condition('srcmac', prefix_set)
            
            # in the rule was defined dst MAC address
            if conditions.has_key("dst_mac"):
                val = conditions["dst_mac"]
                mac = maskedint.MaskedInt(value=(val), mask = 0xffffffffffff,
                                          domain_size = 48, display_format = 'mac')
                
                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('dstmac')
                    prefix_set.add_prefix(mac)
                    new_rule.set_condition('dstmac', prefix_set)
                else:
                    # add src IP rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.add_prefix(mac)
                    new_rule.set_condition('dstmac', prefix_set)
                          
            # in the rule was defined src IP address
            if (conditions.has_key("srcip_prefix") and conditions.has_key("srcip_mask")):
                prefix_value = conditions["srcip_prefix"] & 0xffffffff
                prefix_mask = conditions["srcip_mask"]

                prefix_tmp = prefix.Prefix(value = prefix_value, length = prefix_mask,
                                 domain_size = 32, display_format = 'ipv4_prefix')

                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('srcipv4')
                    prefix_set.add_prefix(prefix_tmp)
                    new_rule.set_condition('srcipv4', prefix_set)
                else:
                    # add src IP rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.add_prefix(prefix_tmp)
                    new_rule.set_condition('srcipv4', prefix_set)

            # in the rule was defined dst IP address
            if (conditions.has_key("dstip_prefix") and conditions.has_key("dstip_mask")):
                prefix_value = conditions["dstip_prefix"] & 0xffffffff
                prefix_mask = conditions["dstip_mask"]

                prefix_tmp = prefix.Prefix(value = prefix_value, length = prefix_mask,
                                 domain_size = 32, display_format = 'ipv4_prefix')

                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('dstipv4')
                    prefix_set.add_prefix(prefix_tmp)
                    new_rule.set_condition('dstipv4', prefix_set)
                else:
                    # add src IP rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.add_prefix(prefix_tmp)
                    new_rule.set_condition('dstipv4', prefix_set)
                     
            # in the rule was defined src port
            if (conditions.has_key("src_port_low") and conditions.has_key("src_port_high")):
            
                # get ranges for the port
                low_range = conditions["src_port_low"]
                high_range = conditions["src_port_high"]
                
                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('src_port')
                    prefix_set.add_range(low_range, high_range, 16);
                    new_rule.set_condition('src_port', prefix_set)
                else:
                    # add src port rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.set_range(low_range, high_range, 16);
                    new_rule.set_condition('src_port', prefix_set)
            
            # in the rule was defined dst port
            if (conditions.has_key("dst_port_low") and conditions.has_key("dst_port_high")):
            
                # get ranges for the port
                low_range = conditions["dst_port_low"]
                high_range = conditions["dst_port_high"]
                
                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('dst_port')
                    prefix_set.add_range(low_range, high_range, 16);
                    new_rule.set_condition('dst_port', prefix_set)
                else:
                    # add src port rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.set_range(low_range, high_range, 16);
                    new_rule.set_condition('dst_port', prefix_set)
            
            # in the rule was defined protocol
            if conditions.has_key("protocol"):
                protocol = conditions["protocol"]
                
                proto_tmp = maskedint.MaskedInt(value = protocol, mask = 0xff,
                                                domain_size = 8, display_format = 'protocol')
                
                if (rule_already_exists):
                    # update the existing rule
                    prefix_set = new_rule.get_condition('protocol')
                    prefix_set.add_prefix(proto_tmp)
                    new_rule.set_condition('protocol', prefix_set)
                else:
                    # add src IP rule condition
                    prefix_set = prefixset.PrefixSet()
                    prefix_set.add_prefix(proto_tmp)
                    new_rule.set_condition('protocol', prefix_set)
                    
            # add rule to ruleset
            if (not rule_already_exists):
                # add newly created rule to the ruleset
                self.ruleset.add_rule(new_rule)
            else:
                # update the ruleset
                self.ruleset.del_rule(existing_rule_index)
                self.ruleset.add_rule(new_rule)
            
        return True

            
##############################################################
# common functions outside the class necessary to parse a file
##############################################################       

def process_priority(linearray):
    """
    Process the priority of the rule and return its value
    """
    
    if not (linearray[0]).isdigit():
        return -1               
    priority = int(linearray[0])
    if priority > 65535:
        return -1
    return priority


def process_action(linearray):
    """
    Process the action of the rule
    Return parsed conditions in a dictionary
    """
    
    conditions = {}
    if len(linearray) < 2:
        return -1
    if linearray[1] == "block":
        conditions["action"] = "block"
    elif linearray[1] == "pass":
        conditions["action"] = "pass"
        # if action is "pass", there must be one interface number
        if len(linearray) < 3:
            return -1 
        if not (linearray[2]).isdigit():
            return -1             
        interface = int(linearray[2])              
        if interface > 15:
            return -1
        conditions["pass_interface"] = interface
    # invalid action occured ("block" neither "pass")
    else:
        return -1
    return conditions


def process_proto(linearray, i):
    """
    Process the protocol of the rule
    Return parsed conditions in a dictionary
    """
    
    conditions = {}
    if len(linearray) <= i+1:
        return -1
    if linearray[i+1] == "tcp":
        conditions["protocol"] = 6
    elif linearray[i+1] == "udp":
        conditions["protocol"] = 17
    elif linearray[i+1] == "icmp":
        conditions["protocol"] = 1
    else:
        # procotol can be a number (0-255)
        if not (linearray[i+1]).isdigit():
            return -1
        tmp = int(linearray[i+1])
        if tmp > 255:
            return -1
        else:
            conditions["protocol"] = tmp
    return conditions


def process_from_to(linearray, i, s):
    """
    Process the 'from' (or 'to') part of the rule
    ['from' src_addr ['mac' src_mac]['port' src_port]]
    
    Return parsed conditions in a dictionary
    """
    
    conditions = {}
    if len(linearray) <= i+1:
        return -1
    # any src/dst IP
    if linearray[i+1] == "any":
        conditions[s + "ip_prefix"] = 0
        conditions[s + "ip_mask"] = 0
    # process the source/destination IP address
    else:
        # split the IP address into the prefix and the mask
        if '/' in (linearray[i+1]):
            ip = (linearray[i+1]).split('/')
            ipprefix = ip[0]
            ipmask = ip[1]
        # if there is no mask, set up 32
        else:
            ipprefix = linearray[i+1]
            ipmask = "32"
        
        # check if the IP prefix is valid
        tmpip = checkip(ipprefix)
        if tmpip != -1:
            conditions[s + "ip_prefix"] = tmpip
        else:
            return -1
        # check if the IP mask is valid
        if (not ipmask.isdigit() or int(ipmask) < 0 or int(ipmask) > 32):
            return -1
        else:
            conditions[s + "ip_mask"] = int(ipmask)
            
    # after IP address could follow MAC address, port or both
    # store the index pointing at IP address
    tmpindex = i+1
    # there follow the mac address
    if (len(linearray) > (tmpindex + 1) and linearray[tmpindex + 1] == "mac"):
        if len(linearray) <= tmpindex+2:
            return -1
        # check if the given MAC address is valid
        mac = checkmac(linearray[tmpindex + 2])
        if mac != -1:
            conditions[s + "_mac"] = mac
        else:
            return -1
        # if there was a MAC address shift the stored index to the right (pointing at MAC address)
        tmpindex += 2
        
    # there follow a port (after ip address or after mac address)
    if (len(linearray) > (tmpindex + 1) and linearray[tmpindex + 1] == "port"):
        if len(linearray) <= tmpindex+2:
            return -1
        port = linearray[tmpindex + 2]
        # the ports are given as a range (e.g. 1024:65535)
        if ':' in port:
            # split it into two parts and check it
            port = port.split(':')
            if len(port) != 2:
                return -1
            if (not port[0].isdigit() or int(port[0]) < 0 or int(port[0]) > 65535):
                return -1
            else:
                conditions[s + "_port_low"] = int(port[0])
            if (not port[1].isdigit() or int(port[1]) < 0 or int(port[1]) > 65535):
                return -1
            else:
                conditions[s + "_port_high"] = int(port[1])
        # there is a single port
        else:
            # check it
            if (not port.isdigit() or int(port) < 0 or int(port) > 65535):
                return -1
            # the low part and the high part are the same
            else:
                conditions[s + "_port_low"] = int(port)
                conditions[s + "_port_high"] = int(port)
                
    return conditions
               
                                
def process_flags(flags):
    """
    Compute the value of the given flags
    """
    
    value = 0
    if 'f' in flags:
        value += 1
    if 's' in flags:
        value += 2
    if 'r' in flags:
        value += 4
    if 'p' in flags:
        value += 8
    if 'a' in flags:
        value += 16
    if 'u' in flags:
        value += 32
    if 'e' in flags:
        value += 64
    if 'c' in flags:
        value += 128
        
    return value
                                   

def checkip(ip):
    """
    Check if the given string contains a valid IPv4 prefix
    Return the value of the address
    """
    
    ip = ip.split('.')
    if  len(ip) != 4:
        return -1
    value = 0
    flag = False
    for i in range(0,4):
        if (not ip[i].isdigit() or int(ip[i]) < 0 or int(ip[i]) > 255):
            flag = True
            break
        value += int(ip[i])
        value = value << 8
    if (flag):
        return -1
    value = value >> 8
    return value


def checkmac(mac):
    """
    Check if the given string contains a valid MAC address
    Return the value of the address
    """
    
    mac = mac.split(':')
    if len(mac) != 6:
        return -1
    value = 0
    flag = False
    for i in range(0,6):
        if (not ishex(mac[i]) or int(mac[i], 16) < 0 or int(mac[i], 16) > 255):
            flag = True
            break
        value += int(mac[i], 16)
        value = value << 8
    if (flag):
        return -1
    value = value >> 8
    return value


def ishex(s):
    """
    Check if given string is a valid hexadecimal number
    """
    
    if s == "":
        return False
    for c in s:
        if not c in string.hexdigits:
            return False
    return True

