###############################################################################
#  maskedint.py: Module for operations with values with mask
#  Copyright (C) 2010 Brno University of Technology, ANT @ FIT
#  Author(s): Vaclav Bartos <xbarto11@stud.fit.vutbr.cz>
###############################################################################
#
#  LICENSE TERMS
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#  3. All advertising materials mentioning features or use of this software
#     or firmware must display the following acknowledgement:
#
#       This product includes software developed by the University of
#       Technology, Faculty of Information Technology, Brno and its
#       contributors.
#
#  4. Neither the name of the Company nor the names of its contributors
#     may be used to endorse or promote products derived from this
#     software without specific prior written permission.
#
#  This software or firmware is provided ``as is'', and any express or implied
#  warranties, including, but not limited to, the implied warranties of
#  merchantability and fitness for a particular purpose are disclaimed.
#  In no event shall the company or contributors be liable for any
#  direct, indirect, incidental, special, exemplary, or consequential
#  damages (including, but not limited to, procurement of substitute
#  goods or services; loss of use, data, or profits; or business
#  interruption) however caused and on any theory of liability, whether
#  in contract, strict liability, or tort (including negligence or
#  otherwise) arising in any way out of the use of this software, even
#  if advised of the possibility of such damage.
#
#  $Id: maskedint.py 485 2010-09-21 13:54:44Z xbarto11 $

"""
Module providing basic operations with values with mask.
"""

import sys
import string

class MaskedInt(object):
    """Class for basic operations with values of arbitrary length\
 with a bitmask (usually prefixes)."""

    def __init__(self,
                 value,
                 mask=None,        # if not given, it's set to all ones (domain size must be set)
                 domain_size=None, # if not given, it's determined from mask
                 display_format=""
                ):
        """
        Constructor, you can set:
        value:       The value.
        mask:        The mask - 0 bit on some position means that this bit of 
                     value is don't care.
        domain_size: Number of bits of value and mask. 32 for IPv4, 16 for port, ...
        display_format: Format in which MaskedInt is displayed,
                        see set_display_format() for possible values.
        """

        self._value = value
        
        # mask and domain_size
        if mask is not None:
            if domain_size is not None:
                self._mask = mask & (1 << domain_size) - 1 # truncate mask to domain_size width
                self._domain_size = domain_size
            else:
                self._mask = mask
                self._domain_size = leftmost_one(mask) + 1 # set domain size to width if mask
        elif domain_size is not None:
            self._mask = (1 << domain_size) - 1 # set to all ones
            self._domain_size = domain_size
        else:
            print "ERROR: MaskedInt constructor: At least mask or domain_size must be set."
            
        
        # Format in which MaskedInt is displayed
        self._display_format = display_format
        
        # Underscore is used to denote private class members.


    def set_data(self, value, mask):
        """
        Set value and mask of MaskedInt.

        value: int or long
        mask:  int or long
        """
        self._value = value
        self._mask = mask

    def get_data(self):
        """
        Return data as tuple (value,mask).
        """
        return (self._value, self._mask)
    
    def set_value(self, value):
        """
        Set value of MaskedInt.

        value: int or long
        """
        self._value = value

    def get_value(self):
        """
        Return value of MaskedInt.
        """
        return self._value
    
    def set_mask(self, mask):
        """
        Set mask of MaskedInt.

        mask: int or long
        """
        self._mask = mask

    def get_mask(self):
        """
        Return mask of MaskedInt.
        """
        return self._mask
    
    def set_domain_size(self, size):
        """
        Set number of bits of the domain.
        """
        self._domain_size = size
        self._mask = self._mask & (1 << size) - 1

    def get_domain_size(self):
        """
        Get number of bits of the domain.
        """
        return self._domain_size
    
    def is_universal(self):
        """
        Return true if MaskedInt is universal (matches every value).
        """
        return (self._mask == 0)
    
    def set_display_format(self, format):
        """
        Set format in which prefix is displayed.
        Possible values are:
        'ipv4_prefix', 'ipv6_prefix', 'range', 'value'
        """
        self._display_format = format
    
    def get_display_format(self):
        """
        Get format in which prefix is displayed.
        """
        return self._display_format
    
    def display(self, format = ""):
        """
        Print prefix in human-readable format, without newline

        format: Format of printing. Possible values: 
        'ipv4_prefix', 'ipv6_prefix', 'range', 'value', ''
        ('' - use previously set format, if none is set, guess from domain_size)
        """
        print self.__str__(format),
    
    def __str__(self, format = ""):
        """
        Return string in prefix format.
        Allows to display prefix by "print" statement.
        """
        # TODO: mac for MAC address
        
        m = self._mask
        v = self._value
        ds = self._domain_size
        
        # Set display_format if it hasn't been set yet
        if self._display_format == "":
            if format:
                self._display_format = format
            # If it's not given, guess it from domain_size
            elif ds == 32:
                self._display_format = "ipv4_prefix"
            elif ds == 128:
                self._display_format = "ipv6_prefix"
            elif ds == 16:
                self._display_format = "port"
            elif self._mask == (1 << ds) - 1:
                self._display_format = "value"
        
        if format != "":
            f = format
        else:
            f = self._display_format
        
        
        
        if (self._mask == 0):
            return "any"
        if ((f == 'ipv4_prefix' or (f == 'prefix' and ds == 32))
            and ('get_length' in dir(self))):
            return "%d.%d.%d.%d/%d"%((v>>24)&0xff, (v>>16)&0xff, 
                                     (v>>8)&0xff, v&0xff, self.get_length())
        #elif (f == 'ipv6_prefix' or (f == 'prefix' and ds == 128)):
        #    TODO
        elif (f == 'range' or f == 'port') and ('get_range' in dir(self)):
            r = self.get_range();
            if (r[0] == r[1]):
                return str(r[0])
            else:
                return str(r[0]) + ":" + str(r[1])
        elif (f == 'value' or f == 'protocol'):
            return str(v)
        elif (f == 'hexvalue'):
            return hex(v)
        else:
            return hex(v) + "/" + hex(m)
    
    def __repr__(self):
        """
        Returns string "<MaskedInt 0xValue 0xMask>"
        """
        return "<MaskedInt 0x%x 0x%x>"%(self._value,self._mask)
    
    #def __hash__(self):
    #    """
    #    Conut hash from maskedInt attributes.
    #    Involves only valid bits of value and a mask. 
    #    """
    #    return hash((self._value & self._mask, self._mask))
        
    
    def match(self, value):
        """
        Return True if valid bits of given value matches the value of MaskedInt.

        value: list of numbers, one number for each 32bits,
        value[0] is MSB.
        """
        return (self._value & self._mask == value & self._mask)
            

    def __eq__(self, other):
        """
        Compare content of two MaskedInts, return True if 
        masks and valid bits of values are same.
        """
        return (self._mask == other._mask and 
                self._value & self._mask == other._value & other._mask)


    def __ne__(self, other):
        """
        Return True if MaskedInts differ.
        """
        return not self == other


    def covers(self, other):
        """
        Return True if the MaskedInt fully covers another MaskedInt, given as 
        parameter (ie. if every possible value of other is also possible value
        of this).

        other: MaskedInt that we want to compare.
        """
        if (self._mask & ~other._mask != 0):
            return False
        
        return (self._value & self._mask == other._value & self._mask)
        


###################################
# Additional package methods (not in MaskedInt class)
###################################

def log2(num):
    """
    Return base-two logarithm of the number (rounded up).
    """
    i = 0
    n = 1
    while n < num:
        n *= 2
        i += 1

    return i

def leftmost_one(x):
    i = -1
    while (x != 0):
        i += 1
        x >>= 1
    return i

def rightmost_one(x):
    if x == 0:
        return None
    i = 0
    while (x & 1 == 0):
        i += 1
        x >>= 1
    return i

def is_prefix_mask(x, width):
    if x == 0:
        return True
    
    i = 0
    while (x & 1 == 0):
        x >>= 1
        i += 1
    
    while (x & 1 != 0):
        x >>= 1
        i += 1
    
    if (i < width):
        return False
    else:
        return True
    
    
