--##############################################################################
--  cp_stage.vhd: Color Processing Stage VHDL implementation
--  Copyright (C) 2011 Brno University of Technology, ANT @ FIT
--  Author(s): Viktor Pus <ipus@fit.vutbr.cz>
--##############################################################################
--
--  LICENSE TERMS
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--  1. Redistributions of source code must retain the above copyright
--     notice, this list of conditions and the following disclaimer.
--  2. Redistributions in binary form must reproduce the above copyright
--     notice, this list of conditions and the following disclaimer in
--     the documentation and/or other materials provided with the
--     distribution.
--  3. All advertising materials mentioning features or use of this software
--     or firmware must display the following acknowledgement:
--
--       This product includes software developed by the University of
--       Technology, Faculty of Information Technology, Brno and its
--       contributors.
--
--  4. Neither the name of the Company nor the names of its contributors
--     may be used to endorse or promote products derived from this
--     software without specific prior written permission.
--
--  This software or firmware is provided ``as is'', and any express or implied
--  warranties, including, but not limited to, the implied warranties of
--  merchantability and fitness for a particular purpose are disclaimed.
--  In no event shall the company or contributors be liable for any
--  direct, indirect, incidental, special, exemplary, or consequential
--  damages (including, but not limited to, procurement of substitute
--  goods or services; loss of use, data, or profits; or business
--  interruption) however caused and on any theory of liability, whether
--  in contract, strict liability, or tort (including negligence or
--  otherwise) arising in any way out of the use of this software, even
--  if advised of the possibility of such damage.
--
--  $Id: cp_stage.vhd 586 2011-01-04 11:00:10Z ipus $

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

-- library containing log2 function
use work.math_pack.all;

entity cp_stage is
   generic(
      DIM      : integer := 5;
      COL      : integer := 8;
      PREF_W   : integer := 8;
      LEN_W    : integer := 6
   );
   port(
      CLK      : in std_logic;
      RESET    : in std_logic;

      -- Prefix is valid (for each dimension and color)
      VLD      : in std_logic_vector(DIM*COL-1 downto 0);

      -- Length of prefix (for each dimension and color)
      LEN      : in std_logic_vector(DIM*COL*LEN_W-1 downto 0);

      -- Prefix number (for each dimension and color)
      PREFIX   : in std_logic_vector(DIM*COL*PREF_W-1 downto 0);

      -- Aggregate Allowed Colors Bitmap
      -- (in each dimension for each dimension and all colors)
      -- (own dimension also present, must be all ones!)
      AACB     : in std_logic_vector(DIM*DIM*COL-1 downto 0);

      -- Output prefix for each dimension
      OUT_PREFIX:out std_logic_vector(DIM*PREF_W-1 downto 0);

      -- Output valid for each dimension
      OUT_VLD  : out std_logic_vector(DIM-1 downto 0)
   );
end entity cp_stage;

architecture full of cp_stage is

   signal reg_vld       : std_logic_vector(DIM*COL-1 downto 0);
   signal reg_len       : std_logic_vector(DIM*COL*LEN_W-1 downto 0);
   signal reg_prefix    : std_logic_vector(DIM*COL*PREF_W-1 downto 0);
   signal reg_aacb      : std_logic_vector(DIM*DIM*COL-1 downto 0);

   signal reg2_len      : std_logic_vector(DIM*COL*LEN_W-1 downto 0);
   signal reg2_prefix   : std_logic_vector(DIM*COL*PREF_W-1 downto 0);

   signal sig_out_vld   : std_logic_vector(DIM-1 downto 0);
   signal reg_out_vld   : std_logic_vector(DIM-1 downto 0);
   signal reg3_prefix   : std_logic_vector(DIM*COL*PREF_W-1 downto 0);

   signal sig_out_prefix:std_logic_vector(DIM*PREF_W-1 downto 0);
   signal reg_out_prefix:std_logic_vector(DIM*PREF_W-1 downto 0);
   signal reg2_out_vld  : std_logic_vector(DIM-1 downto 0);

   -- Simplified AACB (Columns ANDed)
   signal saacb         : std_logic_vector(DIM*COL-1 downto 0);

   -- Final Allowed Colors Bitmap (in each dimension)
   signal facb          : std_logic_vector(DIM*COL-1 downto 0);
   signal reg_facb      : std_logic_vector(DIM*COL-1 downto 0);

   -- Selected prefix in each dimension
   signal sel_pref      : std_logic_vector(DIM*log2(COL)-1 downto 0);
   signal reg_sel_pref  : std_logic_vector(DIM*log2(COL)-1 downto 0);
   

begin

   reg_inputs : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         -- Input registers (first stage)
         reg_vld        <= VLD;
         reg_len        <= LEN;
         reg_prefix     <= PREFIX;
         reg_aacb       <= AACB;
         -- Second stage
         reg_facb       <= facb;
         reg2_len       <= reg_len;
         reg2_prefix    <= reg_prefix;
         -- Third stage
         reg_out_vld    <= sig_out_vld;
         reg3_prefix    <= reg2_prefix;
         reg_sel_pref   <= sel_pref;
         -- Output stage
         reg_out_prefix <= sig_out_prefix;
         reg2_out_vld   <= reg_out_vld;
      end if;
   end process;

   OUT_VLD <= reg2_out_vld;
   OUT_PREFIX <= reg_out_prefix;

   saacb_p : process(reg_aacb)
      variable v_saacb : std_logic_vector(DIM*COL-1 downto 0);
   begin
      v_saacb := (others => '1');

      for i in 0 to DIM-1 loop
         for j in 0 to DIM-1 loop
            v_saacb(COL*(i+1)-1 downto COL*i) := 
               v_saacb(COL*(i+1)-1 downto COL*i) 
               and reg_aacb(DIM*COL*j + COL*(i+1)-1 downto DIM*COL*j + COL*i);
         end loop;
      end loop;

      saacb <= v_saacb;
   end process;
   
   facb_p : process(reg_vld, saacb)
   begin
      for i in 0 to DIM-1 loop
         facb(COL*(i+1)-1 downto COL*i) <= 
            reg_vld(COL*(i+1)-1 downto COL*i) 
            and saacb(COL*(i+1)-1 downto COL*i);
      end loop;
   end process;

   sig_out_vld_p : process(reg_facb)
      variable v_sig_out_vld : std_logic_vector(DIM-1 downto 0);
   begin
      v_sig_out_vld := (others => '0');

      for i in 0 to DIM-1 loop
         for j in 0 to COL-1 loop
            v_sig_out_vld(i) := v_sig_out_vld(i)
               or reg_facb(COL*i + j);
         end loop;
      end loop;

      sig_out_vld <= v_sig_out_vld;
   end process;

   sel_pref_p : process(reg_facb, reg2_len)
      variable longest   : integer;
      variable longest_i : integer;
   begin
      sel_pref <= (others => '0');
      for i in 0 to DIM-1 loop -- Dimension
         longest_i := 0;
         longest   := 0;
         for j in 0 to COL-1 loop -- Color
            if reg_facb(i*COL + j) = '1' then
               if conv_integer(reg2_len(COL*LEN_W*i + LEN_W*(j+1)-1 
                                        downto COL*LEN_W*i + LEN_W*j)) >
                  longest then
                  longest_i := j;
                  longest   := conv_integer(reg2_len(COL*LEN_W*i + LEN_W*(j+1)-1 downto COL*LEN_W*i + LEN_W*j));
               end if;
            end if;
         end loop;
         sel_pref(log2(COL)*(i+1)-1 downto log2(COL)*i) <= 
            conv_std_logic_vector(longest_i, log2(COL));
      end loop;
   end process;

   sig_out_prefix_p : process(reg_sel_pref, reg3_prefix)
   begin
      sig_out_prefix <= (others => '0');
      for i in 0 to DIM-1 loop -- For each dimension
         for j in 0 to COL-1 loop -- Multiplexer
            if conv_std_logic_vector(j, log2(COL)) = 
               reg_sel_pref((log2(COL)*(i+1)-1) downto log2(COL)*i) then
               sig_out_prefix(PREF_W*(i+1)-1 downto PREF_W*i) <=
                  reg3_prefix(COL*PREF_W*i + PREF_W*(j+1)-1 downto COL*PREF_W*i + PREF_W*j);
            end if;
         end loop;
      end loop;
   end process;

end architecture full;
