
#include "main.h"

bool compute_1D_FFT(bool a_forward,unsigned a_size_power,unsigned char *a_data,unsigned a_data_step)
{/*{{{*/
   if (a_size_power < 2) {
      return false;
   }

   unsigned n,i,j,k,n_d2,l,l1,l2;
   float c1,c2,t1,t2,u1,u2,z;

   // - calculate the number of points -
   n = 1 << a_size_power;

   // - calculate a_data_end pointer -
   unsigned char *a_data_end = a_data + n*a_data_step;

   // - do the bit reversal -
   n_d2 = n >> 1;
   j = 0;
   for (i=0;i<n-1;i++) {
      if (i < j) {
         two_floats_s &i_td = *((two_floats_s *)(a_data + i*a_data_step));
         two_floats_s &j_td = *((two_floats_s *)(a_data + j*a_data_step));

         float tmp = i_td.real;
         i_td.real = j_td.real;
         j_td.real = tmp;

         tmp = i_td.imag;
         i_td.imag = j_td.imag;
         j_td.imag = tmp;
      }

      k = n_d2;
      while (k <= j) {
         j -= k;
         k >>= 1;
      }
      j += k;
   }

   // - compute the FFT -
   c1 = -1.0; 
   c2 = 0.0;
   l2 = 1;
   for (l=0;l<a_size_power;l++) {
      l1 = l2;
      l2 <<= 1;
      u1 = 1.0; 
      u2 = 0.0;
      for (j=0;j<l1;j++) {
         if (j<n) {
            unsigned char *i_ptr = a_data + j*a_data_step;
            do {
               two_floats_s &i_td = *((two_floats_s *)i_ptr);
               two_floats_s &i1_td = *((two_floats_s *)(i_ptr + l1*a_data_step));

               t1 = u1 * i1_td.real - u2 * i1_td.imag;
               t2 = u1 * i1_td.imag + u2 * i1_td.real;
               i1_td.real = i_td.real - t1; 
               i1_td.imag = i_td.imag - t2;
               i_td.real += t1;
               i_td.imag += t2;

            } while((i_ptr += l2*a_data_step) < a_data_end);
         }

         z =  u1 * c1 - u2 * c2;
         u2 = u1 * c2 + u2 * c1;
         u1 = z;
      }
      c2 = sqrt((1.0 - c1) / 2.0);

      if (a_forward) {
         c2 = -c2;
      }

      c1 = sqrt((1.0 + c1) / 2.0);
   }

   // - scaling for forward transform -
   if (a_forward) {
      unsigned char *i_ptr = a_data;
      do {
         two_floats_s &i_td = *((two_floats_s *)i_ptr);

         i_td.real /= n;
         i_td.imag /= n;
      } while((i_ptr += a_data_step) < a_data_end);
   }
   
   return true;
}/*}}}*/

bool convert_32S_to_8U(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_trg->nChannels != 1 ||
      (unsigned)a_src->depth != IPL_DEPTH_32S || a_trg->depth != IPL_DEPTH_8U) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;
   unsigned s_image_ls = s_width*s_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + (t_height - 1)*a_trg->widthStep + t_width*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

   do {
      unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
      do {
         //*t_ptr = *((unsigned *)s_ptr) >> c_icsv;
         *t_ptr = (unsigned)(*((int *)s_ptr) - INT_MIN) >> c_icsv;
      } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

      t_ptr += a_trg->widthStep - t_image_ls;
      s_ptr += a_src->widthStep - s_image_ls;
   } while(t_ptr < t_ptr_end);

   return true;
}/*}}}*/

bool convert_8U_to_2x32F(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_trg->nChannels != 2 ||
      a_src->depth != IPL_DEPTH_8U || a_trg->depth != IPL_DEPTH_32F) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;
   unsigned s_image_ls = s_width*s_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + (t_height - 1)*a_trg->widthStep + t_width*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

   do {
      unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
      do {
         register float *f_ptr = (float *)t_ptr;
         f_ptr[0] = *s_ptr/255.0f;
         f_ptr[1] = 0.0f;
      } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

      t_ptr += a_trg->widthStep - t_image_ls;
      s_ptr += a_src->widthStep - s_image_ls;
   } while(t_ptr < t_ptr_end);

   return true;
}/*}}}*/

bool convert_2x32F_to_8U(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 2 || a_trg->nChannels != 1 ||
      a_src->depth != IPL_DEPTH_32F || a_trg->depth != IPL_DEPTH_8U) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;
   unsigned s_image_ls = s_width*s_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + (t_height - 1)*a_trg->widthStep + t_width*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

   do {
      unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
      do {
         *t_ptr = (unsigned char)(*((float *)s_ptr)*255.0f);
      } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

      t_ptr += a_trg->widthStep - t_image_ls;
      s_ptr += a_src->widthStep - s_image_ls;
   } while(t_ptr < t_ptr_end);

   return true;
}/*}}}*/

bool convert_2x32F_to_32F(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 2 || a_trg->nChannels != 1 ||
      a_src->depth != IPL_DEPTH_32F || a_trg->depth != IPL_DEPTH_32F) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;
   unsigned s_image_ls = s_width*s_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + (t_height - 1)*a_trg->widthStep + t_width*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

   do {
      unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
      do {
         *((float *)t_ptr) = *((float *)s_ptr);
      } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

      t_ptr += a_trg->widthStep - t_image_ls;
      s_ptr += a_src->widthStep - s_image_ls;
   } while(t_ptr < t_ptr_end);

   return true;
}/*}}}*/

bool invert(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_trg == NULL || a_src == NULL || a_trg->nChannels != a_src->nChannels || a_trg->depth != a_src->depth) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned image_ls = s_width*pixel_step;

   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*pixel_step;
   unsigned char *s_ptr_end = s_ptr + (s_height - 1)*a_src->widthStep + image_ls;
   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*pixel_step;

#define INVERT_IPLIMAGE(OPERATION) \
{\
   do {\
      unsigned char *s_ptr_w_end = s_ptr + image_ls;\
      do {\
         OPERATION;\
      } while(t_ptr += pixel_step,(s_ptr += pixel_step) < s_ptr_w_end);\
\
      t_ptr += a_trg->widthStep - image_ls;\
      s_ptr += a_src->widthStep - image_ls;\
   } while(s_ptr < s_ptr_end);\
}

   switch (a_src->depth) {
   case IPL_DEPTH_8U:
      switch (a_src->nChannels) {
      case 1:
         INVERT_IPLIMAGE(*t_ptr = 255 - *s_ptr);
         break;
      case 3:
         INVERT_IPLIMAGE(
            t_ptr[0] = 255 - s_ptr[0];
            t_ptr[1] = 255 - s_ptr[1];
            t_ptr[2] = 255 - s_ptr[2];
         );
         break;
      default:
         return false;
      }
      break;
   default:
      return false;
   }

   return true;
}/*}}}*/

bool normalize(int a_border,IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   int border_m2 = a_border << 1;

   if (a_trg == NULL || a_src == NULL || a_trg->nChannels != a_src->nChannels || a_trg->depth != a_src->depth) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height || border_m2 >= s_width || border_m2 >= s_height) {
      return false;
   }

   unsigned pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned image_ls = (s_width - border_m2)*pixel_step;

#define NORMALIZE_IPLIMAGE_GET_VALUES(OPERATION) \
{\
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + (s_y_offset + a_border)*a_src->widthStep + (s_x_offset + a_border)*pixel_step;\
   unsigned char *s_ptr_end = s_ptr + ((s_height - border_m2) - 1)*a_src->widthStep + image_ls;\
\
   do {\
      unsigned char *s_ptr_w_end = s_ptr + image_ls;\
      do {\
	 OPERATION;\
      } while((s_ptr += pixel_step) < s_ptr_w_end);\
\
      s_ptr += a_src->widthStep - image_ls;\
   } while(s_ptr < s_ptr_end);\
}

#define NORMALIZE_IPLIMAGE(OPERATION) \
{\
   unsigned char *ptr = (unsigned char *)a_trg->imageData + (t_y_offset + a_border)*a_trg->widthStep + (t_x_offset + a_border)*pixel_step;\
   unsigned char *ptr_end = ptr + ((t_height - border_m2) - 1)*a_trg->widthStep + image_ls;\
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + (s_y_offset + a_border)*a_src->widthStep + (s_x_offset + a_border)*pixel_step;\
\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
	 OPERATION;\
      } while(s_ptr += pixel_step,(ptr += pixel_step) < ptr_w_end);\
\
      ptr += a_trg->widthStep - image_ls;\
      s_ptr += a_src->widthStep - image_ls;\
   } while(ptr < ptr_end);\
}

   switch (a_src->depth) {
   case IPL_DEPTH_8U:
      switch (a_src->nChannels) {
      case 1:
         {
            unsigned char min_value = UCHAR_MAX;
            unsigned char max_value = 0;

            NORMALIZE_IPLIMAGE_GET_VALUES(
               min_value = min_value > *s_ptr?*s_ptr:min_value;
               max_value = max_value < *s_ptr?*s_ptr:max_value;
            );

            if (min_value != max_value) {
               unsigned char value_diff = max_value - min_value;

               NORMALIZE_IPLIMAGE(
                  unsigned result = ((((unsigned)*s_ptr) - min_value)*UCHAR_MAX)/value_diff;
                  *ptr = (unsigned char)result;
               );
            }
         }
         break;
      default:
         return false;
      }
      break;
   case IPL_DEPTH_32S:
      switch (a_src->nChannels) {
      case 1:
         {
            unsigned min_value = UINT_MAX;
            unsigned max_value = 0;

            NORMALIZE_IPLIMAGE_GET_VALUES(
               register unsigned value = (((unsigned)*((int *)s_ptr)) - INT_MIN);
               min_value = min_value > value?value:min_value;
               max_value = max_value < value?value:max_value;
            );

            if (min_value != max_value) {
               unsigned long long value_diff = max_value - min_value;

               NORMALIZE_IPLIMAGE(
                  register unsigned long long s_value = ((unsigned long long)((((unsigned)*((int *)s_ptr)) - INT_MIN) - min_value));
                  register unsigned long long ull_uint_max = (unsigned long long)UINT_MAX;

                  unsigned long long result = (s_value*ull_uint_max)/value_diff;
                  *((int *)ptr) = ((unsigned)result) + INT_MIN;
               );
            }
         }
         break;
      default:
         return false;
      }
      break;
   case IPL_DEPTH_32F:
      switch (a_src->nChannels) {
      case 1:
         {
            float min_value = INFINITY;
            float max_value = -INFINITY;

            NORMALIZE_IPLIMAGE_GET_VALUES(
               min_value = min_value > *((float *)s_ptr)?*((float *)s_ptr):min_value;
               max_value = max_value < *((float *)s_ptr)?*((float *)s_ptr):max_value;
            );

            if (min_value != max_value) {
               float value_diff = max_value - min_value;
               
               NORMALIZE_IPLIMAGE(
                  *((float *)ptr) = (*((float *)s_ptr) - min_value)/value_diff;
               );
            }
         }
         break;
      default:
         return false;
      }
      break;
   case IPL_DEPTH_64F:
      switch (a_src->nChannels) {
      case 1:
         {
            double min_value = INFINITY;
            double max_value = -INFINITY;

            NORMALIZE_IPLIMAGE_GET_VALUES(
               min_value = min_value > *((double *)s_ptr)?*((double *)s_ptr):min_value;
               max_value = max_value < *((double *)s_ptr)?*((double *)s_ptr):max_value;
            );

            if (min_value != max_value) {
               double value_diff = max_value - min_value;
               
               NORMALIZE_IPLIMAGE(
                  *((double *)ptr) = (*((double *)s_ptr) - min_value)/value_diff;
               );
            }
         }
         break;
      default:
         return false;
      }
      break;
   default:
      return false;
   }

   return true;
}/*}}}*/

bool compute_integral_image(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_trg == NULL || a_src == NULL || a_trg->nChannels != 1 || a_src->nChannels != 1 ||
      (unsigned)a_trg->depth != IPL_DEPTH_32S || a_src->depth != IPL_DEPTH_8U) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;
   unsigned s_image_ls = s_width*s_pixel_step;

   // - compute first line -
   {
      unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
      unsigned char *t_ptr_end = t_ptr + t_image_ls;
      unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

      *((unsigned *)t_ptr) = *s_ptr;

      do {
         *((unsigned *)(t_ptr + t_pixel_step)) = *((unsigned *)t_ptr) + *(s_ptr + s_pixel_step);
      } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_end);
   }

   // - compute first column -
   {
      unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
      unsigned char *t_ptr_end = t_ptr + (t_height - 1)*a_trg->widthStep;
      unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

      do {
         *((unsigned *)(t_ptr + a_trg->widthStep)) = *((unsigned *)t_ptr) + *(s_ptr + a_src->widthStep);
      } while(s_ptr += a_src->widthStep,(t_ptr += a_trg->widthStep) < t_ptr_end);
   }

   // - compute whole image -
   {
      unsigned char *t_ptr = (unsigned char *)a_trg->imageData + (t_y_offset + 1)*a_trg->widthStep + (t_x_offset + 1)*t_pixel_step;
      unsigned char *t_ptr_end = t_ptr + (t_height - 2)*a_trg->widthStep + t_image_ls;
      unsigned char *s_ptr = (unsigned char *)a_src->imageData + (s_y_offset + 1)*a_src->widthStep + (s_x_offset + 1)*s_pixel_step;
      do {
         unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
         unsigned line_sum = 0;

         do {
            line_sum += *s_ptr;
            *((unsigned *)t_ptr) = *((unsigned *)(t_ptr - a_trg->widthStep)) + line_sum;
         } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);

         t_ptr += a_trg->widthStep - t_image_ls;
         s_ptr += a_src->widthStep - s_image_ls;
      } while(t_ptr < t_ptr_end);
   }

   return true;
}/*}}}*/

bool generate_gaabor_function(double a_lambda,double a_phi,double a_psi,double a_sigma,double a_gamma,IplImage *a_img)
{/*{{{*/

   if (a_img == NULL || a_img->nChannels != 1 || a_img->depth != IPL_DEPTH_64F) {
      return false;
   }

   int x_offset,y_offset,width,height;

   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*a_img->widthStep + image_ls;

   double sin_phi = sin(a_phi);
   double cos_phi = cos(a_phi);
   double gamma_power = a_gamma*a_gamma;
   double sigma_power = a_sigma*a_sigma;

   double y = -((double)height/2.0);

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      double x = -((double)width/2.0);
      do {
         double _x = x*cos_phi + y*sin_phi;
         double _y = -x*sin_phi + y*cos_phi;

         double *d_ptr = (double *)ptr;
         d_ptr[0] = exp(-(_x*_x + gamma_power*_y*_y)/(2*(sigma_power)))*cos((c_2pi_number*(_x/a_lambda)) + a_psi);
      } while(++x,(ptr += pixel_step) < ptr_w_end);

      ptr += a_img->widthStep - image_ls;
   } while(++y,ptr < ptr_end);

   return true;
}/*}}}*/

bool compute_fft(IplImage *a_img,bool a_forward,unsigned a_w_power,unsigned a_h_power)
{/*{{{*/

   if (a_img == NULL || a_img->nChannels != 2 || a_img->depth != IPL_DEPTH_32F) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   if (width != (1 << a_w_power) || height != (1 << a_h_power)) {
      return false;
   }

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;

   // - 1D FFT by rows -
   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*a_img->widthStep + width*pixel_step;
   do {
      compute_1D_FFT(a_forward,a_w_power,ptr,pixel_step);
   } while((ptr += a_img->widthStep) < ptr_end);

   // - 1D FFT by columns -
   ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   ptr_end = ptr + width*pixel_step;
   do {
      compute_1D_FFT(a_forward,a_h_power,ptr,a_img->widthStep);
   } while((ptr += pixel_step) < ptr_end);

   return true;
}/*}}}*/

bool swap_quarters(IplImage *a_img)
{/*{{{*/
   if (a_img == NULL) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   if (width%2 != 0 || height%2 != 0) {
      return false;
   }

   unsigned width_d2 = width/2;
   unsigned height_d2 = height/2;

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;
   unsigned image_ls_d2 = width_d2*pixel_step;

   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + height_d2*a_img->widthStep;
   unsigned offset = height_d2*a_img->widthStep;

   unsigned char tmp_mem[image_ls_d2];

   do {
      memcpy(tmp_mem,ptr,image_ls_d2);
      memcpy(ptr,ptr + offset + image_ls_d2,image_ls_d2);
      memcpy(ptr + offset + image_ls_d2,tmp_mem,image_ls_d2);

      memcpy(tmp_mem,ptr + image_ls_d2,image_ls_d2);
      memcpy(ptr + image_ls_d2,ptr + offset,image_ls_d2);
      memcpy(ptr + offset,tmp_mem,image_ls_d2);
   } while((ptr += a_img->widthStep) < ptr_end);

   return true;
}/*}}}*/

bool deinterlace(IplImage *a_src,IplImage *a_trg)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != a_trg->nChannels || a_src->depth != a_trg->depth) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height || s_height%2 != 0) {
      return false;
   }

   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned s_image_ls = s_width*s_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + (t_height >> 1)*a_trg->widthStep;
   unsigned char *t_ptr_1 = (unsigned char *)a_trg->imageData + (t_y_offset + (t_height >> 1))*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;
   do {
      memcpy(t_ptr,s_ptr,s_image_ls);
      memcpy(t_ptr_1,s_ptr + a_src->widthStep,s_image_ls);
   } while(s_ptr += (a_src->widthStep << 1),t_ptr_1 += a_trg->widthStep,(t_ptr += a_trg->widthStep) < t_ptr_end);

   return true;
}/*}}}*/

bool seg_lbp_compute_offset_and_multiplier_array(IplImage *a_src,unsigned a_pattern_length,unsigned a_radius,vector<unsigned> &a_om_array)
{/*{{{*/
   int radius_m2 = (a_radius << 1) + 1;

   if (a_src == NULL || a_src->nChannels != 1 || a_src->depth != IPL_DEPTH_8U || a_pattern_length < 1 ||
      a_pattern_length > 32 || a_radius < 1) {
      return false;
   }

   int s_x_offset,s_y_offset,s_width,s_height;
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (radius_m2 > s_width || radius_m2 > s_height) {
      return false;
   }

   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;

   // - blank offset and multiplier array -
   a_om_array.clear();

   // - create offset and multipliers array -
   unsigned p_idx = 0;
   do {
      float px = (((float)a_radius)*cos(c_2pi_number*(((float)p_idx)/a_pattern_length))) + a_radius;
      float py = (-((float)a_radius)*sin(c_2pi_number*(((float)p_idx)/a_pattern_length))) + a_radius;

      unsigned cpx = ((int)px);
      unsigned cpy = ((int)py);

      float rpx = px - cpx;
      float rpy = py - cpy;

      float rev_rpx = 1.0f - rpx;
      float rev_rpy = 1.0f - rpy;

      // - pixel offset -
      a_om_array.push_back(cpy*a_src->widthStep + cpx*s_pixel_step);

      // - blend multipliers -
      a_om_array.push_back((unsigned)((rev_rpx*rev_rpy)*(UINT_MAX/UCHAR_MAX))); //0,0
      a_om_array.push_back((unsigned)((rpx*rev_rpy)*(UINT_MAX/UCHAR_MAX))); //1,0
      a_om_array.push_back((unsigned)((rev_rpx*rpy)*(UINT_MAX/UCHAR_MAX))); //0,1
      a_om_array.push_back((unsigned)((rpx*rpy)*(UINT_MAX/UCHAR_MAX))); //1,1

   } while(++p_idx < a_pattern_length);

   return true;
}/*}}}*/

bool seg_lbp_compute_LBP_by_offset_and_mutiplier_array(IplImage *a_src,IplImage *a_trg,unsigned a_pattern_length,unsigned a_radius,vector<unsigned> &a_om_array)
{/*{{{*/
   unsigned radius_m2 = (a_radius << 1) + 1;

   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_src->depth != IPL_DEPTH_8U || a_pattern_length < 1 || a_radius < 1 || a_trg->nChannels != 1 || 
      !((a_trg->depth == IPL_DEPTH_8U && a_pattern_length <= 8) || ((unsigned)a_trg->depth == IPL_DEPTH_32S && a_pattern_length <= 32))) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (t_width != s_width || t_height != s_height) {
      return false;
   }

   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned s_image_ls = s_width*s_pixel_step;
   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned t_image_ls = t_width*t_pixel_step;

   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + (t_y_offset + a_radius)*a_trg->widthStep + (t_x_offset + a_radius)*t_pixel_step;
   unsigned char *t_ptr_end = t_ptr + ((t_height - radius_m2) - 1)*a_trg->widthStep + (t_width - radius_m2)*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;

   // - offset to central pixel -
   unsigned s_center_offset = a_radius*a_src->widthStep + a_radius*s_pixel_step;

   // - first bit for pattern rotation -
   unsigned first_bit = 0x01 << (a_pattern_length - 1);

   switch (a_trg->depth) {
   case IPL_DEPTH_8U:
      {
         const unsigned char c_lbp_minimal_value_map[256] = // MAX - blank value
         {0,1,1,2,1,255,2,3,1,255,255,255,2,255,3,4,1,255,255,255,255,255,255,255,2,255,255,255,3,255,4,5,1,255,255,255,255,255,255,255,
         255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,3,255,255,255,4,255,5,6,1,255,255,255,255,255,255,255,255,255,255,
         255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,2,255,255,255,255,255,255,255,255,255,255,
         255,255,255,255,255,3,255,255,255,255,255,255,255,4,255,255,255,5,255,6,7,1,2,255,3,255,255,255,4,255,255,255,255,255,255,255,
         5,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,6,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
         255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,7,2,3,255,4,255,255,255,5,255,255,255,255,255,255,255,6,255,255,255,
         255,255,255,255,255,255,255,255,255,255,255,255,7,3,4,255,5,255,255,255,6,255,255,255,255,255,255,255,7,4,5,255,6,255,255,255,
         7,5,6,255,7,6,7,7,8};

         do {
            unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
            do {
               unsigned result = 0;
      
               // - compute binary pattern -
               vector<unsigned>::iterator poa_it = a_om_array.begin();

               do {
                  unsigned char value = (s_ptr[*poa_it]*poa_it[1] + s_ptr[*poa_it + s_pixel_step]*poa_it[2] + s_ptr[*poa_it + 
                     a_src->widthStep]*poa_it[3] + s_ptr[*poa_it + a_src->widthStep + s_pixel_step]*poa_it[4]) >> (UINT_BIT - CHAR_BIT);
      
                  result <<= 1;
      
                  if (value > s_ptr[s_center_offset]) {
                     result |= 0x01;
                  }
               } while((poa_it += 5) < a_om_array.end());
      
               *t_ptr = c_lbp_minimal_value_map[result];
      
            } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);
      
            t_ptr += a_trg->widthStep - t_image_ls;
            s_ptr += a_src->widthStep - s_image_ls;
         } while(t_ptr < t_ptr_end);
      }
      break;
   case IPL_DEPTH_32S:
      {
         do {
            unsigned char *t_ptr_w_end = t_ptr + t_image_ls;
            do {
               unsigned result = 0;
      
               // - compute binary pattern -
               vector<unsigned>::iterator poa_it = a_om_array.begin();
      
               do {
                  unsigned char value = (s_ptr[*poa_it]*poa_it[1] + s_ptr[*poa_it + s_pixel_step]*poa_it[2] + s_ptr[*poa_it + 
                     a_src->widthStep]*poa_it[3] + s_ptr[*poa_it + a_src->widthStep + s_pixel_step]*poa_it[4]) >> (UINT_BIT - CHAR_BIT);
      
                  result <<= 1;
      
                  if (value > s_ptr[s_center_offset]) {
                     result |= 0x01;
                  }
               } while((poa_it += 5) < a_om_array.end());
      
               // - rotate binary pattern to minimal value -
               unsigned min_result = result;
               unsigned idx = 0;
               do {
                  result = (result & 0x01)?((result >> 1) | first_bit):(result >> 1);
                  min_result = result < min_result?result:min_result;
               } while(++idx < a_pattern_length);
      
               *((unsigned *)t_ptr) = min_result;
      
            } while(s_ptr += s_pixel_step,(t_ptr += t_pixel_step) < t_ptr_w_end);
      
            t_ptr += a_trg->widthStep - t_image_ls;
            s_ptr += a_src->widthStep - s_image_ls;
         } while(t_ptr < t_ptr_end);
      }
      break;
   default:
      return false;
   }

   return true;
}/*}}}*/

bool seg_lbp_compute_LBP(IplImage *a_src,IplImage *a_trg,unsigned a_pattern_length,unsigned a_radius)
{/*{{{*/
   vector<unsigned> om_array;
   bool res;

   res = seg_lbp_compute_offset_and_multiplier_array(a_src,a_pattern_length,a_radius,om_array);
   if (!res) {
      return false;
   }

   res = seg_lbp_compute_LBP_by_offset_and_mutiplier_array(a_src,a_trg,a_pattern_length,a_radius,om_array);
   if (!res) {
      return false;
   }

   return true;
}/*}}}*/

bool seg_compute_cooccurence_matrix(IplImage *a_src,IplImage *a_trg,unsigned a_x_step,unsigned a_y_step)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_src->depth != IPL_DEPTH_8U ||
      a_trg->nChannels != 1 || (unsigned)a_trg->depth != IPL_DEPTH_32S) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (s_width <= (int)a_x_step || s_height <= (int)a_y_step || t_width != (1 << CHAR_BIT) || t_height != (1 << CHAR_BIT)) {
      return false;
   }

   // - clear cooccurence matrix -
   cvFillImage(a_trg,0.0);

   // - create cooccurence matrix -
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   unsigned s_image_ls = (s_width - a_x_step)*s_pixel_step;
   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;

   unsigned char *ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;
   unsigned char *s_ptr_end = s_ptr + ((s_height - a_y_step) - 1)*a_src->widthStep + (s_width - a_x_step)*s_pixel_step;
   unsigned offset = a_y_step*a_src->widthStep + a_x_step*s_pixel_step;

   do {
      unsigned char *s_ptr_w_end = s_ptr + s_image_ls;

      do {
         *((unsigned *)(ptr + *s_ptr*a_trg->widthStep + s_ptr[offset]*t_pixel_step)) += 1;
      } while((s_ptr += s_pixel_step) < s_ptr_w_end);

      s_ptr += a_src->widthStep - s_image_ls;
   } while(s_ptr < s_ptr_end);

   return true;
}/*}}}*/

unsigned char *ws_img_data;
unsigned ws_img_width;
unsigned ws_img_pixel_step;
unsigned ws_img_line_size;

class compare_pixels {
public:
   bool operator()(const unsigned &a_first,const unsigned &a_second) const
   {
      unsigned char *f_ptr = ws_img_data + (a_first/ws_img_width)*ws_img_line_size + (a_first%ws_img_width)*ws_img_pixel_step;
      unsigned char *s_ptr = ws_img_data + (a_second/ws_img_width)*ws_img_line_size + (a_second%ws_img_width)*ws_img_pixel_step;

      return *f_ptr < *s_ptr;
   }
};

bool seg_watershed(IplImage *a_src,IplImage *a_trg,unsigned a_seed_gap,unsigned *a_region_cnt)
{/*{{{*/
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_src->depth != IPL_DEPTH_8U ||
      a_trg->nChannels != 1 || (unsigned)a_trg->depth != IPL_DEPTH_32S || a_seed_gap < 2) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (s_width != t_width || s_height != t_height || a_seed_gap >= (unsigned)s_width || a_seed_gap >= (unsigned)s_height) {
      return false;
   }
   
   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;

   unsigned seed_gap_d2 = a_seed_gap >> 1;

   // - clear target image -
   cvFillImage(a_trg,0);

   // - create done image -
   IplImage *done_img = cvCreateImage(cvSize(s_width,s_height),IPL_DEPTH_8U,1);
   cvFillImage(done_img,0.0);

   // - set borders in done image "must not check borders in algorithm" -
   cvRectangle(done_img,cvPoint(0,0),cvPoint(s_width - 1,s_height - 1),cvScalar(255),1,8,0);

   // - compute g_img data begin and pixel_step -
   ws_img_data = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;
   ws_img_width = s_width;
   ws_img_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;
   ws_img_line_size = a_src->widthStep;

   // - create pixel multiset
   multiset<unsigned,compare_pixels> pp_mset;

   unsigned d_pixel_step = get_IplImage_pixel_size(done_img)*done_img->nChannels;

   // - insert initial seeds to pixel "queue" -
   {
      unsigned offset = seed_gap_d2*t_width + seed_gap_d2;
      unsigned offset_end = (t_height - 1)*t_width;
      unsigned char *d_ptr = (unsigned char *)done_img->imageData + seed_gap_d2*done_img->widthStep + seed_gap_d2*d_pixel_step;
      do {
         unsigned h_offset = offset;
         unsigned h_offset_end = h_offset + ((t_width - seed_gap_d2) - 1);
         unsigned char *hd_ptr = d_ptr;
         do {
            pp_mset.insert(h_offset);
            *hd_ptr = UCHAR_MAX;
         } while(hd_ptr += a_seed_gap*d_pixel_step,(h_offset += a_seed_gap) < h_offset_end);
      } while(d_ptr += a_seed_gap*done_img->widthStep,(offset += a_seed_gap*t_width) < offset_end);
   }

   unsigned init_label = 1;
   do {
      multiset<unsigned>::iterator min_it = pp_mset.begin();

      // - retrieve offset -
      unsigned offset = *min_it;
      unsigned y_offset = offset/t_width;
      unsigned x_offset = offset%t_width;

      unsigned char *t_ptr = (unsigned char *)a_trg->imageData + (t_y_offset + y_offset)*a_trg->widthStep + (t_x_offset + x_offset)*t_pixel_step;
      unsigned char *d_ptr = (unsigned char *)done_img->imageData + y_offset*done_img->widthStep + x_offset*d_pixel_step;

      unsigned color = 0;


#define WATERSHED_TEST_COLOR(OFFSET) \
{\
   unsigned *tmp_ptr = ((unsigned *)(t_ptr + (OFFSET)));\
\
   if (*tmp_ptr != color && *tmp_ptr != 0) {\
      if (color == 0) {\
         color = *tmp_ptr;\
      }\
      else {\
         color = 0xffffffff;\
      }\
   }\
}

      WATERSHED_TEST_COLOR(-t_pixel_step);
      WATERSHED_TEST_COLOR(t_pixel_step);
      WATERSHED_TEST_COLOR(-a_trg->widthStep);
      WATERSHED_TEST_COLOR(a_trg->widthStep);

      WATERSHED_TEST_COLOR(-a_trg->widthStep - t_pixel_step);
      WATERSHED_TEST_COLOR(-a_trg->widthStep + t_pixel_step);
      WATERSHED_TEST_COLOR(a_trg->widthStep - t_pixel_step);
      WATERSHED_TEST_COLOR(a_trg->widthStep + t_pixel_step);

      // - set pixel color -
      if (color != 0xffffffff) {
         if (color == 0) {
            *((unsigned *)t_ptr) = init_label++;
         }
         else {
            *((unsigned *)t_ptr) = color;
         }
      }

#define WATERSHED_INSERT_PIXEL(D_OFFSET,O_OFFSET) \
{\
   unsigned char *tmp_ptr = (d_ptr + (D_OFFSET));\
\
   if (*tmp_ptr == 0) {\
      pp_mset.insert(offset + (O_OFFSET));\
      *tmp_ptr = UCHAR_MAX;\
   }\
}

      // - insert new pixel pointers to set -
      WATERSHED_INSERT_PIXEL(-d_pixel_step,-1);
      WATERSHED_INSERT_PIXEL(d_pixel_step,1);
      WATERSHED_INSERT_PIXEL(-done_img->widthStep,-t_width);
      WATERSHED_INSERT_PIXEL(done_img->widthStep,t_width);

      WATERSHED_INSERT_PIXEL(-done_img->widthStep - d_pixel_step,-t_width - 1);
      WATERSHED_INSERT_PIXEL(-done_img->widthStep + d_pixel_step,-t_width + 1);
      WATERSHED_INSERT_PIXEL(done_img->widthStep - d_pixel_step,t_width - 1);
      WATERSHED_INSERT_PIXEL(done_img->widthStep + d_pixel_step,t_width + 1);

      // - remove pixel pointer from set -
      pp_mset.erase(min_it);

   } while(!pp_mset.empty());

   cvReleaseImage(&done_img);

   // - set region count value -
   *a_region_cnt = init_label;

   return true;
}/*}}}*/

bool compute_feature_vectors_from_lbp_image(IplImage *a_img,unsigned a_lbp_block_side,unsigned a_lbp_hist_size,float **a_res_data,unsigned &a_res_width,unsigned &a_res_height)
{/*{{{*/
   unsigned lbp_block_size_m1 = a_lbp_block_side - 1;

   if (a_img == NULL || *a_res_data != NULL || a_img->nChannels != 1 || a_img->depth != IPL_DEPTH_8U) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;

   a_res_width = a_img->width - lbp_block_size_m1;
   a_res_height = a_img->height - lbp_block_size_m1;

   // - create result data -
   unsigned res_data_b_size = a_res_width*a_res_height*a_lbp_hist_size*sizeof(float);
   *a_res_data = (float *)malloc(res_data_b_size);

   unsigned image_ls = (a_img->width - lbp_block_size_m1)*pixel_step;
   unsigned block_ls = a_lbp_block_side*pixel_step;

   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + ((height - lbp_block_size_m1) - 1)*a_img->widthStep + (width - lbp_block_size_m1)*pixel_step;
   float *lh_ptr = *a_res_data;

   // - window histogram -
   float w_hist[a_lbp_hist_size];

   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      
      // - clear window histogram -
      memset(w_hist,0,a_lbp_hist_size*sizeof(float));
      
      // - compute histogram of first pixel of line -
      {
         unsigned char *b_ptr = ptr;
         unsigned char *b_ptr_end = b_ptr + (a_lbp_block_side - 1)*a_img->widthStep + block_ls;
         do {
            unsigned char *b_ptr_w_end = b_ptr + block_ls;
            do {
               if (*b_ptr != c_lbp_blank_value) {
                  w_hist[*b_ptr]++;
               }
            } while((b_ptr += pixel_step) < b_ptr_w_end);

            b_ptr += a_img->widthStep - block_ls;
         } while(b_ptr < b_ptr_end);
      }

      // - set first pixel histogram -
      memcpy(lh_ptr,w_hist,a_lbp_hist_size*sizeof(float));

      // - move to second pixel of line -
      lh_ptr += a_lbp_hist_size;
      ptr += pixel_step;

      // - float window over line -
      do {
         unsigned char *b_ptr = ptr;
         unsigned char *b_ptr_end = ptr + a_lbp_block_side*a_img->widthStep;
         do {
            if (b_ptr[-1] != c_lbp_blank_value) {
               w_hist[b_ptr[-1]]--;
            }

            if (b_ptr[a_lbp_block_side - 1] != c_lbp_blank_value) {
               w_hist[b_ptr[a_lbp_block_side - 1]]++;
            }

            // - copy histogram -
            memcpy(lh_ptr,w_hist,a_lbp_hist_size*sizeof(float));

         } while((b_ptr += a_img->widthStep) < b_ptr_end);
      } while(lh_ptr += a_lbp_hist_size,(ptr += pixel_step) < ptr_w_end);

      ptr += a_img->widthStep - image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool compute_cooccurence_matrices(IplImage *a_img,unsigned a_x_step,unsigned a_y_step,unsigned a_cooc_block_side,unsigned a_cooc_bin_size,unsigned a_cooc_mat_side,unsigned **a_res_data,unsigned &a_res_width,unsigned &a_res_height)
{/*{{{*/
   unsigned cooc_block_side_m1 = a_cooc_block_side - 1;

   if (a_img == NULL || *a_res_data != NULL || a_img->nChannels != 1 || a_img->depth != IPL_DEPTH_8U) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   if ((unsigned)width < a_cooc_block_side || (unsigned)height < a_cooc_block_side || a_x_step >= a_cooc_block_side || a_y_step >= a_cooc_block_side) {
      return false;
   }

   a_res_width = width - cooc_block_side_m1;
   a_res_height = height - cooc_block_side_m1;

   // - create cooc result data -
   unsigned data_size = a_res_width*a_res_height*a_cooc_mat_side*a_cooc_mat_side;
   unsigned data_b_size = data_size*sizeof(unsigned);
   *a_res_data = (unsigned *)malloc(data_b_size);

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;
   unsigned image_wb_ls = (width - cooc_block_side_m1)*pixel_step;
   unsigned block_m_x_step_ls = (a_cooc_block_side - (1 + a_x_step))*pixel_step;

   // - compute image pointers -
   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + ((height - cooc_block_side_m1) - 1)*a_img->widthStep + (width - cooc_block_side_m1)*pixel_step;
   unsigned *c_ptr = *a_res_data;
   unsigned offset = a_y_step*a_img->widthStep + a_x_step*pixel_step;

   // - window cooccurence matrix -
   unsigned w_cooc[a_cooc_mat_side*a_cooc_mat_side];
   
   do {
      unsigned char *ptr_w_end = ptr + image_wb_ls;

      // - clear window matrix -
      memset(w_cooc,0,a_cooc_mat_side*a_cooc_mat_side*sizeof(unsigned));

      // - compute first pixel matrix -
      {
         unsigned char *b_ptr = ptr;
         unsigned char *b_ptr_end = b_ptr + (a_cooc_block_side - (1 + a_y_step))*a_img->widthStep + block_m_x_step_ls;
         do {
            unsigned char *b_ptr_w_end = b_ptr + block_m_x_step_ls;
            do {
               register unsigned char f_val = *b_ptr/a_cooc_bin_size;
               register unsigned char s_val = b_ptr[offset]/a_cooc_bin_size;
               w_cooc[f_val*a_cooc_mat_side + s_val]++;
            } while((b_ptr += pixel_step) < b_ptr_w_end);

            b_ptr += a_img->widthStep - block_m_x_step_ls;
         } while(b_ptr < b_ptr_end);
      }

      memcpy(c_ptr,w_cooc,a_cooc_mat_side*a_cooc_mat_side*sizeof(unsigned));
      ptr += pixel_step;
      c_ptr += a_cooc_mat_side*a_cooc_mat_side;

      do {
         unsigned char *b_ptr = ptr;
         unsigned char *b_ptr_end = b_ptr + (a_cooc_block_side - a_y_step)*a_img->widthStep;
         do {
            register unsigned char f_val = b_ptr[-1]/a_cooc_bin_size;
            register unsigned char s_val = b_ptr[-1 + offset]/a_cooc_bin_size;
            w_cooc[f_val*a_cooc_mat_side + s_val]--;

            f_val = b_ptr[a_cooc_block_side - (2 + a_x_step)]/a_cooc_bin_size;
            s_val = b_ptr[(a_cooc_block_side - (2 + a_x_step)) + offset]/a_cooc_bin_size;
            w_cooc[f_val*a_cooc_mat_side + s_val]++;
         } while((b_ptr += a_img->widthStep) < b_ptr_end);

         memcpy(c_ptr,w_cooc,a_cooc_mat_side*a_cooc_mat_side*sizeof(unsigned));
      } while(c_ptr += a_cooc_mat_side*a_cooc_mat_side,(ptr += pixel_step) < ptr_w_end);

      ptr += a_img->widthStep - image_wb_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool compute_haralick_features_from_cooccurence_matrices(float *a_cooc_mat_data,float **a_haralick_data,unsigned a_cooc_mat_cnt,unsigned a_cooc_mat_side)
{/*{{{*/
   if (a_cooc_mat_data == NULL || *a_haralick_data != NULL) {
      return false;
   }

   // - create haralick_data -
   unsigned haralick_data_b_size = a_cooc_mat_cnt*c_haralick_feature_size*sizeof(float);
   *a_haralick_data = (float *)malloc(haralick_data_b_size);

   float row_sums[a_cooc_mat_side];
   float col_sums[a_cooc_mat_side];

   float *c_ptr = a_cooc_mat_data;
   float *c_ptr_end = c_ptr + a_cooc_mat_cnt*a_cooc_mat_side*a_cooc_mat_side;
   float *h_ptr = *a_haralick_data;
   do {
      float h_energy = 0;
      float h_entropy = 0;
      float h_contrast = 0;
      float h_correlation = 0;
      float h_homogenity = 0;

      // - compute correlation parameters -
      float cor_mi_x = 0;
      float cor_mi_y = 0;
      float cor_sigma_x = 1.0f;
      float cor_sigma_y = 1.0f;

      // - compute cor_mi_x, and row_sums -
      {
         float *ptr = c_ptr;
         float *ptr_end = ptr + a_cooc_mat_side*a_cooc_mat_side;
         unsigned y = 0;
         do {
            row_sums[y] = 0;
            float *ptr_w_end = ptr + a_cooc_mat_side;
            do {
               row_sums[y] += *ptr;
            } while(++ptr < ptr_w_end);

            cor_mi_x += y*row_sums[y];
         } while(++y,ptr < ptr_end);
      }

      // - compute cor_mi_y, and col_sums -
      {
         float *ptr = c_ptr;
         float *ptr_end = ptr + a_cooc_mat_side;
         unsigned x = 0;
         do {
            col_sums[x] = 0;
            float *cl_ptr = ptr;
            float *cl_ptr_end = cl_ptr + a_cooc_mat_side*a_cooc_mat_side;
            do {
               col_sums[x] += *cl_ptr;
            } while((cl_ptr += a_cooc_mat_side) < cl_ptr_end);

            cor_mi_y += x*col_sums[x];
         } while(++x,++ptr < ptr_end);
      }

      // - compute cor_sigma_x -
      {
         unsigned y = 0;
         do {
            register float value = y - cor_mi_x;
            cor_sigma_x += value*value*row_sums[y];
         } while(++y < a_cooc_mat_side);
      }

      // - compute cor_sigma_y -
      {
         unsigned x = 0;
         do {
            register float value = x - cor_mi_y;
            cor_sigma_y += value*value*col_sums[x];
         } while(++x < a_cooc_mat_side);
      }

      // - compute haralick features -
      float *ptr = c_ptr;
      float *ptr_end = ptr + a_cooc_mat_side*a_cooc_mat_side;
      float y = 0;
      do {
         float *ptr_w_end = ptr + a_cooc_mat_side;
         float x = 0;
         do {
            h_energy += ptr[0]*ptr[0];

            register float value_p1 = ptr[0] + 1;
            h_entropy += value_p1*logf(value_p1);

            register float dd = x - y;
            h_contrast += dd*dd*ptr[0];

            h_correlation += x*y*ptr[0] - cor_mi_x*cor_mi_y;

            register float value = x - y;
            value = value < 0?-value:value;
            h_homogenity += ptr[0]/(1 + value);
         } while(++x,++ptr < ptr_w_end);
      } while(++y,ptr < ptr_end);

      h_contrast /= (float)a_cooc_mat_side*a_cooc_mat_side;
      h_correlation /= cor_sigma_x*cor_sigma_y;

      h_ptr[c_haralick_energy] = h_energy;
      h_ptr[c_haralick_entropy] = h_entropy;
      h_ptr[c_haralick_contrast] = h_contrast;
      h_ptr[c_haralick_correlation] = -h_correlation;
      h_ptr[c_haralick_homogenity] = h_homogenity;

   } while(h_ptr += c_haralick_feature_size,(c_ptr += a_cooc_mat_side*a_cooc_mat_side) < c_ptr_end);

   return true;
}/*}}}*/

void normalize_feature_vectors(float *a_data_ptr,unsigned a_vector_cnt,unsigned a_vector_size)
{/*{{{*/
   assert(a_data_ptr != NULL);

   float min_values[a_vector_size];
   float max_values[a_vector_size];

   // - initialize minimal and maximal values -
   {
      float *min_ptr = min_values;
      float *max_ptr = max_values;
      float *min_ptr_end = min_ptr + a_vector_size;
      do {
         *min_ptr = INFINITY;
         *max_ptr = -INFINITY;
      } while(++max_ptr,++min_ptr < min_ptr_end);
   }

   // - find minimal and maximal values -
   {
      unsigned f_idx = 0;
      do {
         float &min_value = min_values[f_idx];
         float &max_value = max_values[f_idx];

         // - cycle throught all hist data values -
         float *d_ptr = a_data_ptr + f_idx;
         float *d_ptr_end = d_ptr + a_vector_cnt*a_vector_size;
         do {
            *d_ptr < min_value?min_value = *d_ptr:0;
            *d_ptr > max_value?max_value = *d_ptr:0;
         } while((d_ptr += a_vector_size) < d_ptr_end);
      } while(++f_idx < a_vector_size);
   }

   // - normalize values -
   {
      unsigned f_idx = 0;
      do {
         float &min_value = min_values[f_idx];
         float &max_value = max_values[f_idx];

         if (max_value > min_value) {

            float diff_value = max_value - min_value;

            // - cycle throught all hist data values -
            float *d_ptr = a_data_ptr + f_idx;
            float *d_ptr_end = d_ptr + a_vector_cnt*a_vector_size;
            do {
               *d_ptr = (*d_ptr - min_value)/diff_value;
            } while((d_ptr += a_vector_size) < d_ptr_end);
         }
      } while(++f_idx < a_vector_size);
   }
}/*}}}*/

void normalize_feature_vectors_UINT_to_FLOAT(unsigned *a_data_ptr,float *a_trg_data_ptr,unsigned a_vector_cnt,unsigned a_vector_size)
{/*{{{*/
   assert(a_data_ptr != NULL && a_trg_data_ptr != NULL);

   unsigned min_values[a_vector_size];
   unsigned max_values[a_vector_size];

   // - initialize minimal and maximal values -
   {
      unsigned *min_ptr = min_values;
      unsigned *max_ptr = max_values;
      unsigned *min_ptr_end = min_ptr + a_vector_size;
      do {
         *min_ptr = UINT_MAX;
         *max_ptr = 0;
      } while(++max_ptr,++min_ptr < min_ptr_end);
   }

   // - find minimal and maximal values -
   {
      unsigned f_idx = 0;
      do {
         unsigned &min_value = min_values[f_idx];
         unsigned &max_value = max_values[f_idx];

         // - cycle throught all hist data values -
         unsigned *d_ptr = a_data_ptr + f_idx;
         unsigned *d_ptr_end = d_ptr + a_vector_cnt*a_vector_size;
         do {
            *d_ptr < min_value?min_value = *d_ptr:0;
            *d_ptr > max_value?max_value = *d_ptr:0;
         } while((d_ptr += a_vector_size) < d_ptr_end);
      } while(++f_idx < a_vector_size);
   }

   // - normalize values -
   {
      unsigned f_idx = 0;
      do {
         unsigned &min_value = min_values[f_idx];
         unsigned &max_value = max_values[f_idx];

         if (max_value > min_value) {

            float diff_value = (float)(max_value - min_value);

            // - cycle throught all hist data values -
            unsigned *d_ptr = a_data_ptr + f_idx;
            unsigned *d_ptr_end = d_ptr + a_vector_cnt*a_vector_size;
            float *td_ptr = a_trg_data_ptr + f_idx;
            do {
               *td_ptr = (*d_ptr - min_value)/diff_value;
            } while(td_ptr += a_vector_size,(d_ptr += a_vector_size) < d_ptr_end);
         }
      } while(++f_idx < a_vector_size);
   }
}/*}}}*/

bool image_from_vectors(float *a_data_ptr,unsigned a_vector_cnt,unsigned a_vector_size,unsigned a_bin_idx,IplImage *a_img)
{/*{{{*/
   if (a_img == NULL || a_img->nChannels != 1 || a_img->depth != IPL_DEPTH_32F) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   if ((unsigned)(width*height) != a_vector_cnt) {
      return false;
   }

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;
   unsigned image_ls = width*pixel_step;

   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;
   unsigned char *ptr_end = ptr + (height - 1)*a_img->widthStep + image_ls;
   float *v_ptr = a_data_ptr + a_bin_idx;
   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
         *((float *)ptr) = *v_ptr;
      } while(v_ptr += a_vector_size,(ptr += pixel_step) < ptr_w_end);

      ptr += a_img->widthStep - image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool image_from_vectors_op(float *a_data_ptr,unsigned a_vector_cnt,unsigned a_vector_size,unsigned a_operation,IplImage *a_img)
{/*{{{*/
   if (a_img == NULL || a_img->nChannels != 1 || a_img->depth != IPL_DEPTH_32F) {
      return false;
   }

   int x_offset,y_offset,width,height;
   get_IplImage_roi(a_img,x_offset,y_offset,width,height);

   if ((unsigned)(width*height) != a_vector_cnt) {
      return false;
   }

   unsigned pixel_step = get_IplImage_pixel_size(a_img)*a_img->nChannels;
   unsigned image_ls = width*pixel_step;

#define IMAGE_FROM_VECTORS_OP(INIT,OPERATION,SET) \
{\
   unsigned char *ptr = (unsigned char *)a_img->imageData + y_offset*a_img->widthStep + x_offset*pixel_step;\
   unsigned char *ptr_end = ptr + (height - 1)*a_img->widthStep + image_ls;\
   float *v_ptr = a_data_ptr;\
   do {\
      unsigned char *ptr_w_end = ptr + image_ls;\
      do {\
         INIT;\
\
         float *v_ptr_end = v_ptr + a_vector_size;\
         do {\
            OPERATION;\
         } while(++v_ptr < v_ptr_end);\
\
         SET;\
      } while((ptr += pixel_step) < ptr_w_end);\
\
      ptr += a_img->widthStep - image_ls;\
   } while(ptr < ptr_end);\
}

   switch (a_operation) {
   case ifv_SUM:
      IMAGE_FROM_VECTORS_OP(
      float sum = 0.0f,
      sum += *v_ptr,
      *((float *)ptr) = sum);
      break;
   case ifv_MAX:
      IMAGE_FROM_VECTORS_OP(
      float max = -INFINITY,
      max < *v_ptr?max = *v_ptr:0,
      *((float *)ptr) = max);
      break;
   case ifv_MIN:
      IMAGE_FROM_VECTORS_OP(
      float min = INFINITY,
      min > *v_ptr?min = *v_ptr:0,
      *((float *)ptr) = min);
      break;
   default:
      return false;
   }

   return true;
}/*}}}*/

bool lbp_seg_test(IplImage *a_src)
{/*{{{*/
   if (a_src == NULL) {
      return false;
   }

   unsigned lbp_hist_size = 9;

   IplImage *g_img = cvCreateImage(cvSize(a_src->width,a_src->height),IPL_DEPTH_8U,1);
   IplImage *lbp_img = cvCreateImage(cvSize(a_src->width,a_src->height),IPL_DEPTH_8U,1);

   cvCvtColor(a_src,g_img,CV_BGR2GRAY);
   assert(seg_lbp_compute_LBP(g_img,lbp_img,8,2));

   float *res_data = NULL;
   unsigned res_width,res_height;
   assert(compute_feature_vectors_from_lbp_image(lbp_img,16,lbp_hist_size,&res_data,res_width,res_height));

   normalize_feature_vectors(res_data,res_width*res_height,lbp_hist_size);

   // FIXME
   {
      IplImage *s_img = cvCreateImage(cvSize(res_width,res_height),IPL_DEPTH_32F,1);
      IplImage *t_img = cvCreateImage(cvSize(res_width,res_height),IPL_DEPTH_8U,1);

      //assert(image_from_vectors(res_data,res_width*res_height,lbp_hist_size,5,s_img));
      assert(image_from_vectors_op(res_data,res_width*res_height,lbp_hist_size,ifv_SUM,s_img));
      assert(normalize(0,s_img,s_img));
      cvConvertScale(s_img,t_img,255.0);
      
      cvSaveImage("result.bmp",t_img);

      cvReleaseImage(&t_img);
      cvReleaseImage(&s_img);
   }

   // FIXME
   /*{
      float *ptr = res_data;
      float *ptr_end = ptr + res_width*res_height*lbp_hist_size;
      do {
         float *ptr_w_end = ptr + lbp_hist_size;
         do {
            printf("%f\t",*ptr);
         } while(++ptr < ptr_w_end);
         putchar('\n');
      } while(ptr < ptr_end);
   }*/

   // FIXME
   //fprintf(stderr,"w: %u, h: %u\n",res_width,res_height);

   free(res_data);

   // FIXME
   //cvSaveImage("result.bmp",lbp_img);

   cvReleaseImage(&lbp_img);
   cvReleaseImage(&g_img);

   return true;
}/*}}}*/

bool cooc_seg_test(IplImage *a_src)
{/*{{{*/
   if (a_src == NULL) {
      return false;
   }

   unsigned cooc_block_side = 32;
   unsigned cooc_bin_size = 64;
   unsigned cooc_mat_side = 256/cooc_bin_size;

   IplImage *g_img = cvCreateImage(cvSize(a_src->width,a_src->height),IPL_DEPTH_8U,1);
   cvCvtColor(a_src,g_img,CV_BGR2GRAY);

   unsigned res_width,res_height;
   unsigned *res_data = NULL;

   compute_cooccurence_matrices(g_img,2,2,cooc_block_side,cooc_bin_size,cooc_mat_side,&res_data,res_width,res_height);

   float *norm_data = (float *)malloc(res_width*res_height*cooc_mat_side*cooc_mat_side*sizeof(float));
   normalize_feature_vectors_UINT_to_FLOAT(res_data,norm_data,res_width*res_height,cooc_mat_side*cooc_mat_side);

   float *haralick_features = NULL;
   assert(compute_haralick_features_from_cooccurence_matrices(norm_data,&haralick_features,res_width*res_height,cooc_mat_side));
   normalize_feature_vectors(haralick_features,res_width*res_height,c_haralick_feature_size);

   // FIXME
   {
      IplImage *s_img = cvCreateImage(cvSize(res_width,res_height),IPL_DEPTH_32F,1);
      IplImage *t_img = cvCreateImage(cvSize(res_width,res_height),IPL_DEPTH_8U,1);

      //assert(image_from_vectors(haralick_features,res_width*res_height,c_haralick_feature_size,4,s_img));
      assert(image_from_vectors_op(haralick_features,res_width*res_height,c_haralick_feature_size,ifv_MAX,s_img));
      assert(normalize(0,s_img,s_img));
      cvConvertScale(s_img,t_img,255.0);
      
      cvSaveImage("result.bmp",t_img);

      cvReleaseImage(&t_img);
      cvReleaseImage(&s_img);
   }

   // FIXME debug
   //{
   //   float *ptr = haralick_features;
   //   float *ptr_end = ptr + res_width*res_height*c_haralick_feature_size;
   //   do {
   //      float *ptr_w_end = ptr + c_haralick_feature_size;
   //      do {
   //         printf("%f\t",*ptr);
   //      } while(++ptr < ptr_w_end);
   //      putchar('\n');
   //   } while(ptr < ptr_end);
   //}

   free(haralick_features);
   free(norm_data);

   // FIXME debug
   //{
   //   unsigned *ptr = res_data;
   //   unsigned *ptr_end = ptr + res_width*res_height*cooc_mat_side*cooc_mat_side;
   //   do {
   //      unsigned *ptr_w_end = ptr + cooc_mat_side*cooc_mat_side;
   //      do {
   //         printf("%u\t",*ptr);
   //      } while(++ptr < ptr_w_end);
   //      putchar('\n');
   //   } while(ptr < ptr_end);
   //}

   // FIXME
   //fprintf(stderr,"w: %u, h: %u\n",res_width,res_height);

   free(res_data);

   cvReleaseImage(&g_img);

   return true;
}/*}}}*/

// experiments

/*bool compute_avg_image(IplImage *a_src,IplImage *a_trg)
{
   if (a_src == NULL || a_trg == NULL || a_src->nChannels != 1 || a_src->depth != IPL_DEPTH_8U ||
      a_trg->nChannels != 1 || a_trg->depth != IPL_DEPTH_32F) {
      return false;
   }

   int t_x_offset,t_y_offset,t_width,t_height;
   int s_x_offset,s_y_offset,s_width,s_height;

   get_IplImage_roi(a_trg,t_x_offset,t_y_offset,t_width,t_height);
   get_IplImage_roi(a_src,s_x_offset,s_y_offset,s_width,s_height);

   if (s_width != t_width || s_height != t_height) {
      return false;
   }
   
   unsigned t_pixel_step = get_IplImage_pixel_size(a_trg)*a_trg->nChannels;
   unsigned s_pixel_step = get_IplImage_pixel_size(a_src)*a_src->nChannels;

   unsigned char *s_ptr = (unsigned char *)a_src->imageData + s_y_offset*a_src->widthStep + s_x_offset*s_pixel_step;
   unsigned char *s_ptr_end = s_ptr + (s_height - 1)*a_src->widthStep + s_width*s_pixel_step;
   unsigned char *t_ptr = (unsigned char *)a_trg->imageData + t_y_offset*a_trg->widthStep + t_x_offset*t_pixel_step;
   do {
      unsigned char *s_ptr_w_end = s_ptr + s_width*s_pixel_step;
      do {
         *((float *)t_ptr) = 0.9f**((float *)t_ptr) + 0.1f*(*s_ptr/255.0f);
         // *((float *)t_ptr) = fabs(*((float *)t_ptr) - (*s_ptr/255.0f));
      } while(t_ptr += t_pixel_step,(s_ptr += s_pixel_step) < s_ptr_w_end);

      t_ptr += a_trg->widthStep - t_width*t_pixel_step;
      s_ptr += a_src->widthStep - s_width*s_pixel_step;
   } while(s_ptr < s_ptr_end);

   return true;
}*/

int main(int argc,char **argv)
{

      if (argc <= 1) {
         fprintf(stderr,"Expected at leas one argument\n");
      }
      else {
         cvNamedWindow("image_window",0);
         cvNamedWindow("svg_window",0);

         IplImage *image = NULL;
         IplImage *svg_image = NULL;

         char key;
         bool stop = false;
         int a_idx = 1;

         int event_name_size = 256;
         char event_name[event_name_size];
         event_name[0] = '\0';

         do {
            const char *c_str_result_svg_dir = "result_svgs/";
            const char *c_str_svg_jpg = ".svg.jpg";

            char *image_file = argv[a_idx];
            char *image_file_base = basename(image_file);

            int svg_image_file_size = strlen(c_str_result_svg_dir) + strlen(image_file_base) + strlen(c_str_svg_jpg) + 1;
            char svg_image_file[svg_image_file_size];

            strncpy(svg_image_file,c_str_result_svg_dir,svg_image_file_size);
            strncat(svg_image_file,image_file_base,svg_image_file_size);
            strncat(svg_image_file,c_str_svg_jpg,svg_image_file_size);

            // FIXME
            //fprintf(stderr,"svg_image_file: '%s'\n",svg_image_file);

            cvReleaseImage(&image);
            image = cvLoadImage(image_file);
            assert(image != NULL);

            cvReleaseImage(&svg_image);
            svg_image = cvLoadImage(svg_image_file);
            assert(svg_image != NULL);

            cvShowImage("image_window",image);
            cvShowImage("svg_window",svg_image);

            // TODO info o typu detekovaneho tagu
            {
               char *event_start = strstr(image_file,"/");
               assert(event_start != NULL);
               event_start += 1;

               char *event_end = strstr(event_start,"/");
               assert(event_end != NULL);

               *event_end = '\0';
               if (strcmp(event_start,event_name) != 0) {
                  strncpy(event_name,event_start,event_name_size);
                  printf("EVENT NAME: '%s'\n",event_name);

                  const char *mkdir_cmd_fmt = "mkdir 'selected/%s'";
                  int mkdir_cmd_size = strlen(mkdir_cmd_fmt) + strlen(event_name) + 1;
                  char mkdir_cmd[mkdir_cmd_size];
                  assert(snprintf(mkdir_cmd,mkdir_cmd_size,mkdir_cmd_fmt,event_name) < mkdir_cmd_size);
                  system(mkdir_cmd);
               }
               *event_end = '/';
            }

            do {
               key = cvWaitKey(0);
            } while(key != 'y' && key != 'n' && key != 'q');

            switch (key) {
            case 'y':
               {
                  const char *c_str_selected = "selected/";
                  const char *c_str_jpg = ".jpg";

                  int selected_file_size = strlen(c_str_selected) + strlen(event_name)  + strlen("/") + strlen(image_file_base) + strlen(c_str_jpg) + 1;
                  char selected_file[selected_file_size];

                  strncpy(selected_file,c_str_selected,selected_file_size);
                  strncat(selected_file,event_name,selected_file_size);
                  strncat(selected_file,"/",selected_file_size);
                  strncat(selected_file,image_file_base,selected_file_size);
                  strncat(selected_file,c_str_jpg,selected_file_size);

                  const char *copy_cmd_fmt = "cp '%s' '%s'";
                  int copy_cmd_size = strlen(copy_cmd_fmt) + strlen(image_file) + strlen(selected_file) + 1;
                  char copy_cmd[copy_cmd_size];
                  assert(snprintf(copy_cmd,copy_cmd_size,copy_cmd_fmt,image_file,selected_file) < copy_cmd_size);

                  fprintf(stderr,"%s",copy_cmd);
                  fputc('\n',stderr);

                  system(copy_cmd);
               }
               break;
            case 'q':
               stop = true;
               break;
            }

         } while(++a_idx < argc && !stop);

         cvReleaseImage(&svg_image);
         cvReleaseImage(&image);

         cvDestroyWindow("svg_window");
         cvDestroyWindow("image_window");
      }


   //lbp_seg_test(src);
   //cooc_seg_test(src);

   // canny test
   /*{
      CvCapture *capture = cvCaptureFromFile("../wki_videos/cctv-1.mpg");

      assert(cvGrabFrame(capture));
      IplImage *rgb_img = cvRetrieveFrame(capture);

      IplImage *gray_img = cvCreateImage(cvSize(rgb_img->width,rgb_img->height),IPL_DEPTH_8U,1);
      IplImage *edge_img = cvCreateImage(cvSize(rgb_img->width,rgb_img->height),IPL_DEPTH_8U,1);
      IplImage *conv_img = cvCreateImage(cvSize(rgb_img->width,rgb_img->height),IPL_DEPTH_8U,1);
      CvMat *kernel = cvCreateMat(3,3,CV_32FC1);

      // - set kernel -
      {
         float *ptr = kernel->data.fl;
         float *ptr_end = ptr + kernel->rows*kernel->cols;
         do {
            *ptr = 0.0f;
         } while(++ptr < ptr_end);

         //cvmSet(kernel,0,0,-1.0f);
         //cvmSet(kernel,0,1,-1.0f);
         //cvmSet(kernel,0,2,1.0f);
         //cvmSet(kernel,0,3,1.0f);
         //cvmSet(kernel,0,4,1.0f);
         //cvmSet(kernel,0,5,1.0f);
         //cvmSet(kernel,0,6,1.0f);
         //cvmSet(kernel,0,7,-1.0f);
         //cvmSet(kernel,0,8,-1.0f);

         //cvmSet(kernel,0,2,1.0f);
         //cvmSet(kernel,1,2,1.0f);
         //cvmSet(kernel,2,2,1.0f);
         //cvmSet(kernel,3,2,1.0f);
         //cvmSet(kernel,4,2,1.0f);
      }

      IplImage *avg_img = cvCreateImage(cvSize(rgb_img->width,rgb_img->height),IPL_DEPTH_32F,1);
      cvFillImage(avg_img,0.0);

      cvNamedWindow("window",0);

      char key;
      do {
         assert(cvGrabFrame(capture));
         assert((rgb_img = cvRetrieveFrame(capture)) != NULL);

         cvCvtColor(rgb_img,gray_img,CV_RGB2GRAY);
         //compute_avg_image(gray_img,avg_img);
         //cvConvertScale(avg_img,gray_img,255.0);

         //cvSmooth(gray_img,gray_img);
         cvCanny(gray_img,edge_img,0.5,0.5,5); 
         //compute_avg_image(edge_img,avg_img);
         //cvConvertScale(avg_img,gray_img,255.0);
         //cvFilter2D(edge_img,conv_img,kernel); 

         cvShowImage("window",edge_img);
         cvSaveImage("result.bmp",edge_img);

         key = cvWaitKey(2);
      } while(key != 'q');

      cvDestroyWindow("window");
      cvReleaseCapture(&capture);

      cvReleaseImage(&avg_img);
      cvReleaseMat(&kernel);
      cvReleaseImage(&conv_img);
      cvReleaseImage(&edge_img);
      cvReleaseImage(&gray_img);
   }*/

   // - gaabor function test -
   /*{
      IplImage *src = cvLoadImage("pokus.bmp");
      IplImage *gray = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_8U,1);
      IplImage *gray_32F_src = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_32F,1);
      IplImage *gray_32F_trg = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_32F,1);

      cvCvtColor(src,gray,CV_RGB2GRAY);
      cvConvertScale(gray,gray_32F_src,1.0/255.0);

      {
         IplImage *gaabor_img = cvCreateImage(cvSize(10,10),IPL_DEPTH_64F,1);
         IplImage *gaabor_32F_img = cvCreateImage(cvSize(gaabor_img->width,gaabor_img->height),IPL_DEPTH_32F,1);
         IplImage *result_img = cvCreateImage(cvSize(gaabor_img->width,gaabor_img->height),IPL_DEPTH_8U,1);

         //assert(generate_gaabor_function(4.0f,0.0f,0.5*c_pi_number,0.5,0.3,gaabor_img));
         assert(generate_gaabor_function(4.0f,0.0*c_pi_number,0.0*c_pi_number,2.0f,0.7f,gaabor_img));
         cvConvertScale(gaabor_img,gaabor_32F_img,1.0);

         // - convolution test -
         {
            IplImage *conv_img = cvCreateImage(cvSize(gray->width,gray->height),IPL_DEPTH_8U,1);
            CvMat *kernel = cvCreateMat(gaabor_32F_img->width,gaabor_32F_img->height,CV_32FC1);

            assert(gaabor_32F_img->width*sizeof(float) == (unsigned)gaabor_32F_img->widthStep);
            memcpy(kernel->data.fl,gaabor_32F_img->imageData,gaabor_32F_img->height*gaabor_32F_img->widthStep);

            cvFilter2D(gray_32F_src,gray_32F_trg,kernel);

            // FIXME
            cvConvertScale(gray_32F_trg,gray,255.0);
            cvSaveImage("target.bmp",gray);

            cvReleaseMat(&kernel);
            cvReleaseImage(&conv_img);
         }

         cvReleaseImage(&result_img);
         cvReleaseImage(&gaabor_32F_img);
         cvReleaseImage(&gaabor_img);
      }

      cvReleaseImage(&gray_32F_trg);
      cvReleaseImage(&gray_32F_src);
      cvReleaseImage(&gray);
      cvReleaseImage(&src);
   }*/

   // - create window -
   /*{
      //assert(compute_integral_image(grayscale,image_32S));
      //assert(normalize(0,image_32S,image_32S));
      //cvConvertScale(image_32S,image_32F,1.0/INT_MAX);

      //cvSetImageROI(image_src_32F,cvRect(100,100,300,300));
      //assert(normalize(0,image_src_32F,image_src_32F));
      //cvResetImageROI(image_src_32F);

      {
         IplImage *src = cvLoadImage("pokus.bmp");

         IplImage *grayscale = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_8U,1);
         IplImage *image_32S = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_32S,1);
         IplImage *image_32F = cvCreateImage(cvSize(src->width,src->height),IPL_DEPTH_32F,1);

         cvCvtColor(src,grayscale,CV_BGR2GRAY);
         IplImage *cooc_target = cvCreateImage(cvSize(grayscale->width,grayscale->height),IPL_DEPTH_32S,1);
         //cvFillImage(cooc_target,INT_MIN);

         //cvSetImageROI(grayscale,cvRect(100,100,300,300));

         //vector<unsigned> om_array;
         //assert(seg_lbp_compute_offset_and_multiplier_array(grayscale,8,2,om_array));
         //assert(seg_lbp_compute_LBP_by_offset_and_mutiplier_array(grayscale,cooc_target,8,2,om_array));

         //assert(seg_compute_cooccurence_matrix(grayscale,cooc_target,5,5));

         unsigned reg_cnt;
         assert(seg_watershed(grayscale,cooc_target,10,&reg_cnt));
         assert(normalize(0,cooc_target,cooc_target));

         //cvResetImageROI(grayscale);

         {
            IplImage *save_img = cvCreateImage(cvSize(cooc_target->width,cooc_target->height),IPL_DEPTH_8U,1);
            convert_32S_to_8U(cooc_target,save_img);
            //cvConvertScale(cooc_target,save_img,(float)UCHAR_MAX/(float)INT_MAX);

            // FIXME
            //cvFillImage(save_img,0.0);
            //cvRectangle(save_img,cvPoint(0,0),cvPoint(save_img->width - 1,save_img->height - 1),cvScalar(255),1,8,0);

            cvSaveImage("result.bmp",save_img);

            cvReleaseImage(&save_img);
         }

         cvReleaseImage(&cooc_target);

         cvReleaseImage(&image_32F);
         cvReleaseImage(&image_32S);
         cvReleaseImage(&grayscale);
      }

      //{
      //   //IplImage *k2f_img = cvCreateImage(cvSize(grayscale->width,grayscale->height),IPL_DEPTH_32F,2);
      //   //IplImage *f_img = cvCreateImage(cvSize(grayscale->width,grayscale->height),IPL_DEPTH_32F,1);

      //   //convert_8U_to_2x32F(grayscale,k2f_img);

      //   //compute_fft(k2f_img,true,8,8);
      //   //assert(swap_quarters(k2f_img));

      //   //assert(convert_2x32F_to_32F(k2f_img,f_img));
      //   //assert(normalize(0,f_img,f_img));

      //   //cvConvertScale(f_img,grayscale,255.0f);

      //   //cvSaveImage("result.bmp",grayscale);

      //   //cvReleaseImage(&f_img);
      //   //cvReleaseImage(&k2f_img);

      //   IplImage *dei_gs = cvCreateImage(cvSize(grayscale->width,grayscale->height),IPL_DEPTH_8U,1);

      //   assert(deinterlace(grayscale,dei_gs));
      //   cvSaveImage("result.bmp",dei_gs);

      //   cvReleaseImage(&dei_gs);
      //}

      //{
      //   unsigned sig_size_power = 9;
      //   unsigned sig_size = 1 << sig_size_power;
      //   float signal[2*sig_size];

      //   {
      //      float *ptr = signal;
      //      float *ptr_end = ptr + 2*sig_size;
      //      float x = 0.0f;
      //      do {
      //         ptr[0] = sinf(x);
      //         ptr[1] = 0.0f;
      //      } while(x += 0.1,(ptr += 2) < ptr_end);
      //   }

      //   compute_1D_FFT(true,sig_size_power,(unsigned char *)signal,2*sizeof(float));
   
      //   // - test print -
      //   {
      //      float *ptr = signal;
      //      float *ptr_end = ptr + 2*sig_size;
      //      do {
      //         printf("%f\t%f\n",ptr[0],ptr[1]);
      //      } while((ptr += 2) < ptr_end);
      //   }
      //}

      //cvNamedWindow("window",0);
      //cvShowImage("window",image_32F);

      //char key;
      //do {
      //   key = cvWaitKey(2);
      //} while(key != 'q');

      //cvDestroyWindow("window");
   }*/
   

   return 0;
}

