/* 
 * File:   water.cpp
 * Author: ipribyl
 * Based on paper by A. L. Rankin, L. H. Matthies and A. Huertas: "DAYTIME WATER
 * DETECTION BY FUSING MULTIPLE CUES FOR AUTONOMOUS OFF-ROAD NAVIGATION"
 * https://www-robotics.jpl.nasa.gov/publications/Arturo_Rankin/asc2004-jpl-water-detection-final.pdf
 * 
 * Created on 2. únor 2011, 15:27
 */

#include "water.h"

namespace areaClsf {

	/**
	 * Default constructor.
	 */
	Water::Water ()
	{
	}

	/**
	 * Copy constructor.
	 */
	Water::Water (const Water& orig)
	{
	}

	/**
	 * Destructor.
	 */
	Water::~Water ()
	{
	}

	/**
	 *
	 */
	TMapProbab Water::Find (IplImage *orig, IplImage *seg)
	{
		// convert image to grayscale
		IplImage *gray = cvCreateImage(cvGetSize(seg), IPL_DEPTH_8U, 1);
		cvCvtColor(orig, gray, CV_BGR2GRAY);
		// convert image to HSV color space
		IplImage *hsv = cvCreateImage(cvGetSize(seg), IPL_DEPTH_8U, 3);
		cvCvtColor(orig, hsv, CV_BGR2HSV);
		// split respective channels
		IplImage *hChnl = cvCreateImage(cvGetSize(seg), IPL_DEPTH_8U, 1);
		IplImage *sChnl = cvCreateImage(cvGetSize(seg), IPL_DEPTH_8U, 1);
		IplImage *vChnl = cvCreateImage(cvGetSize(seg), IPL_DEPTH_8U, 1);
		cvSplit(hsv, hChnl, sChnl, vChnl, NULL);
		cvReleaseImage(&hsv);

		// is sky in the image? (consider high brightness and low saturation in top 10 rows)
		double sSum = 0.0;
		double vSum = 0.0;
		int skyRows = 10;
		if (skyRows > orig->height) skyRows = orig->height;
		for (int y = 0; y < skyRows; y++) {
			for (int x = 0; x < orig->width; x++) {
				sSum += static_cast<uchar>((sChnl->imageData + sChnl->widthStep*y)[x]);
				vSum += static_cast<uchar>((vChnl->imageData + vChnl->widthStep*y)[x]);
			}
		}
		float sAvg = sSum / (skyRows * orig->width);
		float vAvg = vSum / (skyRows * orig->width);
		bool skyPresent = (sAvg < (0.3 * 256) && vAvg > (0.5 * 256));

		// water detection algorithm thresholds
		const uchar sT1 = 10;//should ~ 0
		const uchar sT2 = static_cast<uchar>(0.17 * 256);
		const uchar sT3 = static_cast<uchar>(0.1 * 256);
		const uchar sT4 = static_cast<uchar>(0.3 * 256);
		const uchar vT2 = static_cast<uchar>(0.83 * 256);
		const uchar vT34 = static_cast<uchar>(256 - sAvg);
		const uchar hTL = 120;//240° - OpenCV uses 0-179 for H channel
		const uchar hTH = 143;//285° - OpenCV uses 0-179 for H channel

		IplImage *tmpProb = static_cast<IplImage*>(cvClone(orig));//erase

		// maps for sum of grass occurence probability and count of pixels in each segment
		std::map<double, float> sum, count;
		// for each pixel do
		for (int y = 0; y < seg->height; y++) {
			for (int x = 0; x < seg->width; x++) {

				uchar gComponent = (orig->imageData  + orig->widthStep*y) [3*x+1];
				uchar hComponent = (hChnl->imageData + hChnl->widthStep*y)[x];
				uchar sComponent = (sChnl->imageData + sChnl->widthStep*y)[x];
				uchar vComponent = (vChnl->imageData + vChnl->widthStep*y)[x];

				// water color probability
				float pColor = 0.0;
				/* - thresholding replaced by probability modelling
				if (
					(sComponent <= sT1) ||
					(sComponent <= sT2 && vComponent >= vT2) ||
					(skyPresent && sComponent <= sT3 && vComponent > vT34) ||
					(skyPresent && sComponent <= sT4 && vComponent > vT34 && hTL < hComponent && hComponent < hTH)
				) {
					pColor = 1.0;
				}
				*/

				float pColor1 = exp(-pow(sComponent, 2)/400.0);
				float pColor2 = exp(-pow(sComponent, 2)/2000.0 - pow(255-vComponent, 2)/5000.0);
				float pColor3 = exp(-pow(sComponent, 2)/500.0 - pow(255-vComponent, 2)/(vT34*10.0));
				const uchar hOptimal = ((hTH - hTL) / 2) + hTL;
				float pColor4 = exp(-pow(sComponent, 2)/2500.0 - pow(255-vComponent, 2)/(vT34*10.0) - pow(hComponent - hOptimal, 2)/1000.0);

				if (skyPresent) {
					pColor = pColor1/4 + pColor2/4 + pColor3/4 + pColor4/4;
				} else {
					pColor = pColor1/2 + pColor2/2;
				}


				// water texture probability (low variance in saturation and green chanels)
				float pTexture = 0.0;
				// variance of current pixel (9x9 window)
				// http://en.wikipedia.org/wiki/Algorithms_for_calculating_variance#On-line_algorithm
				int filterSize = 9;// should be odd number
				int yStart = y - (filterSize/2); if (yStart < 0) yStart = 0;
				int yStop  = y + (filterSize/2); if (yStop >= gray->height) yStop = gray->height - 1;
				int xStart = x - (filterSize/2); if (xStart < 0) xStart = 0;
				int xStop  = x + (filterSize/2); if (xStop >= gray->width) xStop = gray->width - 1;
				int nPixels = 0;
				float meanS = 0.0;
				float M2S = 0.0;
				float meanG = 0.0;
				float M2G = 0.0;
				for (int yy = yStart; yy <= yStop; yy++) {
					for (int xx = xStart; xx <= xStop; xx++) {
						nPixels++;
						uchar pixelS = (sChnl->imageData  + sChnl->widthStep*yy) [xx];
						float deltaS = pixelS - meanS;
						meanS += deltaS / nPixels;
						M2S += deltaS * (pixelS - meanS);//This expression uses the new value of mean

						uchar pixelG = (orig->imageData  + orig->widthStep*yy) [3*xx+1];
						float deltaG = pixelG - meanG;
						meanG += deltaG / nPixels;
						M2G += deltaG * (pixelG - meanG);//This expression uses the new value of mean
					}
				}
				float varianceS = M2S / nPixels; // saturation variance
				float varianceG = M2G / nPixels; // green channel variance
				varianceS /= (128*128); // normalize
				varianceG /= (128*128); // normalize
				float variance = varianceS/2 + varianceG/2; //
				pTexture = exp(-pow(variance, 0.33)); // high water probability = low texture variance


				float pWater = pColor * pTexture;

				// save the probability
				double color = cvGet2D(seg, y, x).val[0];
				sum[color] += pWater;
				count[color] ++;
				(tmpProb->imageData + tmpProb->widthStep*y)[3*x] = (tmpProb->imageData + tmpProb->widthStep*y)[3*x+1] = (tmpProb->imageData + tmpProb->widthStep*y)[3*x+2] = (int)(255 * pWater);//erase
			}
		}

		cvReleaseImage(&vChnl);
		cvReleaseImage(&sChnl);
		cvReleaseImage(&hChnl);
		cvReleaseImage(&hsv);
		cvReleaseImage(&gray);

#if(DEBUG)
		//cvShowImage("resultWin", tmpProb); cvWaitKey(0);//erase
		cvSaveImage(this->frameName.c_str(), tmpProb);
#endif

		cvReleaseImage(&tmpProb);

		TMapProbab probabs;
		for (std::map<double, float>::iterator i = sum.begin(); i != sum.end(); ++i) {
			probabs[i->first] = sum[i->first] / count[i->first];
			//std::cout << int(i->first) << ": " << sum[i->first] << " - " << count[i->first] << std::endl;//erase
		}

		return probabs;
	}

}//namespace areaClsf
