
#ifndef __MEM_CHECK_H
#define __MEM_CHECK_H

@begin
   include "basic.h"
   include "mutex.h"
@end

// - defined by makefile -
//#define MEM_CHECK ENABLE

#if MEM_CHECK == ENABLE
/*
 * definition of generated structures
 */

// -- mc_block_s --
@begin
   struct
   <
   pointer:location
   unsigned:size
   >
   mc_block_s;
@end

// -- mc_block_set_s --
@begin
   set<mc_block_s>

   additions {
      
      /*!
       * \brief cmalloc for mc_block_set_s structure overriding global
       *
       */
      inline void *cmalloc(unsigned a_size);

      /*!
       * \brief cfree for mc_block_set_s structure overriding global
       */
      inline void cfree(void *a_location);
   }

   mc_block_set_s;
@end

// -- mc_struct_s --
@begin
   struct
   <
   mutex_s:mutex
   mc_block_set_s:mc_block_set
   unsigned:alloc_size
   unsigned:max_alloc_size
   unsigned:act_alloc_size
   >

   additions {
      
      /*!
       * \brief find index of memory block in mc_block_set by its location
       */
      unsigned get_idx_by_location(void *a_location);

      /*!
       * \brief return location of new allocated memory block of size a_size
       */
      inline void *get_block(unsigned a_size);

      /*!
       * \brief release block of memory by its location
       */
      inline void release_block(void *a_location);

      /*!
       * \brief check if all blocks of memory was released
       */
      void check();
   }

   mc_struct_s;
@end

/*
 * inline methods of generated structures
 */

// -- mc_block_s --
@begin
   inlines mc_block_s
@end

// -- mc_block_set_s --
@begin
   inlines mc_block_set_s
@end

inline void *mc_block_set_s::cmalloc(unsigned a_size)
{/*{{{*/
   return malloc(a_size);
}/*}}}*/

inline void mc_block_set_s::cfree(void *a_location)
{/*{{{*/
   free(a_location);
}/*}}}*/

// -- mc_struct_s --
@begin
   inlines mc_struct_s
@end

inline void *mc_struct_s::get_block(unsigned a_size)
{/*{{{*/
   mutex.lock();

   void *location = malloc(a_size);

   mc_block_s &mc_block = mc_block_set[mc_block_set.insert_blank()];
   mc_block.set(location,a_size);

   // - increase size of allocated memory -
   alloc_size += a_size;
   act_alloc_size += a_size;
   if (act_alloc_size > max_alloc_size) {
      max_alloc_size = act_alloc_size;
   }

   mutex.unlock();

   return location;
}/*}}}*/

inline void mc_struct_s::release_block(void *a_location)
{/*{{{*/
   mutex.lock();

   unsigned idx = get_idx_by_location(a_location);
   cassert(idx != c_idx_not_exist);

   // - decrease size of allocated memory -
   act_alloc_size -= mc_block_set[idx].size;

   mc_block_set.remove(idx);
   free(a_location);

   mutex.unlock();
}/*}}}*/

extern mc_struct_s mc_struct;

/*
 * mem_check global functions
 */
inline void mc_init()
{
   mc_struct.init();
   mc_struct.alloc_size = 0;
   mc_struct.max_alloc_size = 0;
   mc_struct.act_alloc_size = 0;
}

inline void mc_clear()
{
   mc_struct.check();
   mc_struct.clear();
}

inline void *cmalloc(unsigned a_size)
{/*{{{*/
   return mc_struct.get_block(a_size);
}/*}}}*/

inline void cfree(void *a_location)
{/*{{{*/
   mc_struct.release_block(a_location);
}/*}}}*/
#else
inline void mc_init() {}
inline void mc_clear() {}

inline void *cmalloc(unsigned a_size)
{/*{{{*/
   return malloc(a_size);
}/*}}}*/

inline void cfree(void *a_location)
{/*{{{*/
   free(a_location);
}/*}}}*/
#endif

#endif

