
#ifndef __CONDITION_H
#define __CONDITION_H

@begin
   include "basic.h"
   include "thread.h"
@end

/*
 * definition of structure conditoin_s
 */

@begin
   define condition_s dynamic
@end

struct condition_s
{
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   pthread_mutex_t mutex;
   pthread_cond_t cond;
#else
#endif

   inline unsigned init();
   inline unsigned clear();
   inline void flush_all() {}
   inline void swap(condition_s &a_src) {cassert(0);}
   inline condition_s &operator=(condition_s &a_src) {cassert(0); return *this;}
   inline bool operator==(condition_s &a_second) {cassert(0); return false;}

   /*!
    * \brief lock of condition mutex
    * \return error code
    */
   inline unsigned mutex_lock();

   /*!
    * \brief try to lock condition mutex
    * \return error code
    */
   inline unsigned mutex_try_lock();

   /*!
    * \brief unlock condition mutex
    * \return error code
    */
   inline unsigned mutex_unlock();

   /*!
    * \brief save lock of two condition mutexes
    * \return error code
    */
   inline unsigned mutex_safe_double_lock(condition_s &a_second);

   /*!
    * \brief save lock of three condition mutexes
    * \return error code
    */
   inline unsigned mutex_safe_triple_lock(condition_s &a_second,condition_s &a_third);

   /*!
    * \brief wait on condition signal
    * \return error code
    */
   inline unsigned wait();

   /*!
    * \brief broadcast signal to all waiting threads
    * \return error code
    */
   inline unsigned broadcast();
};

/*
 * inline methods of structure condition_s
 */

inline unsigned condition_s::init()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   pthread_mutexattr_t attr;
   pthread_mutexattr_init(&attr);
   pthread_mutexattr_settype(&attr,PTHREAD_MUTEX_RECURSIVE_NP);
   pthread_mutex_init(&mutex,&attr);
   pthread_mutexattr_destroy(&attr);

   pthread_cond_init(&cond,NULL);
   return c_error_OK;
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/


inline unsigned condition_s::clear()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   int ret = pthread_cond_destroy(&cond);
   switch (ret) {
   case 0:
      break;
   case EBUSY:
      return c_error_EBUSY;
   default:
      return c_error_UNKNOWN;
   }

   ret = pthread_mutex_destroy(&mutex);
   switch (ret) {
   case 0:
      return c_error_OK;
   case EBUSY:
      return c_error_EBUSY;
   default:
      return c_error_UNKNOWN;
   }
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

inline unsigned condition_s::mutex_lock()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   int ret = pthread_mutex_lock(&mutex);
   switch (ret) {
   case 0:
      return c_error_OK;
   case EINVAL:
      return c_error_EINVAL;
   case EDEADLK:
      return c_error_EDEADLK;
   default:
      return c_error_UNKNOWN;
   }
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

inline unsigned condition_s::mutex_try_lock()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   int ret = pthread_mutex_trylock(&mutex);
   switch (ret) {
   case 0:
      return c_error_OK;
   case EBUSY:
      return c_error_EBUSY;
   case EINVAL:
      return c_error_EINVAL;
   default:
      return c_error_UNKNOWN;
   }
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

inline unsigned condition_s::mutex_unlock()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   int ret = pthread_mutex_unlock(&mutex);
   switch (ret) {
   case 0:
      return c_error_OK;
   case EINVAL:
      return c_error_EINVAL;
   case EPERM:
      return c_error_EPERM;
   default:
      return c_error_UNKNOWN;
   }
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

inline unsigned condition_s::mutex_safe_double_lock(condition_s &a_second)
{/*{{{*/
   unsigned res;

   do {
      res = mutex_lock();
      if (res != c_error_OK) {
	 return res;
      }

      res = a_second.mutex_try_lock();
      switch (res) {
      case c_error_OK:
	 return c_error_OK;
      case c_error_EBUSY:
	 mutex_unlock();

	 thread_s::yield();
	 continue;
      default:
	 mutex_unlock();
	 return res;
      }
   } while(1);
}/*}}}*/

inline unsigned condition_s::mutex_safe_triple_lock(condition_s &a_second,condition_s &a_third)
{/*{{{*/
   unsigned res;

   do {
      res = mutex_lock();
      if (res != c_error_OK) {
	 return res;
      }

      res = a_second.mutex_try_lock();
      switch (res) {
      case c_error_OK:
	 break;
      case c_error_EBUSY:
	 mutex_unlock();

	 thread_s::yield();
	 continue;
      default:
	 mutex_unlock();
	 return res;
      }

      res = a_third.mutex_try_lock();
      switch (res) {
      case c_error_OK:
	 return c_error_OK;
      case c_error_EBUSY:
	 mutex_unlock();
	 a_second.mutex_unlock();

	 thread_s::yield();
	 continue;
      default:
	 mutex_unlock();
	 a_second.mutex_unlock();
	 return res;
      }
   } while(1);
}/*}}}*/

inline unsigned condition_s::wait()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   pthread_cond_wait(&cond,&mutex);
   return c_error_OK;
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

inline unsigned condition_s::broadcast()
{/*{{{*/
#if CONDITION_TYPE == CONDITION_TYPE_PTHREAD
   pthread_cond_broadcast(&cond);
   return c_error_OK;
#else
   cassert(0);
   return c_error_UNKNOWN;
#endif
}/*}}}*/

#endif

